package ar.com.sdd.agrofyapi.core;


import ar.com.sdd.agrofyapi.io.CommonResponse;
import ar.com.sdd.agrofyapi.io.PaymentIntentRequest;
import ar.com.sdd.agrofyapi.io.ErrorResponse;
import ar.com.sdd.agrofyapi.io.WebhookBindRequest;
import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSignSecurityManager;
import ar.com.sdd.commons.rest.model.TokenOAuth2;
import ar.com.sdd.commons.util.SimpleCache;
import ar.com.sdd.commons.util.SimpleCacheManager;
import org.apache.commons.lang3.NotImplementedException;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.ws.rs.client.Invocation;
import javax.ws.rs.core.Form;
import javax.ws.rs.core.MediaType;

@SuppressWarnings("unused")
public class AgrofyApiConnector implements RestSignSecurityManager {

    private final static Logger log = LogManager.getLogger(AgrofyApiConnector.class);

    private final RestConnector restConnector;
    private final String accessTokenBaseUrl;
    private final String applicationId;
    private final String deviceId;
    private final String userId;
    private final String userPassword;
    private final String accessModeType;

    public AgrofyApiConnector(AgrofyApiConnectorContext context) {
        this.accessTokenBaseUrl = context.getAccessTokenBaseUrl();
        this.applicationId = context.getApplicationId();
        this.deviceId = context.getDeviceId();
        this.userId = context.getUserId();
        this.userPassword = context.getUserPassword();
        this.accessModeType = context.getAccessModeType();

        log.trace("Creando AgrofyApiConnector para url [{}]", context.getAccessTokenBaseUrl());

        RestConnectorEnvironment environment = new RestConnectorEnvironment(context.getAccessTokenBaseUrl());
        restConnector = new RestConnector(environment, this);
    }

    public CommonResponse createPaymentIntent(PaymentIntentRequest request) throws RestConnectorException {
        final String path = "/payment/intent";
        log.debug("[createPaymentIntent] Request POST createPaymentIntent para el request [{}] path [{}]", request, path);

        return restConnector.genericPost(request, CommonResponse.class, ErrorResponse.class, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
    }

    public CommonResponse webhookBind(WebhookBindRequest request) throws RestConnectorException {
        final String path = "/bind-api/webhooks";
        log.debug("[webhookBind] Request POST webhookBind para el request [{}] path [{}]", request, path);

        return restConnector.genericPost(request, CommonResponse.class, ErrorResponse.class, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
    }


    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder, String requestBody) throws RestConnectorException {
        throw new NotImplementedException("user addHeader con dos parametros");
    }

    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        if (builder != null) {
            builder.header("applicationId", applicationId);
            builder.header("deviceId", deviceId);
            builder.header("Authorization", "Basic " + getAccessToken(false)); // Esto no es el accessToken, no se que es (EJ: Basic 29688ffc-662d-402d-9292-f508527098e5)
            builder.header("accessToken", getAccessToken(false));
        }
        return builder;
    }

    @Override
    public boolean retryOnUnauthorized() {
        try {
            getAccessToken(true);
            return true;
        } catch (RestConnectorException e) {
            log.error(e);
            return false;
        }
    }

    @Override
    public boolean getDisableHTTPSErrors() {
        return false;
    }

    public String getAccessToken(boolean force) throws RestConnectorException {
        SimpleCache cache = SimpleCacheManager.getInstance().getCache(AgrofyApiConnector.class.getName());
        String cacheKey = SimpleCacheManager.buildKey("accessToken");
        String accessToken = (String)cache.get(cacheKey);
        if (force || accessToken == null) {
            Form form = new Form();
            form.param("userId", userId);
            form.param("userPassword", userPassword);
            form.param("accessModeType", accessModeType);

            log.debug("[getAccessToken] Por recuperar token con userId [{}], userPassword [{}] y accessModeType [{}]", userId, userPassword, accessModeType);

            RestConnector connector = new RestConnector(new RestConnectorEnvironment(accessTokenBaseUrl), null);
            TokenOAuth2 tokenOAuth2Response = connector.genericPost(form, TokenOAuth2.class, "/login", MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);

            log.debug("[getAccessToken] Token recuperado [{}]. Expira en [{}]segs. Lo agrego al a cache", tokenOAuth2Response, tokenOAuth2Response.getExpiresIn());
            accessToken = tokenOAuth2Response.getAccessToken();
            cache.put(cacheKey, accessToken, tokenOAuth2Response.getExpiresIn()-10);
        }

        return accessToken;
    }
}
