package ar.com.sdd.agrotokenapi.core;

import ar.com.sdd.agrotokenapi.io.CreatePaymentRequest;
import ar.com.sdd.agrotokenapi.io.CreatePaymentResponse;
import ar.com.sdd.agrotokenapi.io.GetPaymentResponse;
import ar.com.sdd.agrotokenapi.model.Price;
import ar.com.sdd.agrotokenapi.util.SimpleCache;
import ar.com.sdd.agrotokenapi.util.SimpleCacheManager;
import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSecurityManager;
import ar.com.sdd.commons.rest.model.TokenOAuth2;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.ws.rs.client.Invocation;
import javax.ws.rs.core.Form;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;

public class AgrotokenApiConnector implements RestSecurityManager {

    private final static Logger log = LogManager.getLogger(AgrotokenApiConnector.class);

    private final RestConnector restConnector;
    private final String loginToken;
    private final String authBaseUrl;

    public AgrotokenApiConnector(AgrotokenApiConnectorContext context) {

        loginToken = Base64.getEncoder().encodeToString((context.getClientId() + ":" + context.getClientSecret()).getBytes(StandardCharsets.UTF_8));
        authBaseUrl = context.getAuthBaseUrl();

        log.debug("Creando AgrotokenConnector para url [" + context.getBaseUrl() + "] con token[" + context.getClientId() + ":" + context.getClientSecret() + "/" + loginToken + "]");
        RestConnectorEnvironment environment = new RestConnectorEnvironment(context.getBaseUrl());
        restConnector = new RestConnector(environment, this);
    }

    public CreatePaymentResponse createPaymentRequest(CreatePaymentRequest request) throws RestConnectorException {
        final String path = "/payment-requests/";
        log.debug("[createPaymentRequest] Request POST procesar crear pago, request [" + request + "] path [" + path + "]");
        return restConnector.genericPost(request, CreatePaymentResponse.class, path);
    }

    public GetPaymentResponse getPaymentRequest(String externalReference) throws RestConnectorException {
        final String path = "/payment-requests/";
        log.debug("[getPaymentRequest] Request con externalReference [" + externalReference + "]");
        GetPaymentResponse response = restConnector.genericGet(null, GetPaymentResponse.class, path + externalReference);
        log.debug("[getPaymentRequest] Response con externalReference [" + externalReference + "],  [" + response + "]");
        return response;
    }

    public Price getPrice(String symbol) throws RestConnectorException {
        final String path = "/prices/{symbol}";
        log.debug("[getPrice] Request con symbol [" + symbol + "]");
        Price response = restConnector.genericGet(null, Price.class, path, "symbol", symbol);
        log.debug("[getPrice] Response con symbol [" + symbol + "],  [" + response + "]");
        return response;
    }
    public List<Price> getPrices() throws RestConnectorException {
        final String path = "/prices";
        log.debug("[getPrice] Request");
        // Tengo SEGURO un lio de tipos aca: no se como tipar el resutlado a List<Price>
        List<Price> response = restConnector.genericGet(null, ArrayList.class, path);
        log.debug("[getPrice] Response [" + response + "]");
        return response;
    }

    public Response  getHealth() throws RestConnectorException {
        final String path = "/core/health/";
        log.debug("[getHealth] Request");
        Response response = restConnector.genericGet(null, Response.class, path);
        log.debug("[getHealth] Response  [" + response + "]");
        return response;
    }


    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        if (builder != null) {
            builder.header("Authorization", "Bearer " + getAccessToken());
        }
        return builder;
    }

    @Override
    public boolean getDisableHTTPSErrors() {
        return false;
    }

    public String getAccessToken() throws RestConnectorException {
        final String path = "/oauth2/token";

        SimpleCache tokenCache = SimpleCacheManager.getInstance().getCache(AgrotokenApiConnector.class.getName());
        String accessToken = (String)tokenCache.get("accessToken");
        if (accessToken==null) {

            log.debug("[getAccessToken] Por hacer login y obtener token con [" + loginToken + "]");

            RestConnectorEnvironment environment = new RestConnectorEnvironment(authBaseUrl);
            RestConnector tokenRestConnector = new RestConnector(environment, new RestSecurityManager() {
                @Override
                public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
                    builder.header("Authorization", "Basic " + loginToken);
                    return builder;
                }

                @Override
                public boolean getDisableHTTPSErrors() {
                    return false;
                }
            });

            //Inicialmente lo pido siempre
            Form form = new Form();
            form.param("scope", AgrotokenApiConnectorContext.defaultScope);
            form.param("grant_type", "client_credentials");

            TokenOAuth2 tokenOAuth2Response = tokenRestConnector.genericPost(form, TokenOAuth2.class, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
            log.debug("[getAccessToken] Token recuperado [" + tokenOAuth2Response + "]");

            accessToken = tokenOAuth2Response.getAccessToken();
            //Lo guardo en la cache para la proxima
            tokenCache.put("accessToken", accessToken, tokenOAuth2Response.getExpiresIn()-5); //Le saco 5 como margen
        }

        return accessToken;
    }
}