package ar.com.sdd.akanamonsanto.core;

import ar.com.sdd.akanamonsanto.io.TokenOAuth2Response;
import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSecurityManager;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;

import javax.ws.rs.client.Invocation;
import javax.ws.rs.core.Form;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.NewCookie;
import javax.ws.rs.core.Response;
import java.util.Map;

public class AkanaMonsantoSecurityManager implements RestSecurityManager {

    private static final Logger log = Logger.getLogger(AkanaMonsantoSecurityManager.class);

    /**
     * CLIENT_ID proporcionados por Monsanto
     */
    private String clientId;

    /**
     * CLIENT_SECRET proporcionados por Monsanto
     */
    private String clientSecret;

    /**
     * Usuario y password en Base64, en formato "usuario:passord". Para pasar a Base64, ver: https://www.base64encode.org/
     */
    private String userPass;

    /**
     * URL donde se recupera el Token de seguridad
     */
    private String tokenUrl;

    /**
     * Si este campo tiene un valor, para este request se va a recuperar del path puesto en este campo los headers para XCSRF
     */
    private String pathToRequestXCSRFTokenForThisPost;

    public AkanaMonsantoSecurityManager(String clientId, String clientSecret, String userPass, String tokenUrl) {
        this.clientId = clientId;
        this.clientSecret = clientSecret;
        this.userPass = userPass;
        this.tokenUrl = tokenUrl;
    }

    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {

        String token = recoverToken();
        log.trace("Token recuperado [" + token + "]");

        if (!StringUtils.isEmpty(pathToRequestXCSRFTokenForThisPost)) {

            log.trace("Recuperando XCSRF para el path [" + pathToRequestXCSRFTokenForThisPost + "]");

            RestConnectorEnvironment environment = new RestConnectorEnvironment();
            environment.baseUrl = pathToRequestXCSRFTokenForThisPost;

            //Lo borro para evitar futuros requests al mismo path (es solo para este post)
            pathToRequestXCSRFTokenForThisPost = null;

            //Armo un connector local para que pida el CSRF token
            RestConnector connector = new RestConnector(environment, null);

            Response postResponseHeaders = connector.buildRequest(environment.baseUrl, null)
                    .header("Authorization", "Bearer " + token + ", Basic " + userPass)
                    .header("X-CSRF-Token", "Fetch")
                    .get();

            String headerXCSRFToken = postResponseHeaders.getHeaderString("x-csrf-token");
            builder.header("X-CSRF-Token", headerXCSRFToken);
            log.trace("HeaderXCSRFToken [" + headerXCSRFToken + "]");

            Map<String, NewCookie> cookies = postResponseHeaders.getCookies();
            for (NewCookie cookie : cookies.values()) {
                log.trace("Cookie [" + cookie + "]");
                builder.cookie(cookie);
            }

            postResponseHeaders.close();
        }


        if (!StringUtils.isEmpty(token)) {
            builder.header("Authorization", "Bearer " + token + ", Basic " + userPass);
        }

        return builder;
    }

    /**
     * Setea que para este request, se pidan los headers X-CSRF
     */
    public void requestXCSRFTokenForThisPost(String path) {
        pathToRequestXCSRFTokenForThisPost = path;
    }

    /**
     * Recupera el token
     *
     * @return
     * @throws RestConnectorException
     */
    public String recoverToken() throws RestConnectorException {

        Form form = new Form();
        form.param("client_id", clientId);
        form.param("client_secret", clientSecret);
        form.param("grant_type", "client_credentials");

        log.trace("Por recuperar token con clientId [" + clientId + "], clientSecret [" + clientSecret + "]");

        RestConnectorEnvironment environment = new RestConnectorEnvironment();
        environment.baseUrl = tokenUrl;
        RestConnector connector = new RestConnector(environment, null);
        TokenOAuth2Response tokenOAuth2Response = connector.genericPost(form, TokenOAuth2Response.class, "", MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
        return tokenOAuth2Response.getAccessToken();
    }
}
