package ar.com.sdd.antlrparser;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.text.NumberFormat;
import java.text.ParseException;
import java.time.LocalDate;
import java.util.*;

public class FavoritaListenerImp extends FavoritaBaseListener{
    private ParserDocumentData dpgData;
    private ParserDocumentData docData;
    private Map<String, ParserDocumentData> payments;
    private Map<String, List<ParserDocumentData>> payable;
    private Map<String, List<ParserDocumentData>> deductions;
    private Map<String, List<ParserDocumentData>> retentions;
    private Map<String, List<ParserDocumentData>> instruments;


    @Override
    public void enterLotePago(FavoritaParser.LotePagoContext ctx) {
        dpgData = new ParserDocumentData();
        deductions = new HashMap<>();
        payable = new HashMap<>();
        retentions = new HashMap<>();
        instruments = new HashMap<>();
        payments = new HashMap<>();
    }


    @Override
    public void exitHeaderLinea8(FavoritaParser.HeaderLinea8Context ctx) {
        dpgData.setIssuerBank(ctx.getChild(1).getText());
    }


    @Override
    public void exitHeaderLinea9(FavoritaParser.HeaderLinea9Context ctx) {
        dpgData.setPayerDocRef(ctx.NUMBER().getText());
        dpgData.setLegalDocRef(ctx.NUMBER().getText());
    }

    @Override
    public void exitHeaderLinea10(FavoritaParser.HeaderLinea10Context ctx) {
        LocalDate creationDate = LocalDate.parse(ctx.DATE().toString());
        dpgData.setCreationDate(creationDate);
        String dueAmount = ctx.AMOUNT().getText();
        try {
            BigDecimal dueAmountFormatted = formatAmountToBigDecimal(dueAmount);
            dpgData.setDueAmount(dueAmountFormatted);
        } catch (ParseException e) {
            e.printStackTrace();
        }
    }


    @Override
    public void enterDetalleLine(FavoritaParser.DetalleLineContext ctx) {
        docData = new ParserDocumentData();
    }


    @Override
    public void exitDetalleLine(FavoritaParser.DetalleLineContext ctx) {

        try {

            docData.setLineNumber(ctx.NUMBER(0).toString());

            LocalDate localDate = LocalDate.parse(ctx.DATE().getText());
            docData.setDueDate(localDate);

            String legalDocRef = ctx.NUMBER(1).toString();
            docData.setLegalDocRef(legalDocRef);
            docData.setPayerDocRef(legalDocRef);



            docData.setTotalAmount(formatAmountToBigDecimal(ctx.AMOUNT(0).toString()));
            docData.setAmount1(formatAmountToBigDecimal(ctx.AMOUNT(1).toString()));
            docData.setAmount2(formatAmountToBigDecimal(ctx.AMOUNT(2).toString()));
            docData.setDueAmount(formatAmountToBigDecimal(ctx.AMOUNT(3).toString()));

            String docType = ctx.WORD(0).toString();

            docData.setDocType(docType);
            addDocumentToMap(dpgData.getPayerDocRef(), docData, payable);
        }catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public void exitFooterPago(FavoritaParser.FooterPagoContext ctx) {
        String totalAmount = ctx.AMOUNT().getText();
        try {
            BigDecimal amountFormatted = formatAmountToBigDecimal(totalAmount);
            dpgData.setTotalAmount(amountFormatted);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        System.out.println(ctx.AMOUNT().getText());
    }

    @Override
    public void exitLotePago(FavoritaParser.LotePagoContext ctx) {
        System.out.println("DPG= " + dpgData.toString());
    }

    private BigDecimal formatAmountToBigDecimal(String amount) throws ParseException{
        amount =  amount.replaceAll("[$)]", "").replaceAll("[(]", "-").trim();
        NumberFormat numberFormat = NumberFormat.getInstance( new Locale("en", "US"));
        Number numberFormatted = numberFormat.parse(amount);
        return new BigDecimal( numberFormatted.doubleValue()).setScale(2, BigDecimal.ROUND_HALF_UP);
    }

    private void addDocumentToMap(String key, ParserDocumentData document, Map<String, List<ParserDocumentData>> map) {
        List<ParserDocumentData> documentList =  map.computeIfPresent(key, (k, documentDataList) -> {
            documentDataList.add(document);
            return documentDataList;
        });

        if (documentList == null) {
            documentList = new ArrayList<>();
            map.put(key, documentList);
            documentList.add(document);
        }
    }


    public ParserDocumentData getDpgData() {
        return dpgData;
    }

    public void setDpgData(ParserDocumentData dpgData) {
        this.dpgData = dpgData;
    }

    public ParserDocumentData getDocData() {
        return docData;
    }

    public void setDocData(ParserDocumentData docData) {
        this.docData = docData;
    }

    public Map<String, ParserDocumentData> getPayments() {
        return payments;
    }

    public void setPayments(Map<String, ParserDocumentData> payments) {
        this.payments = payments;
    }

    public Map<String, List<ParserDocumentData>> getPayable() {
        return payable;
    }

    public void setPayable(Map<String, List<ParserDocumentData>> payable) {
        this.payable = payable;
    }

    public Map<String, List<ParserDocumentData>> getDeductions() {
        return deductions;
    }

    public void setDeductions(Map<String, List<ParserDocumentData>> deductions) {
        this.deductions = deductions;
    }

    public Map<String, List<ParserDocumentData>> getRetentions() {
        return retentions;
    }

    public void setRetentions(Map<String, List<ParserDocumentData>> retentions) {
        this.retentions = retentions;
    }

    public Map<String, List<ParserDocumentData>> getInstruments() {
        return instruments;
    }

    public void setInstruments(Map<String, List<ParserDocumentData>> instruments) {
        this.instruments = instruments;
    }
}
