package ar.com.sdd.bancoestadoapi.core;

import ar.com.sdd.bancoestadoapi.io.*;
import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSecurityManager;
import ar.com.sdd.commons.util.SimpleCache;
import ar.com.sdd.commons.util.SimpleCacheManager;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.ws.rs.client.Invocation;
import java.util.UUID;

public class BancoEstadoFrontendApiConnector implements RestSecurityManager {

    private final static Logger log = LogManager.getLogger(BancoEstadoFrontendApiConnector.class);

    private final SimpleCache cache = SimpleCacheManager.getInstance().getCache(getClass().getName());
    private final RestConnector connector;
    private final String rutEmpresa;
    private final String rutPersona;
    private String authToken;
    private Etapa etapa;

    private enum Etapa {
        PRELOGIN,
        LOGIN,
        OBTENER_CARTOLAS_HISTORICAS("BCHOCH"),
        OBTENER_DETALLE_CARTOLA("BCHODC");

        final String value;

        Etapa() {
            this.value = this.name();
        }

        Etapa(String value) {
            this.value = value;
        }
    }
    public BancoEstadoFrontendApiConnector(BancoEstadoFrontendApiConnectorContext context) {
        this.rutEmpresa = context.getRutEmpresa();
        this.rutPersona = context.getRutPersona();

        RestConnectorEnvironment environment = new RestConnectorEnvironment(context.getBaseUrl());
        connector = new RestConnector(environment, this);
    }

    public PreLoginResponse preLogin() throws RestConnectorException {
        final String path = "/home-nwe-publico-bff/prelogin";
        log.debug("[preLogin] Request GET por hacer pre login");
        etapa = Etapa.PRELOGIN;
        return connector.genericGet(null, PreLoginResponse.class, CommonError.class, path);
    }

    public LoginResponse login(LoginRequest loginRequest) throws RestConnectorException {
        final String path = "/home-nwe-publico-bff/login";
        log.debug("[login] Request POST por hacer login con {}", loginRequest);
        etapa = Etapa.LOGIN;
        return connector.genericPost(loginRequest, LoginResponse.class, CommonError.class, path);
    }

    public PostCartolasHistoricasResponse postCartolasHistoricas(String authToken, PostCartolasHistoricasRequest postCartolasHistoricasRequest) throws RestConnectorException {
        final String path = "/cct-cartolas-historica-pj-bff/historica/cartolas";
        log.debug("[getCartolasHistoricas] Request POST por obtener las cartolas historias con {}", postCartolasHistoricasRequest);
        etapa = Etapa.OBTENER_CARTOLAS_HISTORICAS;
        this.authToken = authToken;
        return connector.genericPost(postCartolasHistoricasRequest, PostCartolasHistoricasResponse.class, CommonError.class, path);
    }

    public PostDetalleCartolaResponse postDetalleCartola(String authToken, PostDetalleCartolaRequest postDetalleCartolaRequest) throws RestConnectorException {
        final String path = "/cct-cartolas-historica-pj-bff/detalle-cartola";
        log.debug("[postDetalleCartola] Request POST por obtener el detalle de la cartolas con {}", postDetalleCartolaRequest);
        etapa = Etapa.OBTENER_DETALLE_CARTOLA;
        this.authToken = authToken;
        return connector.genericPost(postDetalleCartolaRequest, PostDetalleCartolaResponse.class, CommonError.class, path);
    }

    private String getSessionCode() {
        final String cacheKey = SimpleCacheManager.buildKey("sessionCode");
        String sessionCode = (String) cache.get(cacheKey);
        if (sessionCode == null) {
            sessionCode = UUID.randomUUID().toString();
            cache.put(cacheKey, sessionCode, SimpleCache.TEN_MINUTES);
        }
        return sessionCode;
    }

    private String getXTrackId() {
        final String cacheKey = SimpleCacheManager.buildKey("xTrackId");
        String xTrackId = (String) cache.get(cacheKey);
        if (xTrackId == null) {
            xTrackId = UUID.randomUUID().toString();
            cache.put(cacheKey, xTrackId, SimpleCache.TEN_MINUTES);
        }
        return xTrackId;
    }


    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        if (builder != null) {
            builder.header("canal", "NWE");
            builder.header("codigosesion", getSessionCode());
            builder.header("dvempresa", "6");
            builder.header("dvpersona", "5");
            builder.header("rutempresa", rutEmpresa);
            builder.header("rutpersona", rutPersona);
            builder.header("xtrackid", getXTrackId());
            // Para simular que soy un usuario desde un navegador web
            builder.header("User-Agent", "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36");
            //builder.header("xtrackid", getUserAgent());
            builder.header("etapa", etapa.value);
            if (etapa.equals(Etapa.PRELOGIN) || etapa.equals(Etapa.LOGIN)) {
                builder.header("funcionalidad", "login");
                builder.header("topico", "WEBOMT");
            } else if (etapa.equals(Etapa.OBTENER_CARTOLAS_HISTORICAS) || etapa.equals(Etapa.OBTENER_DETALLE_CARTOLA)) {
                builder.header("funcionalidad", "SVXCCT005");
                builder.header("authorization", authToken);
            }
        }
        return builder;
    }

    @Override
    public boolean retryOnUnauthorized() {
        return false;
    }

    @Override
    public boolean getDisableHTTPSErrors() {
        return false;
    }
}
