package ar.com.sdd.bindapi.core;

import ar.com.sdd.bindapi.io.*;
import ar.com.sdd.bindapi.model.Account;
import ar.com.sdd.bindapi.model.Transaction;
import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSecurityManager;
import ar.com.sdd.commons.util.DateUtil;
import ar.com.sdd.commons.util.SimpleCache;
import ar.com.sdd.commons.util.SimpleCacheManager;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.ws.rs.client.Invocation;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

@SuppressWarnings("unused")
public class BindApiConnector implements RestSecurityManager {

    private static final Logger log = LogManager.getLogger(BindApiConnector.class);

    private final RestConnector connector;
    private final String username;
    private final String password;
    private boolean doingLogin = false;

    private final Map<String, Object> headers = new HashMap<>();

    public BindApiConnector(BindApiConnectorContext context) {
        this.username = context.getUsername();
        this.password = context.getPassword();

        log.trace("Creando BindApiConnector con {}", context);
        RestConnectorEnvironment environment = new RestConnectorEnvironment(context.getBaseUrl(), context.getKeyStorePath(), context.getKeyStorePassword(), null, null);
        connector = new RestConnector(environment, this);
    }

    // ----------------------------------------- CUENTAS ----------------------------------------- //

    public Account[] getAccounts(long bankId, String viewId) throws RestConnectorException {
        final String path = "/banks/" + bankId + "/accounts/" + viewId;
        log.debug("[getAccounts] Listando cuentas con path [{}]", path);
        return connector.genericGet(null, Account[].class, CommonError.class, path);
    }

    public GetConsultaCuentaPorAliasCbuCvuResponse getAccountAlias(String alias) throws RestConnectorException {
        final String path = "/accounts/alias/" + alias;
        log.debug("[getAccountAlias] Consultando cuenta por alias con path [{}]", path);
        return connector.genericGet(null, GetConsultaCuentaPorAliasCbuCvuResponse.class, CommonError.class, path);
    }

    public GetConsultaCuentaPorAliasCbuCvuResponse getAccountCbuCvu(String cbu) throws RestConnectorException {
        final String path = "/accounts/cbu/" + cbu;
        log.debug("[getAccountCbuCvu] Consultando cuenta por CBU/CVU con path [{}]", path);
        return connector.genericGet(null, GetConsultaCuentaPorAliasCbuCvuResponse.class, CommonError.class, path);
    }

    // ----------------------------------------- DEBIN ----------------------------------------- //

    public PostCrearPedidoDebinResponse postCrearPedidoDebinResponse(long bankId, String accountId, String viewId, PostCrearPedidoDebinRequest postCrearPedidoDebinRequest) throws RestConnectorException {
        final String path = "/banks/" + bankId + "/accounts/" + accountId +"/" + viewId + "/transaction-request-types/DEBIN/transaction-requests";
        log.debug("[postCrearPedidoDebinResponse] Creando pedido Debin con path [{}] y [{}]", path, postCrearPedidoDebinRequest);
        return connector.genericPost(postCrearPedidoDebinRequest, PostCrearPedidoDebinResponse.class, CommonError.class, path);
    }

    public GetConsultaDatosVendedorResponse getConsultaDatosVendedorResponse(long bankId, String accountId, String viewId) throws RestConnectorException {
        final String path = "/banks/" + bankId + "/accounts/" + accountId +"/" + viewId + "/transaction-request-types/DEBIN/info";
        log.debug("[getConsultaDatosVendedorResponse] Consultando datos de vendedor Debin con path [{}]", path);
        return connector.genericGet(null, GetConsultaDatosVendedorResponse.class, CommonError.class, path);
    }

    public Transaction[] getObtenerDebinesParaCobrar(long bankId, String accountId, String viewId) throws RestConnectorException {
        final String path = "/banks/" + bankId + "/accounts/" + accountId +"/" + viewId + "/transaction-request-types/DEBIN";
        log.debug("[getObtenerDebinesParaCobrar] Obteniendo Debines para cobrar con path [{}]", path);
        return connector.genericGet(null, Transaction[].class, CommonError.class, path);
    }

    public Transaction getObtenerDebin(long bankId, String accountId, String viewId, String transactioId) throws RestConnectorException {
        final String path = "/banks/" + bankId + "/accounts/" + accountId +"/" + viewId + "/transaction-request-types/DEBIN/" + transactioId;
        log.debug("[getObtenerDebin] Obteniendo Debin con path [{}]", path);
        return connector.genericGet(null, Transaction.class, CommonError.class, path);
    }

    public DeleteEliminarPedidoDebinResponse deleteEliminarPedidoDebinResponse(long bankId, String accountId, String viewId, String transactioId) throws RestConnectorException {
        final String path = "/banks/" + bankId + "/accounts/" + accountId +"/" + viewId + "/transaction-request-types/DEBIN/" + transactioId;
        log.debug("[deleteEliminarPedidoDebinResponse] Eliminando pedido Debin con path [{}]", path);
        return connector.genericDelete(null, DeleteEliminarPedidoDebinResponse.class, CommonError.class, path);
    }

    public PutAltaBajaCuentaVendedorResponse putAltaBajaCuentaVendedorResponse(long bankId, String accountId, String viewId, PutAltaBajaCuentaVendedorRequest putAltaBajaCuentaVendedorRequest) throws RestConnectorException {
        final String path = "/banks/" + bankId + "/accounts/" + accountId +"/" + viewId + "/transaction-request-types/DEBIN/";
        log.debug("[putAltaBajaCuentaVendedorResponse] Dando alta/baja cuenta vendedor Debin con path [{}] y [{}]", path, putAltaBajaCuentaVendedorRequest);
        return connector.genericPut(putAltaBajaCuentaVendedorRequest, PutAltaBajaCuentaVendedorResponse.class, CommonError.class, path);
    }

    public Transaction postABMPedidoDeRecurrenciaDeDebinResponse(long bankId, String viewId, PostABMPedidoDeRecurrenciaDeDebinRequest postABMPedidoDeRecurrenciaDeDebinRequest) throws RestConnectorException {
        final String path = "/banks/" + bankId + "/" + viewId + "/transaction-request-types/DEBIN-SUBSCRIPTION/transaction-requests";
        log.debug("[postABMPedidoDeRecurrenciaDeDebinResponse] Creando pedido de recurrencia Debin con path [{}] y [{}]", path, postABMPedidoDeRecurrenciaDeDebinRequest);
        return connector.genericPost(postABMPedidoDeRecurrenciaDeDebinRequest, Transaction.class, CommonError.class, path);
    }

    // ----------------------------------------- BILLETERA --------------------------------------- //

    public PostAltaCVUClienteResponse postAltaCVUClienteResponse(long bankId, String accountId, String viewId, PostAltaCVUClienteRequest postAltaCVUClienteRequest) throws RestConnectorException {
        final String path = "/banks/" + bankId + "/accounts/" + accountId +"/" + viewId + "/wallet/cvu";
        log.debug("[postAltaCVUClienteResponse] Creando CVU cliente con path [{}] y [{}]", path, postAltaCVUClienteRequest);
        return connector.genericPost(postAltaCVUClienteRequest, PostAltaCVUClienteResponse.class, CommonError.class, path);
    }

    public PostAsignarModificarAliasCVUResponse postAsignarModificarAliasCVUResponse(long bankId, String accountId, String viewId, PostAsignarModificarAliasCVURequest postAsignarModificarAliasCVURequest) throws RestConnectorException {
        final String path = "/banks/" + bankId + "/accounts/" + accountId +"/" + viewId + "/wallet/alias";
        log.debug("[postAsignarModificarAliasCVUResponse] Asignando/modificando alias CVU con path [{}] y [{}]", path, postAsignarModificarAliasCVURequest);
        return connector.genericPost(postAsignarModificarAliasCVURequest, PostAsignarModificarAliasCVUResponse.class, CommonError.class, path);
    }

    // Este no es necesario segun Alfred porque solo recibimos, no enviamos (caso PAE)
    public Transaction postRealizarTransferenciaDesdeCVU(long bankId, String accountId, String viewId, PostRealizarTransferenciaDesdeCVURequest postRealizarTransferenciaDesdeCVURequest) throws RestConnectorException {
        final String path = "/banks/" + bankId + "/accounts/" + accountId +"/" + viewId + "/transaction-request-types/TRANSFER-CVU/transaction-requests";
        log.debug("[postRealizarTransferenciaDesdeCVUResponse] Realizando transferencia desde CVU con path [{}] y [{}]", path, postRealizarTransferenciaDesdeCVURequest);
        return connector.genericPost(postRealizarTransferenciaDesdeCVURequest, Transaction.class, CommonError.class, path);
    }

    public Transaction[] getObtenerListadoDeTransferencias(long bankId, String accountId, String viewId, Date filterFrom, Date filterTo, String filterOrigin, String filterStatus, int offset, int limit) throws RestConnectorException {
        final String path = "/banks/" + bankId + "/accounts/" + accountId +"/" + viewId + "/transaction-request-types/TRANSFER-CVU";
        log.debug("[getObtenerListadoDeTransferencias] Obteniendo listado de transferencias con path [{}]", path);

        if (filterFrom != null) headers.put("obp_from_date", DateUtil.formatJsonDate(filterFrom));
        if (filterTo != null) headers.put("obp_to_date", DateUtil.formatJsonDate(filterTo));
        if (filterOrigin != null) headers.put("obp_origin", filterOrigin);
        if (filterStatus != null) headers.put("obp_status", filterStatus);
        headers.put("obp_offset", offset);
        headers.put("obp_limit", limit);

        Transaction[] transactions = connector.genericGet(null, Transaction[].class, CommonError.class, path);
        // Limpio los headers para el proximo request
        headers.clear();
        return transactions;
    }

    public Transaction getObtenerTransferencia(long bankId, String accountId, String viewId, String transactionId) throws RestConnectorException {
        final String path = "/banks/" + bankId + "/accounts/" + accountId +"/" + viewId + "/transaction-request-types/TRANSFER-CVU/" + transactionId;
        log.debug("[getObtenerTransferencialong] Obteniendo transferencia con path [{}]", path);
        return connector.genericGet(null, Transaction.class, CommonError.class, path);
    }

    public DeleteBajaCVUResponse deleteBajaCVUResponse(long bankId, String accountId, String viewId, String cvu, String cuit) throws RestConnectorException {
        final String path = "/banks/" + bankId + "/accounts/" + accountId +"/" + viewId + "/wallet/cvu/" + cvu + "/" + cuit;
        log.debug("[DeleteBajaCVU] Dando de baja CVU con path [{}]", path);
        return connector.genericDelete(null, DeleteBajaCVUResponse.class, CommonError.class, path);
    }

    public PutModificarCVUClienteResponse putModificarCVUCliente(long bankId, String accountId, String viewId, String cvu, PutModificarCVUClienteRequest putModificarCVUClienteRequest) throws RestConnectorException {
        final String path = "/banks/" + bankId + "/accounts/" + accountId +"/" + viewId + "/wallet/cvu/" + cvu;
        log.debug("[putModificarCVUCliente] Modificando CVU cliente con path [{}] y [{}]", path, putModificarCVUClienteRequest);
        return connector.genericPut(putModificarCVUClienteRequest, PutModificarCVUClienteResponse.class, CommonError.class, path);
    }

    // ----------------------------------------- CHEQUES --------------------------------------- //

    public PostEmitirChequeResponse postEmitirChequeResponse(long bankId, String accountId, String viewId, PostEmitirChequeRequest postEmitirChequeRequest) throws RestConnectorException {
        final String path = "/banks/" + bankId + "/accounts/" + accountId +"/" + viewId + "/transaction-request-types/CHECK";
        log.debug("[postEmitirCheque] Emitiendo cheque con path [{}] y [{}]", path, postEmitirChequeRequest);
        return connector.genericPost(postEmitirChequeRequest, PostEmitirChequeResponse.class, CommonError.class, path);
    }

    public PutGestionarChequeResponse putGestionarChequeResponse(long bankId, String accountId, String viewId, PutGestionarChequeRequest putGestionarChequeRequest) throws RestConnectorException {
        final String path = "/banks/" + bankId + "/accounts/" + accountId +"/" + viewId + "/transaction-request-types/CHECK";
        log.debug("[putGestionarChequeResponse] Gestionando cheque con path [{}] y [{}]", path, putGestionarChequeRequest);
        return connector.genericPut(putGestionarChequeRequest, PutGestionarChequeResponse.class, CommonError.class, path);
    }

    public GetInformaNovedadesDeEcheqs getInformaNovedadesDeEcheqsResponse(long bankId, String viewId) throws RestConnectorException {
        final String path = "/banks/" + bankId + "/echeq/" + viewId + "/notification";
        log.debug("[getInformaNovedadesDeEcheqsResponse] Obteniendo novedades de Echeqs con path [{}]", path);
        return connector.genericGet(null, GetInformaNovedadesDeEcheqs.class, CommonError.class, path);
    }

    public Transaction[] getObtenerCheques(long bankId, String accountId, String viewId) throws RestConnectorException {
        final String path = "/banks/" + bankId + "/accounts/" + accountId +"/" + viewId + "/transaction-request-types/CHECK";
        log.debug("[getObtenerCheques] Obteniendo cheques con path [{}]", path);
        return connector.genericGet(null, Transaction[].class, CommonError.class, path);
    }

    public Transaction getObtenerCheque(long bankId, String accountId, String viewId, String id) throws RestConnectorException {
        final String path = "/banks/" + bankId + "/accounts/" + accountId +"/" + viewId + "/transaction-request-types/CHECK/" + id;
        log.debug("[getObtenerCheque] Obteniendo cheque con path [{}]", path);
        return connector.genericGet(null, Transaction.class, CommonError.class, path);
    }

    // ----------------------------------------------------------------------------------------- //

    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        if (builder != null) {
            if (!doingLogin) builder.header("Authorization", "JWT " + getJWTToken(false));
            if (!headers.isEmpty()) {
                for (String headerKey : headers.keySet()) {
                    builder.header(headerKey, headers.get(headerKey));
                }
            }
        }
        return builder;
    }

    @Override
    public boolean retryOnUnauthorized() {
        if (!doingLogin) {
            try {
                getJWTToken(true);
                return true;
            } catch (RestConnectorException e) {
                log.error(e);
                return false;
            }
        }
        return false;
    }

    @Override
    public boolean getDisableHTTPSErrors() {
        return false;
    }

    public String getJWTToken(boolean force) throws RestConnectorException {
        final SimpleCache cache = SimpleCacheManager.getInstance().getCache(BindApiConnector.class.getName());
        final String cacheKey = SimpleCacheManager.buildKey("jwtToken");
        String jwtToken = (String) cache.get(cacheKey);

        if (force || jwtToken == null) {
            log.debug("[getJWTToken] Obteniendo token con username [{}]", username);

            // Objeto request
            PostLoginRequest request = new PostLoginRequest(username, password);

            log.debug("[getJWTToken] Por hacer login con {}", request);
            doingLogin = true;
            PostLoginResponse postLoginResponse = connector.genericPost(request, PostLoginResponse.class, CommonError.class, "/login/jwt");
            doingLogin = false;

            jwtToken = postLoginResponse.getToken();
            log.debug("[getJWTToken] Token recuperado [{}], expira en [{}] segs", jwtToken, postLoginResponse.getExpiresIn() - 10);
            cache.put(cacheKey, jwtToken, postLoginResponse.getExpiresIn() - 10);
        }
        return jwtToken;
    }
}
