package ar.com.sdd.commons.rest.core;

import ar.com.sdd.commons.rest.log.ClientLoggingFilter;
import org.apache.log4j.Logger;

import javax.ws.rs.ProcessingException;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.Entity;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.util.HashMap;
import java.util.Map;

public class RestConnector {

    private static final Logger log = Logger.getLogger(RestConnector.class);

    /**
     * URL de API
     */
    private String baseUrl;

    /**
     * El proveedor de seguridad que inyecta los headers necesarios
     */
    private SecurityManager securityManager;

    /**
     * Filtro del lado cliente
     */
    private ClientLoggingFilter clientLoggingFilter;

    public RestConnector(String baseUrl, SecurityManager securityManager) {
        this.baseUrl = baseUrl;
        this.securityManager = securityManager;

        //Inicializo los filtros
        clientLoggingFilter = new ClientLoggingFilter();
    }


    public <Res, Req> Res genericGet(Req request, Class<Res> responseType, String path) throws RestConnectorException {
        return genericGet(request, responseType, path, null);
    }

    public <Res, Req> Res genericGet(Req request, Class<Res> responseType, String path, String templateKey, String templateValue) throws RestConnectorException {
        Map<String, Object> templateParameters = new HashMap<>();
        templateParameters.put(templateKey, templateValue);
        return genericGet(request, responseType, path, templateParameters);
    }

    private <Res, Req> Res genericGet(Req request, Class<Res> responseType, String path, Map<String, Object> templateParameters) throws RestConnectorException {
        log.trace("Get cliente request [" + request + "]");

        try {
            WebTarget resource = ClientBuilder.newClient().target(baseUrl + path);
            resource.register(clientLoggingFilter);

            if (templateParameters != null) {
                resource = resource.resolveTemplates(templateParameters);
            }

            Response getResponse = securityManager.requestWithHeaders(resource)
                    .accept(MediaType.APPLICATION_JSON)
                    .get();
            log.trace("Get cliente response [" + getResponse.getStatus() + "/" + getResponse.getStatusInfo() + "]");

            if (getResponse.getStatus() != Response.Status.OK.getStatusCode()) {
                throw new RestConnectorException("La consulta no devolvio OK. Status: [" + getResponse.getStatus() + "/" + getResponse.getStatusInfo() + "]");
            }

            Res entityResponse;
            //Caso especial para cuando se invoca con Void y no se espera respuesta (solo que responda OK)
            if (getResponse.getEntity() == null && responseType == Void.class) {
                entityResponse = null;
                log.trace("Entity response [vacio]");
            } else {
                entityResponse = getResponse.readEntity(responseType);
                log.trace("Entity response [" + entityResponse + "]");
            }
            getResponse.close();

            return entityResponse;
        } catch (ProcessingException e) {
            throw new RestConnectorException("Error de comunicaciones", e);
        }

    }

    public <Res, Req> Res genericPost(Req request, Class<Res> responseType, String path) throws RestConnectorException {
        log.trace("Post cliente request [" + request + "]");

        try {
            WebTarget resource = ClientBuilder.newClient().target(baseUrl + path);
            resource.register(clientLoggingFilter);

            Response postResponse = securityManager.requestWithHeaders(resource)
                    .accept(MediaType.APPLICATION_JSON)
                    .post(Entity.entity(request, MediaType.APPLICATION_JSON));
            log.trace("Post cliente response [" + postResponse.getStatus() + "/" + postResponse.getStatusInfo() + "]");

            if (postResponse.getStatus() != Response.Status.OK.getStatusCode()) {
                throw new RestConnectorException("La consulta no devolvio OK. Status: [" + postResponse.getStatus() + "/" + postResponse.getStatusInfo() + "]");
            }

            Res entityResponse;
            //Caso especial para cuando se invoca con Void y no se espera respuesta (solo que responda OK)
            if (postResponse.getEntity() == null && responseType == Void.class) {
                entityResponse = null;
                log.trace("Entity response [vacio]");
            } else {
                entityResponse = postResponse.readEntity(responseType);
                log.trace("Entity response [" + entityResponse + "]");
            }
            postResponse.close();

            return entityResponse;

        } catch (ProcessingException e) {
            throw new RestConnectorException("Error de comunicaciones", e);
        }
    }

    public <Res, Req> Res genericDelete(Req request, Class<Res> responseType, String path, Map<String, Object> templateParameters) throws RestConnectorException {
        log.trace("Delete cliente request [" + request + "]");

        try {
            WebTarget resource = ClientBuilder.newClient().target(baseUrl + path);
            resource.register(clientLoggingFilter);

            if (templateParameters != null) {
                resource = resource.resolveTemplates(templateParameters);
            }

            Response deleteResponse = securityManager.requestWithHeaders(resource)
                    .accept(MediaType.APPLICATION_JSON)
                    .delete();
            log.trace("Delete cliente response [" + deleteResponse.getStatus() + "/" + deleteResponse.getStatusInfo() + "]");

            if (deleteResponse.getStatus() != Response.Status.OK.getStatusCode()) {
                throw new RestConnectorException("La consulta no devolvio OK. Status: [" + deleteResponse.getStatus() + "/" + deleteResponse.getStatusInfo() + "]");
            }

            Res entityResponse = deleteResponse.readEntity(responseType);
            log.trace("Entity response [" + entityResponse + "]");
            deleteResponse.close();

            return entityResponse;

        } catch (ProcessingException e) {
            throw new RestConnectorException("Error de comunicaciones", e);
        }

    }

}
