package ar.com.sdd.commons.rest.log;

import ar.com.sdd.commons.rest.util.RestConnectorUtil;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;

import javax.annotation.Priority;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.client.ClientRequestContext;
import javax.ws.rs.client.ClientResponseContext;
import javax.ws.rs.ext.WriterInterceptorContext;
import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;

@Priority(Integer.MIN_VALUE)
public class ClientFileLoggingFilter implements ClientLoggingFilter {

    private static final Logger log = Logger.getLogger(ClientFileLoggingFilter.class);

    private static final String ENTITY_STREAM_PROPERTY = "ClientLoggingFilter.entityStream";

    /**
     * Request filter
     *
     * @param requestContext
     */
    @Override
    public void filter(ClientRequestContext requestContext) {
        String timestamp36 = RestConnectorUtil.getCacheSafeValue();
        requestContext.setProperty("timestamp36", timestamp36);
        String method = requestContext.getMethod() != null ? requestContext.getMethod() : "N/A";
        String uri = requestContext.getUri() != null ? requestContext.getUri().toString() : "N/A";
        String respuesta = "[" + timestamp36 + "]:[RestConnector] Client Request - Method [" + method + "], URI [" + uri + "]";

        if (requestContext.hasEntity()) {
            OutputStream stream = new LoggingStream(requestContext.getEntityStream(), timestamp36, respuesta);
            requestContext.setEntityStream(stream);
            requestContext.setProperty(ENTITY_STREAM_PROPERTY, stream);
        } else {
            log.info(respuesta + "(vacio)");
        }
    }

    @Override
    public void aroundWriteTo(WriterInterceptorContext context) throws IOException, WebApplicationException {
        final LoggingStream stream = (LoggingStream) context.getProperty(ENTITY_STREAM_PROPERTY);
        context.proceed();
        if (stream != null) {
            stream.logStream();
        }
    }

    private class LoggingStream extends FilterOutputStream {

        private final ByteArrayOutputStream baos = new ByteArrayOutputStream();
        private String timestamp36;
        private String respuesta;

        LoggingStream(OutputStream out, String timestamp36, String respuesta) {
            super(out);
            this.timestamp36 = timestamp36;
            this.respuesta = respuesta;
        }

        void logStream() throws IOException {
            File logFile = RestConnectorUtil.getLogFile(timestamp36, "client", "request");
            log.info(respuesta + ", logFile [" + logFile + "]");

            // write entity to the builder
            byte[] bytes = baos.toByteArray();

            try (BufferedOutputStream bos = new BufferedOutputStream(new FileOutputStream(logFile))) {
                bos.write((respuesta + "\n").getBytes(StandardCharsets.UTF_8));
                bos.write(bytes);

            } catch (Exception e) {
                log.error("[" + timestamp36 + "] No se pudo escribir el archivo de log [" + logFile + "]", e);
            }
        }

        @Override
        public void write(final int i) throws IOException {
            baos.write(i);
            out.write(i);
        }
    }


    /**
     * Response filter
     *
     * @param requestContext
     * @param responseContext
     * @throws IOException
     */
    @Override
    public void filter(ClientRequestContext requestContext, ClientResponseContext responseContext) throws IOException {
        String timestamp36 = (String) requestContext.getProperty("timestamp36");
        if (StringUtils.isEmpty(timestamp36)) {
            timestamp36 = RestConnectorUtil.getCacheSafeValue();
            log.warn("No se pudo recuperar el timestamp36, se computa uno nuevo [" + timestamp36 + "]");
        }
        String statusCode = responseContext.getStatusInfo() != null ? String.valueOf(responseContext.getStatusInfo().getStatusCode()) : "N/A";
        String respuesta = "[" + timestamp36 + "]:[RestConnector] Client Response - Method [" + requestContext.getMethod() + "], Status [" + statusCode + "], URI [" + requestContext.getUri() + "]";

        //Analizo si la respuesta viene chunked
        boolean chunked = responseContext.getHeaders() != null
                && responseContext.getHeaderString("Transfer-Encoding") != null
                && responseContext.getHeaderString("Transfer-Encoding").equalsIgnoreCase("chunked");

        if (responseContext.hasEntity() && !chunked) {
            responseContext.setEntityStream(logInboundEntity(responseContext.getEntityStream(), timestamp36, respuesta));

        } else {
            if (chunked) {
                //FIXME encontrar la manera de loguear los bloques chunked
                log.info(respuesta + "(chunked)");
            } else {
                log.info(respuesta + "(vacio)");
            }
        }
    }

    private InputStream logInboundEntity(InputStream stream, String timestamp36, String respuesta) throws IOException {
        String result = IOUtils.toString(stream, StandardCharsets.UTF_8);
        File logFile = RestConnectorUtil.getLogFile(timestamp36, "client", "response");
        log.info(respuesta + ", logFile [" + logFile + "]");

        try (BufferedOutputStream bos = new BufferedOutputStream(new FileOutputStream(logFile))) {
            bos.write((respuesta + "\n").getBytes(StandardCharsets.UTF_8));
            bos.write(result.getBytes(StandardCharsets.UTF_8));

        } catch (Exception e) {
            log.error("[" + timestamp36 + "] No se pudo escribir el archivo de log [" + logFile + "]", e);
        }

        stream.reset();
        return stream;
    }

}
