package ar.com.sdd.commons.rest.log;

import ar.com.sdd.commons.rest.util.RestConnectorUtil;
import org.apache.commons.io.IOUtils;
import org.apache.log4j.Logger;

import javax.annotation.Priority;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.client.ClientRequestContext;
import javax.ws.rs.client.ClientResponseContext;
import javax.ws.rs.core.Response;
import javax.ws.rs.ext.WriterInterceptorContext;
import java.io.ByteArrayOutputStream;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;

@Priority(Integer.MIN_VALUE)
public class ClientLoggerLoggingFilter implements ClientLoggingFilter {

    private static final Logger log = Logger.getLogger(ClientLoggerLoggingFilter.class);
    private static final Logger logAll = Logger.getLogger("ar.com.sdd.commons.rest.log.LogAll");

    private static final String ENTITY_STREAM_PROPERTY = "ClientLoggingFilter.entityStream";
    private static final Charset DEFAULT_CHARSET = StandardCharsets.UTF_8;

    //REQUEST

    @Override
    public void filter(ClientRequestContext requestContext) {
        if (requestContext.hasEntity()) {
            final OutputStream stream = new LoggingStream(requestContext.getEntityStream(), requestContext.getUri(), requestContext.getMethod());
            requestContext.setEntityStream(stream);
            requestContext.setProperty(ENTITY_STREAM_PROPERTY, stream);
        } else {
            final StringBuilder sb = new StringBuilder("[" + RestConnectorUtil.getCacheSafeValue() + "]:");
            String uri = requestContext.getUri() != null ? requestContext.getUri().toString() : "N/A";
            String method = requestContext.getMethod() != null ? requestContext.getMethod() : "N/A";
            sb.append("[RestConnector] Client Request - Method [").append(method).append("], URI [").append(uri).append("]:NO_ENTITY");
            log.info(sb);
            logAll.trace(sb);
        }
    }

    @Override
    public void aroundWriteTo(WriterInterceptorContext context) throws IOException, WebApplicationException {
        final LoggingStream stream = (LoggingStream) context.getProperty(ENTITY_STREAM_PROPERTY);
        context.proceed();
        if (stream != null) {
            stream.logStream();
        }
    }

    private class LoggingStream extends FilterOutputStream {

        private final StringBuilder sb = new StringBuilder("[" + RestConnectorUtil.getCacheSafeValue() + "]:");
        private final ByteArrayOutputStream baos = new ByteArrayOutputStream();
        private URI uri;
        private String method;

        LoggingStream(OutputStream out, URI uri, String method) {
            super(out);
            this.uri = uri;
            this.method = method;
        }

        void logStream() {
            // write entity to the builder
            final byte[] entity = baos.toByteArray();

            sb.append("[RestConnector] Client Request - Method [").append(method).append("], URI [").append(uri).append("]:");
            sb.append(new String(entity, 0, entity.length, DEFAULT_CHARSET));
            logAll.trace(sb);

            if (entity.length > RestConnectorUtil.MAX_ENTITY_SIZE) {
                log.info(sb.substring(0, RestConnectorUtil.MAX_ENTITY_SIZE) + "...more...");
            } else {
                log.info(sb);
            }
        }

        @Override
        public void write(final int i) throws IOException {
            baos.write(i);
            out.write(i);
        }
    }




    //RESPONSE

    @Override
    public void filter(ClientRequestContext requestContext, ClientResponseContext responseContext) throws IOException {
        final StringBuilder sb = new StringBuilder("[" + RestConnectorUtil.getCacheSafeValue() + "]:");

        //Analizo si la respuesta viene chunked
        boolean chunked = responseContext.getHeaders() != null
                && responseContext.getHeaderString("Transfer-Encoding") != null
                && responseContext.getHeaderString("Transfer-Encoding").equalsIgnoreCase("chunked");

        if (responseContext.hasEntity()) {
            responseContext.setEntityStream(logInboundEntity(responseContext.getStatusInfo(), responseContext.getEntityStream(), chunked));
        } else {
            String statusCode = responseContext.getStatusInfo() != null ? String.valueOf(responseContext.getStatusInfo().getStatusCode()) : "N/A";
            sb.append("[RestConnector] Client Response - Status [").append(statusCode).append("]:NO_ENTITY");
        }
        log.info(sb);
        logAll.trace(sb);
    }

    private InputStream logInboundEntity(Response.StatusType statusInfo, InputStream stream, boolean chunked) throws IOException {
        String result = IOUtils.toString(stream, DEFAULT_CHARSET);
        StringBuilder sb = new StringBuilder("[" + RestConnectorUtil.getCacheSafeValue() + "]:");
        sb.append("[RestConnector] Client Response - Status [").append(statusInfo.getStatusCode()).append("]:");
        if (chunked) {
            //FIXME encontrar la manera de loguear los bloques chunked
            sb.append("[chunked]");
            log.info(sb);
            logAll.trace(sb);

        } else {
            if (result.length() > RestConnectorUtil.MAX_ENTITY_SIZE) {
                log.info(sb.substring(0, RestConnectorUtil.MAX_ENTITY_SIZE) + "...more...");
            } else {
                log.info(sb);
            }
            logAll.trace(sb);
        }
        stream.reset();
        return stream;
    }

}
