package ar.com.sdd.commons.rest.core;

import ar.com.sdd.commons.rest.log.ClientFileLoggingFilter;
import ar.com.sdd.commons.rest.log.ClientLoggerLoggingFilter;
import ar.com.sdd.commons.rest.log.ClientLoggingFilter;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;

import javax.net.ssl.SSLSession;
import javax.ws.rs.ProcessingException;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.Entity;
import javax.ws.rs.client.Invocation;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.io.FileInputStream;
import java.security.KeyStore;
import java.util.HashMap;
import java.util.Map;

public class RestConnector {

    private static final Logger log = Logger.getLogger(RestConnector.class);

    /**
     * Todos los datos para la conexion
     */
    private RestConnectorEnvironment environment;

    /**
     * El proveedor de seguridad que inyecta los headers necesarios
     */
    private RestSecurityManager securityManager;

    /**
     * KeyStore de certificados privados (nuestros)
     */
    private KeyStore keyStore;

    /**
     * KeyStore de certificados publicos (ellos)
     */
    private KeyStore trustStore;

    /**
     * Filtro del lado cliente
     */
    private ClientLoggingFilter clientLoggingFilter;


    public RestConnector(RestConnectorEnvironment environment, RestSecurityManager securityManager) {
        this(environment, securityManager, false);
    }

    public RestConnector(RestConnectorEnvironment environment, RestSecurityManager securityManager, boolean logClientToFile) {
        this.environment = environment;
        this.securityManager = securityManager;

        //Inicializo los KeyStores (de haber)
        if (!StringUtils.isEmpty(environment.keyStorePath)) {
            try {
                keyStore = loadStore(environment.keyStorePath, environment.keyStorePassword);
            } catch (Exception e) {
                log.error("No se pudo cargar el keystore de [" + environment.keyStorePath + "] con password [" + environment.keyStorePassword + "]", e);
            }
        }

        if (!StringUtils.isEmpty(environment.trustStorePath)) {
            try {
                trustStore = loadStore(environment.trustStorePath, environment.trustStorePassword);
            } catch (Exception e) {
                log.error("No se pudo cargar el trustStore de [" + environment.trustStorePath + "] con password [" + environment.trustStorePassword + "]", e);
            }
        }


        //Inicializo los filtros
        if (logClientToFile) {
            clientLoggingFilter = new ClientFileLoggingFilter();
        } else {
            clientLoggingFilter = new ClientLoggerLoggingFilter();
        }
    }


    public <Res, Req> Res genericGet(Req request, Class<Res> responseType, String path) throws RestConnectorException {
        return genericGet(request, responseType, path, null);
    }

    public <Res, Req, Err> Res genericGet(Req request, Class<Res> responseType, Class<Err> errorType, String path) throws RestConnectorException {
        return genericGet(request, responseType, errorType, path, null);
    }

    public <Res, Req> Res genericGet(Req request, Class<Res> responseType, String path, String templateKey, String templateValue) throws RestConnectorException {
        Map<String, Object> templateParameters = new HashMap<>();
        templateParameters.put(templateKey, templateValue);
        return genericGet(request, responseType, path, templateParameters);
    }

    private <Res, Req> Res genericGet(Req request, Class<Res> responseType, String path, Map<String, Object> templateParameters) throws RestConnectorException {
        return genericGet(request, responseType, null, path, templateParameters);
    }

    private <Res, Req, Err> Res genericGet(Req request, Class<Res> responseType, Class<Err> errorType, String path, Map<String, Object> templateParameters) throws RestConnectorException {
        log.trace("[RestConnector] Get cliente request [" + request + "]");

        Response getResponse = null;
        try {
            getResponse = buildRequest(environment.baseUrl + path, templateParameters)
                    .accept(MediaType.APPLICATION_JSON)
                    .get();
            log.trace("[RestConnector] Get cliente response [" + getResponse.getStatus() + "/" + getResponse.getStatusInfo() + "]");

            checkSuccessfulOrException(getResponse, errorType);

            return processResponse(getResponse, responseType);

        } catch (ProcessingException e) {
            throw new RestConnectorException("Error de comunicaciones", e);

        } finally {
            if (getResponse != null) {
                getResponse.close();
            }
        }

    }

    public <Res, Req> Res genericPost(Req request, Class<Res> responseType, String path) throws RestConnectorException {
        return genericPost(request, responseType, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON);
    }

    public <Res, Req> Res genericPost(Req request, Class<Res> responseType, String path, String acceptMediaType, String postMediaType) throws RestConnectorException {
        return genericPost(request, responseType,null, path, acceptMediaType, postMediaType);
    }

    public <Res, Req, Err> Res genericPost(Req request, Class<Res> responseType, Class<Err> errorType, String path) throws RestConnectorException {
        return genericPost(request, responseType, errorType, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON);
    }

    public <Res, Req, Err> Res genericPost(Req request, Class<Res> responseType, Class<Err> errorType, String path, String acceptMediaType, String postMediaType) throws RestConnectorException {
        log.trace("[RestConnector] Post cliente request [" + request + "]");

        Response postResponse = null;
        try {
            postResponse = buildRequest(environment.baseUrl + path, null)
                    .accept(acceptMediaType)
                    .post(Entity.entity(request, postMediaType));
            log.trace("[RestConnector] Post cliente response [" + postResponse.getStatus() + "/" + postResponse.getStatusInfo() + "]");

            checkSuccessfulOrException(postResponse, errorType);

            return processResponse(postResponse, responseType);

        } catch (ProcessingException e) {
            throw new RestConnectorException("Error de comunicaciones", e);

        } finally {
            if (postResponse != null) {
                postResponse.close();
            }
        }
    }


    public <Res, Req> Res genericDelete(Req request, Class<Res> responseType, String path, Map<String, Object> templateParameters) throws RestConnectorException {
        return genericDelete(request, responseType, null, path, templateParameters);
    }

    public <Res, Req, Err> Res genericDelete(Req request, Class<Res> responseType, Class<Err> errorType, String path, Map<String, Object> templateParameters) throws RestConnectorException {
        log.trace("[RestConnector] Delete cliente request [" + request + "]");

        Response deleteResponse = null;
        try {
            deleteResponse = buildRequest(environment.baseUrl + path, templateParameters)
                    .accept(MediaType.APPLICATION_JSON)
                    .delete();
            log.trace("[RestConnector] Delete cliente response [" + deleteResponse.getStatus() + "/" + deleteResponse.getStatusInfo() + "]");

            checkSuccessfulOrException(deleteResponse, errorType);

            Res entityResponse = deleteResponse.readEntity(responseType);
            log.trace("[RestConnector] Entity response [" + entityResponse + "]");

            return entityResponse;

        } catch (ProcessingException e) {
            throw new RestConnectorException("Error de comunicaciones", e);

        } finally {
            if (deleteResponse != null) {
                deleteResponse.close();
            }
        }

    }


    /**
     * Comprueba que venga una respuesta successful (200 o 2NN), y sino tira una excepcion. Si vino informado un errorType (el body de la respuesta erronea tiene un json),
     * este se sube a la excepcion.
     *
     * @param response
     * @param errorType
     * @param <Err>
     * @throws RestConnectorException
     */
    private <Err> void checkSuccessfulOrException(Response response, Class<Err> errorType) throws RestConnectorException {
        if (response.getStatusInfo().getFamily() != Response.Status.Family.SUCCESSFUL) {
            Object entityResponseError = null;
            if (errorType != null) {
                entityResponseError = response.readEntity(errorType);
            }
            throw new RestConnectorException("La consulta no devolvio OK. Status: [" + response.getStatus() + "/" + response.getStatusInfo() + "]", response.getStatus(), response.getStatusInfo(), entityResponseError);
        }
    }

    /**
     * Devuelve la response con la entity correcta considerando el caso en que pueda venir vacia
     *
     * @param response
     * @param responseType
     * @param <Res>
     * @return
     */
    private <Res> Res processResponse(Response response, Class<Res> responseType) {
        Res entityResponse;
        //Caso especial para cuando se invoca con Void y no se espera respuesta (solo que responda OK)
        if (response.getEntity() == null && responseType == Void.class) {
            entityResponse = null;
            log.trace("[RestConnector] Entity response [vacio]");
        } else {
            entityResponse = response.readEntity(responseType);
            log.trace("[RestConnector] Entity response [" + entityResponse + "]");
        }
        return entityResponse;
    }

    public Invocation.Builder buildRequest(String url, Map<String, Object> templateParameters) throws RestConnectorException {

        ClientBuilder builder = ClientBuilder.newBuilder();

        if (keyStore != null) {
            builder.keyStore(keyStore, environment.keyStorePassword);
        }

        if (trustStore != null) {
            builder.trustStore(trustStore);
            builder.hostnameVerifier((String hostName, SSLSession session) -> true);
        }

        WebTarget resource = builder.build().target(url);
        resource.register(clientLoggingFilter);

        if (templateParameters != null) {
            resource = resource.resolveTemplates(templateParameters);
        }

        Invocation.Builder request = resource.request();

        if (securityManager != null) {
            request = securityManager.addHeaders(request);
        }

        return request;
    }

    private KeyStore loadStore(String storePath, String storePassword) throws Exception {
        KeyStore store = KeyStore.getInstance("JKS");
        store.load(new FileInputStream(storePath), storePassword.toCharArray());
        return store;
    }

}
