package ar.com.sdd.commons.rest.log;

import ar.com.sdd.commons.rest.util.RestConnectorUtil;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;

import javax.ws.rs.container.ContainerRequestContext;
import javax.ws.rs.container.ContainerRequestFilter;
import javax.ws.rs.container.ContainerResponseContext;
import javax.ws.rs.container.ContainerResponseFilter;
import javax.ws.rs.ext.Provider;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.charset.StandardCharsets;

@ServerFileLogged
@Provider
public class ServerFileLogginFilter implements ContainerRequestFilter, ContainerResponseFilter {

    private static final Logger log = Logger.getLogger(ServerFileLogginFilter.class);

    /**
     * Request filter
     *
     * @param requestContext
     * @throws IOException
     */
    @Override
    public void filter(ContainerRequestContext requestContext) throws IOException {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        IOUtils.copy(requestContext.getEntityStream(), baos);
        String timestamp36 = RestConnectorUtil.getCacheSafeValue();
        requestContext.setProperty("timestamp36", timestamp36);

        String method = requestContext.getMethod();
        String uri = requestContext.getUriInfo().getRequestUri() != null ? requestContext.getUriInfo().getRequestUri().toString() : "";
        String respuesta = "[" + timestamp36 + "]:[RestConnector] Server Request - Method [" + method + "], URI [" + uri + "]";

        String bearer = requestContext.getHeaderString("Bearer");
        if (!StringUtils.isEmpty(bearer)) {
            requestContext.setProperty("bearer", bearer);
            respuesta += ", Bearer [" + bearer + "]";
        }

        File logFile = RestConnectorUtil.getLogFile(timestamp36, "server", "request", RestConnectorUtil.createTag(method, uri));
        log.info(respuesta + ", logFile [" + logFile + "]");
        byte[] bytes = baos.toByteArray();
        try (BufferedOutputStream bos = new BufferedOutputStream(new FileOutputStream(logFile))) {
            bos.write((respuesta + "\n").getBytes(StandardCharsets.UTF_8));
            bos.write(bytes);
        } catch (Exception e) {
            log.error("[" + timestamp36 + "] No se pudo escribir el archivo de log [" + logFile + "]", e);
        }

        requestContext.setEntityStream(new ByteArrayInputStream(bytes));
    }

    /**
     * Response filter
     *
     * @param requestContext
     * @param responseContext
     * @throws IOException
     */
    @Override
    public void filter(ContainerRequestContext requestContext, ContainerResponseContext responseContext) throws IOException {
        String timestamp36 = (String) requestContext.getProperty("timestamp36");
        if (StringUtils.isEmpty(timestamp36)) {
            timestamp36 = RestConnectorUtil.getCacheSafeValue();
            log.warn("No se pudo recuperar el timestamp36, se computa uno nuevo [" + timestamp36 + "]");
        }
        String method = requestContext.getMethod();
        String uri = requestContext.getUriInfo().getRequestUri() != null ? requestContext.getUriInfo().getRequestUri().toString() : "";
        String respuesta = "[" + timestamp36 + "]:[RestConnector] Server Response - Method [" + method + "], Status [" + responseContext.getStatus() + "], URI [" + uri + "]";
        String bearer = (String) requestContext.getProperty("bearer");
        if (!StringUtils.isEmpty(bearer)) {
            respuesta += ", Bearer [" + bearer + "]";
        }
        Object entity = responseContext.getEntity();
        if (entity != null) {
            File logFile = RestConnectorUtil.getLogFile(timestamp36, "server", "response", RestConnectorUtil.createTag(method, uri));
            log.info(respuesta + ", logFile [" + logFile + "]");
            try (PrintWriter logFileWriter = new PrintWriter(new BufferedOutputStream(new FileOutputStream(logFile)))) {
                logFileWriter.write(respuesta + "\n");
                new ObjectMapper().writerWithDefaultPrettyPrinter().writeValue(logFileWriter, entity);

            } catch (Exception e) {
                log.error("[" + timestamp36 + "] No se pudo escribir el archivo de log [" + logFile + "]", e);
            }

        } else {
            log.info(respuesta + "(vacio)");
        }
    }

}
