package ar.com.sdd.commons.rest.core;

import ar.com.sdd.commons.rest.log.ClientLoggingFilter;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;

import javax.net.ssl.SSLSession;
import javax.ws.rs.ProcessingException;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.Entity;
import javax.ws.rs.client.Invocation;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.io.FileInputStream;
import java.security.KeyStore;
import java.util.HashMap;
import java.util.Map;

public class RestConnector {

    private static final Logger log = Logger.getLogger(RestConnector.class);

    /**
     * Todos los datos para la conexion
     */
    private RestConnectorEnvironment environment;

    /**
     * El proveedor de seguridad que inyecta los headers necesarios
     */
    private RestSecurityManager securityManager;

    /**
     * KeyStore de certificados privados (nuestros)
     */
    private KeyStore keyStore;

    /**
     * KeyStore de certificados publicos (ellos)
     */
    private KeyStore trustStore;

    /**
     * Filtro del lado cliente
     */
    private ClientLoggingFilter clientLoggingFilter;

    public RestConnector(RestConnectorEnvironment environment, RestSecurityManager securityManager) {
        this.environment = environment;
        this.securityManager = securityManager;

        //Inicializo los KeyStores (de haber)
        if (!StringUtils.isEmpty(environment.keyStorePath)) {
            try {
                keyStore = loadStore(environment.keyStorePath, environment.keyStorePassword);
            } catch (Exception e) {
                log.error("No se pudo cargar el keystore de [" + environment.keyStorePath + "] con password [" + environment.keyStorePassword + "]", e);
            }

            try {
                trustStore = loadStore(environment.trustStorePath, environment.trustStorePassword);
            } catch (Exception e) {
                log.error("No se pudo cargar el trustStore de [" + environment.trustStorePath + "] con password [" + environment.trustStorePassword + "]", e);
            }

        }


        //Inicializo los filtros
        clientLoggingFilter = new ClientLoggingFilter();
    }


    public <Res, Req> Res genericGet(Req request, Class<Res> responseType, String path) throws RestConnectorException {
        return genericGet(request, responseType, path, null);
    }

    public <Res, Req> Res genericGet(Req request, Class<Res> responseType, String path, String templateKey, String templateValue) throws RestConnectorException {
        Map<String, Object> templateParameters = new HashMap<>();
        templateParameters.put(templateKey, templateValue);
        return genericGet(request, responseType, path, templateParameters);
    }

    private <Res, Req> Res genericGet(Req request, Class<Res> responseType, String path, Map<String, Object> templateParameters) throws RestConnectorException {
        log.trace("[RestConnector] Get cliente request [" + request + "]");

        try {
            Response getResponse = buildRequest(environment.baseUrl + path, templateParameters)
                    .accept(MediaType.APPLICATION_JSON)
                    .get();
            log.trace("[RestConnector] Get cliente response [" + getResponse.getStatus() + "/" + getResponse.getStatusInfo() + "]");

            if (getResponse.getStatusInfo().getFamily() != Response.Status.Family.SUCCESSFUL) {
                throw new RestConnectorException("La consulta no devolvio OK. Status: [" + getResponse.getStatus() + "/" + getResponse.getStatusInfo() + "]");
            }

            Res entityResponse;
            //Caso especial para cuando se invoca con Void y no se espera respuesta (solo que responda OK)
            if (getResponse.getEntity() == null && responseType == Void.class) {
                entityResponse = null;
                log.trace("[RestConnector] Entity response [vacio]");
            } else {
                entityResponse = getResponse.readEntity(responseType);
                log.trace("[RestConnector] Entity response [" + entityResponse + "]");
            }
            getResponse.close();

            return entityResponse;
        } catch (ProcessingException e) {
            throw new RestConnectorException("Error de comunicaciones", e);
        }

    }

    public <Res, Req> Res genericPost(Req request, Class<Res> responseType, String path) throws RestConnectorException {
        return genericPost(request, responseType, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON);
    }

    public <Res, Req> Res genericPost(Req request, Class<Res> responseType, String path, String acceptMediaType, String postMediaType) throws RestConnectorException {
        log.trace("[RestConnector] Post cliente request [" + request + "]");

        try {
            Response postResponse = buildRequest(environment.baseUrl + path, null)
                    .accept(acceptMediaType)
                    .post(Entity.entity(request, postMediaType));
            log.trace("[RestConnector] Post cliente response [" + postResponse.getStatus() + "/" + postResponse.getStatusInfo() + "]");

            if (postResponse.getStatusInfo().getFamily() != Response.Status.Family.SUCCESSFUL) {
                throw new RestConnectorException("La consulta no devolvio OK. Status: [" + postResponse.getStatus() + "/" + postResponse.getStatusInfo() + "]");
            }

            Res entityResponse;
            //Caso especial para cuando se invoca con Void y no se espera respuesta (solo que responda OK)
            if (postResponse.getEntity() == null && responseType == Void.class) {
                entityResponse = null;
                log.trace("[RestConnector] Entity response [vacio]");
            } else {
                entityResponse = postResponse.readEntity(responseType);
                log.trace("[RestConnector] Entity response [" + entityResponse + "]");
            }
            postResponse.close();

            return entityResponse;

        } catch (ProcessingException e) {
            throw new RestConnectorException("Error de comunicaciones", e);
        }
    }

    public <Res, Req> Res genericDelete(Req request, Class<Res> responseType, String path, Map<String, Object> templateParameters) throws RestConnectorException {
        log.trace("[RestConnector] Delete cliente request [" + request + "]");

        try {
            Response deleteResponse = buildRequest(environment.baseUrl + path, templateParameters)
                    .accept(MediaType.APPLICATION_JSON)
                    .delete();
            log.trace("[RestConnector] Delete cliente response [" + deleteResponse.getStatus() + "/" + deleteResponse.getStatusInfo() + "]");

            if (deleteResponse.getStatusInfo().getFamily() != Response.Status.Family.SUCCESSFUL) {
                throw new RestConnectorException("La consulta no devolvio OK. Status: [" + deleteResponse.getStatus() + "/" + deleteResponse.getStatusInfo() + "]");
            }

            Res entityResponse = deleteResponse.readEntity(responseType);
            log.trace("[RestConnector] Entity response [" + entityResponse + "]");
            deleteResponse.close();

            return entityResponse;

        } catch (ProcessingException e) {
            throw new RestConnectorException("Error de comunicaciones", e);
        }

    }


    public Invocation.Builder buildRequest(String url, Map<String, Object> templateParameters) throws RestConnectorException {

        ClientBuilder builder = ClientBuilder.newBuilder();

        if (keyStore != null) {
            builder.keyStore(keyStore, environment.keyStorePassword);
        }

        if (trustStore != null) {
            builder.trustStore(trustStore);
            builder.hostnameVerifier((String hostName, SSLSession session) -> true);
        }


        WebTarget resource = ClientBuilder.newClient().target(url);
        resource.register(clientLoggingFilter);

        if (templateParameters != null) {
            resource = resource.resolveTemplates(templateParameters);
        }

        Invocation.Builder request = resource.request();

        if (securityManager != null) {
            request = securityManager.addHeaders(request);
        }

        return request;
    }

    private KeyStore loadStore(String storePath, String storePassword) throws Exception {
        KeyStore store = KeyStore.getInstance("JKS");
        store.load(new FileInputStream(storePath), storePassword.toCharArray());
        return store;
    }

}
