package ar.com.sdd.commons.rest.core;

import ar.com.sdd.commons.rest.log.ClientFileLoggingFilter;
import ar.com.sdd.commons.rest.log.ClientLoggerLoggingFilter;
import ar.com.sdd.commons.rest.log.ClientLoggingFilter;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.logging.log4j.*;

import javax.net.ssl.SSLSession;
import javax.ws.rs.ProcessingException;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.Entity;
import javax.ws.rs.client.Invocation;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.core.Response;
import java.io.FileInputStream;
import java.security.KeyStore;
import java.util.HashMap;
import java.util.Map;

/*
    Para debugear conexions, agregar esto al entorno (o a la runConfiguraicon en VM Options
            -Djavax.net.debug=all

    Genera MUCHO log de la conexion SL

 */


public class RestConnector {

    private static final Logger log = LogManager.getLogger(RestConnector.class);

    /**
     * Todos los datos para la conexion
     */
    private RestConnectorEnvironment environment;

    /**
     * El proveedor de seguridad que inyecta los headers necesarios
     */
    private RestSecurityManager securityManager;

    /**
     * KeyStore de certificados privados (nuestros)
     */
    private KeyStore keyStore;

    /**
     * KeyStore de certificados publicos (ellos)
     */
    private KeyStore trustStore;

    /**
     * Cuando los certificados del trustStore vienen dentro del keyStore, activar esta opcion
     */
    private boolean useKeyStoreWithEmbeddedTrustStore;

    /**
     * Filtro del lado cliente
     */
    private ClientLoggingFilter clientLoggingFilter;


    public RestConnector(RestConnectorEnvironment environment, RestSecurityManager securityManager) {
        this(environment, securityManager, true);
    }

    public RestConnector(RestConnectorEnvironment environment, RestSecurityManager securityManager, boolean logClientToFile) {
        this(environment, securityManager, true, false);
    }

    public RestConnector(RestConnectorEnvironment environment, RestSecurityManager securityManager, boolean logClientToFile, boolean useKeyStoreWithEmbeddedTrustStore) {
        this.environment = environment;
        this.securityManager = securityManager;

        //Inicializo los KeyStores (de haber)
        if (!StringUtils.isEmpty(environment.keyStorePath)) {
            try {
                keyStore = loadStore(environment.keyStorePath, environment.keyStorePassword);
            } catch (Exception e) {
                log.error("No se pudo cargar el keystore de [" + environment.keyStorePath + "] con password [" + environment.keyStorePassword + "]", e);
            }
        }

        if (!StringUtils.isEmpty(environment.trustStorePath)) {
            try {
                trustStore = loadStore(environment.trustStorePath, environment.trustStorePassword);
            } catch (Exception e) {
                log.error("No se pudo cargar el trustStore de [" + environment.trustStorePath + "] con password [" + environment.trustStorePassword + "]", e);
            }
        }

        if (useKeyStoreWithEmbeddedTrustStore) {
            log.debug("Usando el keyStore [" + environment.keyStorePath + "] como trustStore");
            trustStore = keyStore;
        }

        //Inicializo los filtros
        if (logClientToFile) {
            clientLoggingFilter = new ClientFileLoggingFilter();
        } else {
            clientLoggingFilter = new ClientLoggerLoggingFilter();
        }
    }


    public <Res, Req> Res genericGet(Req request, Class<Res> responseType, String path) throws RestConnectorException {
        return genericGet(request, responseType, null, path, (Map<String, Object>) null);
    }

    public <Res, Req, Err> Res genericGet(Req request, Class<Res> responseType, Class<Err> errorType, String path) throws RestConnectorException {
        return genericGet(request, responseType, errorType, path, null);
    }

    public <Res, Req> Res genericGet(Req request, Class<Res> responseType, String path, String templateKey, String templateValue) throws RestConnectorException {
        return genericGet(request, responseType, null, path, templateKey, templateValue);
    }

    public <Res, Req, Err> Res genericGet(Req request, Class<Res> responseType, Class<Err> errorType, String path, String templateKey, String templateValue) throws RestConnectorException {
        Map<String, Object> templateParameters = new HashMap<>();
        templateParameters.put(templateKey, templateValue);
        return genericGet(request, responseType, errorType, path, templateParameters);
    }

    public  <Res, Req, Err> Res genericGet(Req request, Class<Res> responseType, Class<Err> errorType, String path, Map<String, Object> templateParameters) throws RestConnectorException {
        return genericGetWithResponseHeader(request, responseType, errorType, path, templateParameters).getKey();
    }

    private <Res, Req, Err> Pair<Res, MultivaluedMap<String,Object>> genericGetWithResponseHeader(Req request, Class<Res> responseType, Class<Err> errorType, String path, Map<String, Object> templateParameters) throws RestConnectorException {
        log.trace("[RestConnector] Get cliente request [" + request + "]");

        Response getResponse = null;
        try {
            getResponse = buildRequest(environment.baseUrl + path, templateParameters)
                    .accept(MediaType.APPLICATION_JSON)
                    .get();
            log.trace("[RestConnector] Get cliente response [" + getResponse.getStatus() + "/" + getResponse.getStatusInfo() + "]");

            checkSuccessfulOrException(getResponse, errorType);

            return Pair.of(processResponse(getResponse, responseType), getResponse.getHeaders());

        } catch (ProcessingException e) {
            throw new RestConnectorException("Error de comunicaciones", e);

        } finally {
            if (getResponse != null) {
                getResponse.close();
            }
        }

    }

    public <Req, Err> String genericGetTextResponse(Req request, Class<Err> errorType, String path, Map<String, Object> templateParameters) throws RestConnectorException {
        log.trace("[RestConnector] Get cliente request [" + request + "] con text response");

        Response getResponse = null;
        try {
            getResponse = buildRequest(environment.baseUrl + path, templateParameters)
                    .accept(MediaType.TEXT_PLAIN)
                    .get();
            log.trace("[RestConnector] Get cliente text response [" + getResponse.getStatus() + "/" + getResponse.getStatusInfo() + "]");

            checkSuccessfulOrException(getResponse, errorType);

            return processResponse(getResponse, String.class);

        } catch (ProcessingException e) {
            throw new RestConnectorException("Error de comunicaciones", e);

        } finally {
            if (getResponse != null) {
                getResponse.close();
            }
        }

    }

    public <Res, Req> Res genericPost(Req request, Class<Res> responseType, String path) throws RestConnectorException {
        return genericPost(request, responseType, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON);
    }

    public <Res, Req, Err> Pair<Res, MultivaluedMap<String,Object>> genericPostWithResponseHeader(Req request, Class<Res> responseType, String path) throws RestConnectorException {
        return genericPostWithResponseHeader(request, responseType, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON);
    }

    public <Res, Req> Res genericPost(Req request, Class<Res> responseType, String path, String acceptMediaType, String postMediaType) throws RestConnectorException {
        return genericPost(request, responseType,null, path, acceptMediaType, postMediaType);
    }

    public <Res, Req, Err> Pair<Res, MultivaluedMap<String,Object>> genericPostWithResponseHeader(Req request, Class<Res> responseType, String path, String acceptMediaType, String postMediaType) throws RestConnectorException {
        return genericPostWithResponseHeader(request, responseType, null, path, acceptMediaType, postMediaType);
    }

    public <Res, Req, Err> Res genericPost(Req request, Class<Res> responseType, Class<Err> errorType, String path) throws RestConnectorException {
        return genericPost(request, responseType, errorType, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON);
    }

    public <Res, Req, Err> Pair<Res, MultivaluedMap<String,Object>> genericPostWithResponseHeader(Req request, Class<Res> responseType, Class<Err> errorType, String path) throws RestConnectorException {
        return genericPostWithResponseHeader(request, responseType, errorType, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON);
    }

    public <Res, Req, Err> Res genericPost(Req request, Class<Res> responseType, Class<Err> errorType, String path, String acceptMediaType, String postMediaType) throws RestConnectorException {
        return genericPostWithResponseHeader(request, responseType, errorType, path, acceptMediaType, postMediaType).getKey();
    }

    public <Res, Req, Err> Pair<Res, MultivaluedMap<String,Object>> genericPostWithResponseHeader(Req request, Class<Res> responseType, Class<Err> errorType, String path, String acceptMediaType, String postMediaType) throws RestConnectorException {
        log.trace("[RestConnector] Post cliente request [" + request + "]");

        Response postResponse = null;
        try {
            postResponse = buildRequest(environment.baseUrl + path, null)
                    .accept(acceptMediaType)
                    .post(Entity.entity(request, postMediaType));
            log.trace("[RestConnector] Post cliente response [" + postResponse.getStatus() + "/" + postResponse.getStatusInfo() + "]");

            checkSuccessfulOrException(postResponse, errorType);

            return Pair.of(processResponse(postResponse, responseType), postResponse.getHeaders());

        } catch (ProcessingException e) {
            throw new RestConnectorException("Error de comunicaciones", e);

        } finally {
            if (postResponse != null) {
                postResponse.close();
            }
        }
    }

    public <Res, Req> Res genericPut(Req request, Class<Res> responseType, String path) throws RestConnectorException {
        return genericPut(request, responseType, null, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON);
    }

    public <Res, Req, Err> Res genericPut(Req request, Class<Res> responseType, Class<Err> errorType, String path) throws RestConnectorException {
        return genericPut(request, responseType, errorType, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON);
    }

    public <Res, Req, Err> Res genericPut(Req request, Class<Res> responseType, Class<Err> errorType, String path, String acceptMediaType, String postMediaType) throws RestConnectorException {
        return genericPut(request, responseType, errorType, path, acceptMediaType, postMediaType, null);
    }

    public <Res, Req, Err> Res genericPut(Req request, Class<Res> responseType, Class<Err> errorType, String path, String acceptMediaType, String postMediaType, Map<String, Object> templateParameters) throws RestConnectorException {
        return genericPutWithResponseHeader(request, responseType, errorType, path, acceptMediaType, postMediaType, null).getKey();
    }

    public <Res, Req, Err> Pair<Res, MultivaluedMap<String,Object>> genericPutWithResponseHeader(Req request, Class<Res> responseType, Class<Err> errorType, String path, String acceptMediaType, String postMediaType, Map<String, Object> templateParameters) throws RestConnectorException {
        log.trace("[RestConnector] Put cliente request [" + request + "]");

        Response putResponse = null;
        try {
            putResponse = buildRequest(environment.baseUrl + path, templateParameters)
                    .accept(acceptMediaType)
                    .put(Entity.entity(request, postMediaType));
            log.trace("[RestConnector] Put cliente response [" + putResponse.getStatus() + "/" + putResponse.getStatusInfo() + "]");

            checkSuccessfulOrException(putResponse, errorType);

            return Pair.of(processResponse(putResponse, responseType), putResponse.getHeaders());

        } catch (ProcessingException e) {
            throw new RestConnectorException("Error de comunicaciones", e);

        } finally {
            if (putResponse != null) {
                putResponse.close();
            }
        }
    }


    public <Res, Req> Res genericDelete(Req request, Class<Res> responseType, String path, Map<String, Object> templateParameters) throws RestConnectorException {
        return genericDelete(request, responseType, null, path, templateParameters);
    }

    public <Res, Req, Err> Res genericDelete(Req request, Class<Res> responseType, Class<Err> errorType, String path, Map<String, Object> templateParameters) throws RestConnectorException {
        log.trace("[RestConnector] Delete cliente request [" + request + "]");

        Response deleteResponse = null;
        try {
            deleteResponse = buildRequest(environment.baseUrl + path, templateParameters)
                    .accept(MediaType.APPLICATION_JSON)
                    .delete();
            log.trace("[RestConnector] Delete cliente response [" + deleteResponse.getStatus() + "/" + deleteResponse.getStatusInfo() + "]");

            checkSuccessfulOrException(deleteResponse, errorType);

            Res entityResponse = deleteResponse.readEntity(responseType);
            log.trace("[RestConnector] Entity response [" + entityResponse + "]");

            return entityResponse;

        } catch (ProcessingException e) {
            throw new RestConnectorException("Error de comunicaciones", e);

        } finally {
            if (deleteResponse != null) {
                deleteResponse.close();
            }
        }

    }


    /**
     * Comprueba que venga una respuesta successful (200 o 2NN), y sino tira una excepcion. Si vino informado un errorType (el body de la respuesta erronea tiene un json),
     * este se sube a la excepcion.
     *
     * @param response
     * @param errorType
     * @param <Err>
     * @throws RestConnectorException
     */
    private <Err> void checkSuccessfulOrException(Response response, Class<Err> errorType) throws RestConnectorException {
        if (response.getStatusInfo().getFamily() != Response.Status.Family.SUCCESSFUL) {
            Object entityResponseError = null;
            if (errorType != null) {
                entityResponseError = response.readEntity(errorType);
            }
            throw new RestConnectorException("La consulta no devolvio OK. Status: [" + response.getStatus() + "/" + response.getStatusInfo() + "]", response.getStatus(), response.getStatusInfo(), entityResponseError);
        }
    }

    /**
     * Devuelve la response con la entity correcta considerando el caso en que pueda venir vacia
     *
     * @param response
     * @param responseType
     * @param <Res>
     * @return
     */
    private <Res> Res processResponse(Response response, Class<Res> responseType) throws RestConnectorException{
        Res entityResponse;
        //Caso especial para cuando se invoca con Void y no se espera respuesta (solo que responda OK)
        if (response.getEntity() == null || responseType == Void.class) {
            entityResponse = null;
            log.trace("[RestConnector] Entity response [vacio]");
        } else {
            try {
                //NEcesito esto porque readEntity consume el stream, y no puedo
                //volver a llamarlo en caso de tener que manejarlo distinto en el manejo de excepcion
                response.bufferEntity();
                entityResponse = response.readEntity(responseType);
                log.trace("[RestConnector] Entity response [" + entityResponse + "]");
            } catch ( Exception e) {
                //Pordria haber tenido un rpoblema de tupos al intentar leer la responde como un objeto
                //Traro de leerlo de otra forma, pero igual es un error
                String stringResponse = response.readEntity(String.class);
                throw new RestConnectorException("La consulta no tiene tipo["+responseType.getCanonicalName()+"]. Resultado: [" + stringResponse+ "]", e);
            }
        }
        return entityResponse;
    }

    public Invocation.Builder buildRequest(String url, Map<String, Object> templateParameters) throws RestConnectorException {

        ClientBuilder builder = ClientBuilder.newBuilder();

        if (keyStore != null) {
            builder.keyStore(keyStore, environment.keyStorePassword);
        }

        if (trustStore != null) {
            builder.trustStore(trustStore);
            builder.hostnameVerifier((String hostName, SSLSession session) -> true);
        }

        if (securityManager != null && securityManager.getDisableHTTPSErrors()) {
            /*
            // No esa funcionando. Es posible que haya que tomar otro SSLContext, ej SSL, o TLS, o TLSv1.2
            // Igual, da errores
            SSLContext sslContext = null;
            TrustManager[] trustAllCerts = new X509TrustManager[]{new X509TrustManager() {
                @Override
                public java.security.cert.X509Certificate[] getAcceptedIssuers() {
                    return null;
                }
                @Override
                public void checkClientTrusted(X509Certificate[] certs, String authType) {
                }
                @Override
                public void checkServerTrusted(X509Certificate[] certs, String authType) {
                }
            }};
            try {
                sslContext = SSLContext.getInstance("TLSv1.2"); //Podria ser SSL, o TLSv1.2
                sslContext.init(null, trustAllCerts, null);
            } catch (NoSuchAlgorithmException | KeyManagementException e) {

            }
            builder.sslContext(sslContext);
            */
            // Disable PKIX path validation errors when running tests using SSL
            builder.hostnameVerifier((String hostName, SSLSession session) -> true);
        }

        WebTarget resource = builder.build().target(url);
        resource.register(clientLoggingFilter);

        if (templateParameters != null) {
            resource = resource.resolveTemplates(templateParameters);
        }



        Invocation.Builder request = resource.request();

        if (securityManager != null) {
            request = securityManager.addHeaders(request);
        }




        return request;
    }

    private KeyStore loadStore(String storePath, String storePassword) throws Exception {
        KeyStore store = KeyStore.getInstance("JKS");
        store.load(new FileInputStream(storePath), storePassword.toCharArray());
        return store;
    }

}
