package ar.com.sdd.commons.rest.log;

import ar.com.sdd.commons.rest.util.RestConnectorUtil;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.ws.rs.container.ContainerRequestContext;
import javax.ws.rs.container.ContainerRequestFilter;
import javax.ws.rs.container.ContainerResponseContext;
import javax.ws.rs.container.ContainerResponseFilter;
import javax.ws.rs.ext.Provider;
import java.io.*;
import java.util.Date;

@ServerFileLogged
@Provider
public class ServerFileLogginFilter implements ContainerRequestFilter, ContainerResponseFilter {

    private static final Logger log = LogManager.getLogger(ServerFileLogginFilter.class);
    private static QueuedLogWriter queuedLogWriterInstance;

    /**
     * Request filter
     *
     * @param requestContext
     * @throws IOException
     */
    @Override
    public void filter(ContainerRequestContext requestContext) throws IOException {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        IOUtils.copy(requestContext.getEntityStream(), baos);
        String timestamp36 = RestConnectorUtil.getCacheSafeValue();
        requestContext.setProperty("timestamp36", timestamp36);

        String method = requestContext.getMethod();
        String uri = requestContext.getUriInfo().getRequestUri() != null ? requestContext.getUriInfo().getRequestUri().toString() : "";
        String headers = RestConnectorUtil.getHeadersStringString(requestContext.getHeaders());
        String date = RestConnectorUtil.getFormattedDate(new Date());
        String respuesta = "[" + date + "][" + timestamp36 + "]:[RestConnector] serreq - Method [" + method + "], Headers [" + headers + "], URI [" + uri + "]";

        File logFile = RestConnectorUtil.getLogFile(date, timestamp36, "ser", "req", RestConnectorUtil.createTag(method, uri));
        log.info(respuesta + ", logFile [" + logFile + "]");
        byte[] bytes = baos.toByteArray();

        queuedLogWriterInstance.add(new QueueLogHolderByteArray(bytes, logFile.getAbsolutePath(), respuesta, timestamp36));

        requestContext.setEntityStream(new ByteArrayInputStream(bytes));
    }

    /**
     * Response filter
     *
     * @param requestContext
     * @param responseContext
     * @throws IOException
     */
    @Override
    public void filter(ContainerRequestContext requestContext, ContainerResponseContext responseContext) throws IOException {
        String timestamp36 = (String) requestContext.getProperty("timestamp36");
        if (StringUtils.isEmpty(timestamp36)) {
            timestamp36 = RestConnectorUtil.getCacheSafeValue();
            log.warn("No se pudo recuperar el timestamp36, se computa uno nuevo [" + timestamp36 + "]");
        }
        String method = requestContext.getMethod();
        String uri = requestContext.getUriInfo().getRequestUri() != null ? requestContext.getUriInfo().getRequestUri().toString() : "";
        String headers = RestConnectorUtil.getHeadersStringString(requestContext.getHeaders());
        String date = RestConnectorUtil.getFormattedDate(new Date());
        String respuesta = "[" + date + "][" + timestamp36 + "]:[RestConnector] serres - Method [" + method + "], Status [" + responseContext.getStatus() + "], Headers [" + headers + "], URI [" + uri + "]";
        String bearer = (String) requestContext.getProperty("bearer");
        if (!StringUtils.isEmpty(bearer)) {
            respuesta += ", Bearer [" + bearer + "]";
        }
        Object entity = responseContext.getEntity();
        if (entity != null) {
            File logFile = RestConnectorUtil.getLogFile(date, timestamp36, "ser", "res", RestConnectorUtil.createTag(method, uri));
            log.info(respuesta + ", logFile [" + logFile + "]");

            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            try (PrintWriter baosWriter = new PrintWriter(new ByteArrayOutputStream())) {
                baosWriter.write(respuesta + "\n");
                new ObjectMapper().writerWithDefaultPrettyPrinter().writeValue(baosWriter, entity);
            } catch (Exception e) {
                log.error("[" + timestamp36 + "] No se pudo escribir el baos", e);
            }
            byte[] bytes = baos.toByteArray();
            queuedLogWriterInstance.add(new QueueLogHolderByteArray(bytes, logFile.getAbsolutePath(), respuesta, timestamp36));

        } else {
            log.info(respuesta + "(vacio)");
        }
    }

    public static void setQueuedLogWriterInstance(QueuedLogWriter queuedLogWriterInstance) {
        ServerFileLogginFilter.queuedLogWriterInstance = queuedLogWriterInstance;
    }
}
