package ar.com.sdd.commons.rest.util;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.time.DateFormatUtils;

import javax.ws.rs.HttpMethod;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.core.Response;
import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Date;
import java.util.List;

public class RestConnectorUtil {

    public static final int MAX_ENTITY_SIZE = 1024 * 8;
    public static final int MAX_FILENAME_NO_EXT_SIZE = 255 - 4; //255 menos los 4 de la extension (ej: .log)

    /**
     * El metodo retorna la current-time en milisegundos (en base 36).
     * @return
     */
    public static String getCacheSafeValue() {
        long ts = System.currentTimeMillis();
        return Long.toString(ts, Character.MAX_RADIX);
    }

        public static File getLogFile(String formattedDate, String timestamp36, String serverClientTag, String requestResponseTag, String tag) throws IOException {
            //El formatedDate es yyyyMMdd-HHmmSS
            String subdirTag = formattedDate.substring(0,11);
            String logFilenameNoExt = "restconnector_" + serverClientTag + "_" + formattedDate + "_" + requestResponseTag + "_" + timestamp36 + ((tag != null) ? "_" + tag : "");
            if (logFilenameNoExt.length() > MAX_FILENAME_NO_EXT_SIZE) {
                logFilenameNoExt = logFilenameNoExt.substring(0, MAX_FILENAME_NO_EXT_SIZE);
            }
        return new File(FilenameUtils.concat(getLogDirectory(subdirTag), logFilenameNoExt + ".log"));
    }

    private static String getLogDirectory() throws IOException {
        return getLogDirectory(null);
    }
    private static String getLogDirectory(String subdir) throws IOException {
        String jbossLogDir = System.getProperty("jboss.server.log.dir");
        if (jbossLogDir==null) jbossLogDir=".";
        Path path;
        if (subdir!=null) {
            path = Paths.get(jbossLogDir, "io", subdir);
        } else {
            path = Paths.get(jbossLogDir, "io");
        }
        File logDirectory = new File(path.toUri());
        if (!logDirectory.exists()) {
            boolean result = logDirectory.mkdirs(); //result es true si creo el archivo y false si no, no implica que no haya podido crearlo
            if (!result) {
                //Hay un caso raro donde el result es false porque no lo creo...porque ya estaba
                //Por las dudas intento de nuevo antes de fallar
                File logDirectoryAgain = new File(path.toUri());
                if (!logDirectoryAgain.exists()) {
                    throw new IOException("No se pudo crear el directorio de log [" + logDirectory + "]");
                }
            }
        }
        return logDirectory.getAbsolutePath();
    }
    public static String createTag(String method, String uri) {
        //Reemplazo caracteres raros en la uri
        uri = uri.replace("?", "_")
                .replace("&", "_")
                .replace("//", "")
                .replace("https", "")
                .replace("http", "")
                .replace(":", "")
                .replace("=", "_")
                .replace("/", "_")
                .replace(".", "_")
                .replace(",", "_")
                .replace("__", "_");
        if (method != null) {
            switch (method.toUpperCase()) {
                case HttpMethod.GET: method = "GET"; break;
                case HttpMethod.POST: method = "POS"; break;
                case HttpMethod.PUT: method = "PUT"; break;
                case HttpMethod.DELETE: method = "DEL"; break;
                case HttpMethod.HEAD: method = "HEA"; break;
                case HttpMethod.OPTIONS: method = "OPT"; break;
                case HttpMethod.PATCH: method = "PAT"; break;
                default: method = "UNK";
            }
        } else {
            method = "   ";
        }
        return method + "_" + uri;
    }

    /*
        Solucion rapida para MultivaluedMap<String, Object> y MultivaluedMap<String, String>
     */

    public static String getHeadersStringObject(MultivaluedMap<String, Object> headers) {
        if (headers == null) {
            return "";
        }
        StringBuilder result = new StringBuilder();
        String sep = "";
        for (var headerKey : headers.keySet()) {
            List<Object> values = headers.get(headerKey);
            sep = getHeaderValues(result, sep, headerKey, values);
        }
        return result.toString();
    }

    public static String getHeadersStringString(MultivaluedMap<String, String> headers) {
        if (headers == null) {
            return "";
        }
        StringBuilder result = new StringBuilder();
        String sep = "";
        for (var headerKey : headers.keySet()) {
            List<String> values = headers.get(headerKey);
            sep = getHeaderValues(result, sep, headerKey, new ArrayList<>(values));
        }
        return result.toString();
    }

    private static String getHeaderValues(StringBuilder result, String sep, String headerKey, List<Object> values) {
        result.append(sep).append(headerKey).append(":");
        if (values != null && values.size() > 0) {
            if (values.size() > 1) {
                result.append("[");
            }
            String valueSep = "";
            for (var value : values) {
                result.append(valueSep).append(value);
                valueSep = ";";
            }
            if (values.size() > 1) {
                result.append("]");
            }
        }
        sep = "|";
        return sep;
    }


    /**
     * Dados un user y pass, los convierte a Base64, los une y devuelve el string "Basic xxxxxx" donde 'xxxxxx' es el Base64 de user y pass
     * Ej; para user: USRBANCOS y pass: tester, devuelve el string "Basic VVNSQkFOQ09TOnRlc3Rlcg=="
     *
     * @param user
     * @param pass
     * @return
     */
    public static String getBasicAuthHeader(String user, String pass) {
        if (user == null) user = "";
        if (pass == null) pass = "";
        String authString = user + ":" + pass;
        String authStringEnc = Base64.getEncoder().encodeToString(authString.getBytes());
        return "Basic " + authStringEnc;
    }

    public static String getFormattedDate(Date date) {
        return DateFormatUtils.format(date, "yyyyMMdd-HHmmss-SSS");
    }

    /**
     * Copia del Response.Status para ir agregando los Status que no tiene jaxrs
     */
    public enum ResponseStatus implements Response.StatusType {
        /*
         * Utilizado por:
         * HOLCIM/GetNet API en el refund de pagos con tarjetas a traves del Checkout (iFrame)
         */
        UNPROCESSABLE_ENTITY(422, "Unprocessable Entity");

        private final int code;
        private final String reason;
        private final Response.Status.Family family;

        ResponseStatus(int statusCode, String reasonPhrase) {
            this.code = statusCode;
            this.reason = reasonPhrase;
            this.family = Response.Status.Family.familyOf(statusCode);
        }

        @Override
        public final int getStatusCode() {
            return this.code;
        }

        @Override
        public Response.Status.Family getFamily() {
            return family;
        }

        /**
         * @deprecated Since 1.1. Use {@link #getReasonPhrase()} instead to get the reason phrase. Future releases will return the name of the enum constant instead
         *             of the reason phrase (see {@link java.lang.Enum#toString()}).
         */
        @Override
        public final String toString() {
            return this.reason;
        }

        @Override
        public final String getReasonPhrase() {
            return toString();
        }

        public static ResponseStatus fromStatusCode(final int statusCode) {
            for (ResponseStatus s : ResponseStatus.values()) {
                if (s.code == statusCode) {
                    return s;
                }
            }
            return null;
        }
    }
}
