package ar.com.sdd.commons.rest.core;

import ar.com.sdd.commons.rest.log.ClientFileLoggingFilter;
import ar.com.sdd.commons.rest.log.ClientLoggerLoggingFilter;
import ar.com.sdd.commons.rest.log.ClientLoggingFilter;
import ar.com.sdd.commons.rest.util.SSLSocketFactoryGenerator;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.net.ssl.*;
import javax.ws.rs.HttpMethod;
import javax.ws.rs.ProcessingException;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.Entity;
import javax.ws.rs.client.Invocation;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.core.Response;
import java.io.*;
import java.nio.charset.StandardCharsets;
import java.security.*;
import java.security.cert.X509Certificate;
import java.util.HashMap;
import java.util.Map;

/*
    Para debugear conexions, agregar esto al entorno (o a la runConfiguraicon en VM Options
            -Djavax.net.debug=all

    Genera MUCHO log de la conexion SL

 */


@SuppressWarnings("unused")
public class RestConnector {

    private static final Logger log = LogManager.getLogger(RestConnector.class);

    //Por defecto, intento esto
    private static final int DEFAULT_RETRY = 1;
    private static final int DEFAULT_RETRY_MILLIS = 10;

    /**
     * Todos los datos para la conexion
     */
    private final RestConnectorEnvironment environment;

    /**
     * El proveedor de seguridad que inyecta los headers necesarios
     */
    private final RestSecurityManager securityManager;

    /**
     * KeyStore de certificados privados (nuestros)
     */
    private KeyStore keyStore;

    /**
     * KeyStore de certificados publicos (ellos)
     */
    private KeyStore trustStore;

    /**
     * Cuando los certificados del trustStore vienen dentro del keyStore, activar esta opcion
     */
    private boolean useKeyStoreWithEmbeddedTrustStore;

    /**
     * Filtro del lado cliente
     */
    private final ClientLoggingFilter clientLoggingFilter;


    public RestConnector(RestConnectorEnvironment environment, RestSecurityManager securityManager) {
        this(environment, securityManager, true);
    }

    public RestConnector(RestConnectorEnvironment environment, RestSecurityManager securityManager, boolean logClientToFile) {
        this(environment, securityManager, true, false);
    }

    public RestConnector(RestConnectorEnvironment environment, RestSecurityManager securityManager, boolean logClientToFile, boolean useKeyStoreWithEmbeddedTrustStore) {
        this.environment = environment;
        this.securityManager = securityManager;

        //Inicializo los KeyStores (de haber)
        if (!StringUtils.isEmpty(environment.keyStorePath)) {
            try {
                keyStore = loadStore(environment.keyStorePath, environment.keyStorePassword);
            } catch (Exception e) {
                log.error("No se pudo cargar el keystore de [{}] con password [{}]", environment.keyStorePath, environment.keyStorePassword, e);
            }
        }

        if (!StringUtils.isEmpty(environment.trustStorePath)) {
            try {
                trustStore = loadStore(environment.trustStorePath, environment.trustStorePassword);
            } catch (Exception e) {
                log.error("No se pudo cargar el trustStore de [{}] con password [{}]", environment.trustStorePath, environment.trustStorePassword, e);
            }
        }

        if (useKeyStoreWithEmbeddedTrustStore) {
            log.debug("Usando el keyStore [{}] como trustStore", environment.keyStorePath);
            trustStore = keyStore;
        }

        //Inicializo los filtros
        if (logClientToFile) {
            clientLoggingFilter = new ClientFileLoggingFilter();
        } else {
            clientLoggingFilter = new ClientLoggerLoggingFilter();
        }
    }

    // -----------------------------------------------------------------------------------
    // -----------------------------------------------------------------------------------
    // --- Implementeaciones para GET
    // -----------------------------------------------------------------------------------
    // -----------------------------------------------------------------------------------
    //Hay muchas variantes, probablemente la mayoria innecesarias
    //
    // Manejando JSON como contentType response
    //   get                    -> response
    //   get                    -> response, con error type
    //   get (key,value)        -> response
    //   get (key,value)        -> response, con error type
    //   get (map{key,value})   -> response, con error type
    //   get (map{key,value})   -> response, con error type + response headers
    //   get (map{key,value})   -> response, con error type + response headers con retry automatico si falla comunicaciones
    //
    // Manejando STRING como contentType response
    //   get (map{key,value})   -> response, con error type
    //
    // Adicionalmente, todas, si falla la autorizacion hacen 1 reintento mas permitiendo la actualizacion de tokens
    //
    public <Res, Req> Res genericGet(Req requestBody, Class<Res> responseType, String path) throws RestConnectorException {
        return internalGenericRequest( HttpMethod.GET,requestBody, responseType, null, path,  null,DEFAULT_RETRY,DEFAULT_RETRY_MILLIS,MediaType.APPLICATION_JSON ,MediaType.APPLICATION_JSON).getKey();
    }
    public <Res, Req, Err> Res genericGet(Req requestBody, Class<Res> responseType, Class<Err> errorType, String path) throws RestConnectorException {
        return internalGenericRequest(HttpMethod.GET,requestBody,  responseType,  errorType, path, null,  DEFAULT_RETRY,DEFAULT_RETRY_MILLIS,MediaType.APPLICATION_JSON ,MediaType.APPLICATION_JSON).getKey();
    }
    public <Res, Req> Res genericGet(Req requestBody, Class<Res> responseType, String path, String templateKey, String templateValue) throws RestConnectorException {
        Map<String, Object> templateParameters = new HashMap<>();
        templateParameters.put(templateKey, templateValue);
        return internalGenericRequest(HttpMethod.GET,requestBody,  responseType,  null, path, templateParameters,  DEFAULT_RETRY,DEFAULT_RETRY_MILLIS,MediaType.APPLICATION_JSON ,MediaType.APPLICATION_JSON).getKey();
    }
    public <Res, Req, Err> Res genericGet(Req requestBody, Class<Res> responseType, Class<Err> errorType, String path, String templateKey, String templateValue) throws RestConnectorException {
        Map<String, Object> templateParameters = new HashMap<>();
        templateParameters.put(templateKey, templateValue);
        return internalGenericRequest(HttpMethod.GET,requestBody,  responseType,  errorType, path, templateParameters,  DEFAULT_RETRY,DEFAULT_RETRY_MILLIS,MediaType.APPLICATION_JSON ,MediaType.APPLICATION_JSON).getKey();
    }
    public  <Res, Req, Err> Res genericGet(Req requestBody, Class<Res> responseType, Class<Err> errorType, String path, Map<String, Object> templateParameters) throws RestConnectorException {
        return internalGenericRequest(HttpMethod.GET,requestBody, responseType, errorType, path, templateParameters, DEFAULT_RETRY,DEFAULT_RETRY_MILLIS, MediaType.APPLICATION_JSON,MediaType.APPLICATION_JSON).getKey();
    }
    public <Res, Req, Err> Pair<Res, MultivaluedMap<String,Object>> genericGetWithResponseHeader(Req requestBody, Class<Res> responseType, Class<Err> errorType, String path, Map<String, Object> templateParameters) throws RestConnectorException {
        return internalGenericRequest(HttpMethod.GET,requestBody, responseType, errorType, path, templateParameters, DEFAULT_RETRY,DEFAULT_RETRY_MILLIS, MediaType.APPLICATION_JSON,MediaType.APPLICATION_JSON);
    }
    public <Res, Req, Err> Pair<Res, MultivaluedMap<String,Object>> genericGetWithResponseHeader(Req requestBody, Class<Res> responseType, Class<Err> errorType, String path, Map<String, Object> templateParameters, int retryCount,long retryWaitMillis) throws RestConnectorException {
        return internalGenericRequest(HttpMethod.GET,requestBody,  responseType,  errorType, path, templateParameters,  retryCount,retryWaitMillis,MediaType.APPLICATION_JSON ,MediaType.APPLICATION_JSON);
    }
    public <Req, Err> String genericGetTextResponse(Req requestBody, Class<Err> errorType, String path, Map<String, Object> templateParameters) throws RestConnectorException {
        return internalGenericRequest(HttpMethod.GET,requestBody,  String.class,  errorType, path, templateParameters,  DEFAULT_RETRY,DEFAULT_RETRY_MILLIS,MediaType.TEXT_PLAIN,MediaType.APPLICATION_JSON ).getKey();
    }



    // -----------------------------------------------------------------------------------
    // -----------------------------------------------------------------------------------
    // --- Implementeaciones para POST
    // -----------------------------------------------------------------------------------
    // -----------------------------------------------------------------------------------
    //Hay muchas variantes, probablemente la mayoria innecesarias
    //
    // Manejando JSON como contentType response
    //   post                    -> response
    //   post (map{key,value})   -> response
    //   post                    -> response  +response headers
    //   post (map{key,value})   -> response  + response headers
    //
    // Manejando ANY como contentType response
    //   post                    -> response
    //   post (map{key,value})   -> response
    //   post                    -> response  +response headers
    //   post (map{key,value})   -> response  + response headers

    // Adicionalmente, todas, si falla la autorizacion hacen 1 reintento mas permitiendo la actualizacion de tokens
    //
    public <Res, Req> Res genericPost(Req requestBody, Class<Res> responseType, String path) throws RestConnectorException {
        //return genericPost(request, responseType, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON, null);
        return internalGenericRequest(HttpMethod.POST,requestBody, responseType, null, path, null,  DEFAULT_RETRY,DEFAULT_RETRY_MILLIS,MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON ).getKey();
    }
    public <Res, Req> Res genericPost(Req requestBody, Class<Res> responseType, String path, Map<String, Object> templateParameters) throws RestConnectorException {
        //return genericPost(request, responseType, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON, templateParameters);
        return internalGenericRequest(HttpMethod.POST,requestBody, responseType, null, path, templateParameters,  DEFAULT_RETRY,DEFAULT_RETRY_MILLIS,MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON ).getKey();
    }
    public <Res, Req> Pair<Res, MultivaluedMap<String,Object>> genericPostWithResponseHeader(Req requestBody, Class<Res> responseType, String path) throws RestConnectorException {
        //return genericPostWithResponseHeader(request, responseType, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON, null);
        return internalGenericRequest(HttpMethod.POST,requestBody, responseType, null, path, null,  DEFAULT_RETRY,DEFAULT_RETRY_MILLIS,MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON );
    }
    public <Res, Req> Pair<Res, MultivaluedMap<String,Object>> genericPostWithResponseHeader(Req requestBody, Class<Res> responseType, String path, Map<String, Object> templateParameters) throws RestConnectorException {
        return genericPostWithResponseHeader(requestBody, responseType, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON, templateParameters);
    }
    public <Res, Req> Res genericPost(Req requestBody, Class<Res> responseType, String path, String acceptMediaType, String postMediaType) throws RestConnectorException {
        //return genericPost(request, responseType,null, path, acceptMediaType, postMediaType, null);
        return internalGenericRequest(HttpMethod.POST,requestBody, responseType, null, path, null,  DEFAULT_RETRY,DEFAULT_RETRY_MILLIS,acceptMediaType, postMediaType ).getKey();
    }

    public <Res, Req> Res genericPost(Req requestBody, Class<Res> responseType, String path, String acceptMediaType, String postMediaType, Map<String, Object> templateParameters) throws RestConnectorException {
        //return genericPost(request, responseType,null, path, acceptMediaType, postMediaType, templateParameters);
        return internalGenericRequest(HttpMethod.POST,requestBody, responseType, null, path, templateParameters,  DEFAULT_RETRY,DEFAULT_RETRY_MILLIS,acceptMediaType, postMediaType ).getKey();
    }
    public <Res, Req> Pair<Res, MultivaluedMap<String,Object>> genericPostWithResponseHeader(Req requestBody, Class<Res> responseType, String path, String acceptMediaType, String postMediaType) throws RestConnectorException {
        //return genericPostWithResponseHeader(request, responseType, null, path, acceptMediaType, postMediaType, null);
        return internalGenericRequest(HttpMethod.POST,requestBody, responseType, null, path, null,  DEFAULT_RETRY,DEFAULT_RETRY_MILLIS,acceptMediaType, postMediaType );
    }
    public <Res, Req> Pair<Res, MultivaluedMap<String,Object>> genericPostWithResponseHeader(Req requestBody, Class<Res> responseType, String path, String acceptMediaType, String postMediaType, Map<String, Object> templateParameters) throws RestConnectorException {
        //return genericPostWithResponseHeader(request, responseType, null, path, acceptMediaType, postMediaType, templateParameters);
        return internalGenericRequest(HttpMethod.POST,requestBody, responseType, null, path, templateParameters,  DEFAULT_RETRY,DEFAULT_RETRY_MILLIS,acceptMediaType, postMediaType );
    }
    public <Res, Req, Err> Res genericPost(Req requestBody, Class<Res> responseType, Class<Err> errorType, String path) throws RestConnectorException {
        //return genericPost(request, responseType, errorType, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON, null);
        return internalGenericRequest(HttpMethod.POST,requestBody, responseType, errorType, path, null,  DEFAULT_RETRY,DEFAULT_RETRY_MILLIS,MediaType.APPLICATION_JSON,MediaType.APPLICATION_JSON ).getKey();
    }
    public <Res, Req, Err> Res genericPost(Req requestBody, Class<Res> responseType, Class<Err> errorType, String path, Map<String, Object> templateParameters) throws RestConnectorException {
        //return genericPost(request, responseType, errorType, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON, templateParameters);
        return internalGenericRequest(HttpMethod.POST,requestBody, responseType, errorType, path, templateParameters,  DEFAULT_RETRY,DEFAULT_RETRY_MILLIS,MediaType.APPLICATION_JSON,MediaType.APPLICATION_JSON ).getKey();
    }
    public <Res, Req, Err> Pair<Res, MultivaluedMap<String,Object>> genericPostWithResponseHeader(Req requestBody, Class<Res> responseType, Class<Err> errorType, String path) throws RestConnectorException {
        //return genericPostWithResponseHeader(request, responseType, errorType, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON, null);
        return internalGenericRequest(HttpMethod.POST,requestBody, responseType, errorType, path, null,  DEFAULT_RETRY,DEFAULT_RETRY_MILLIS,MediaType.APPLICATION_JSON,MediaType.APPLICATION_JSON );
            }
    public <Res, Req, Err> Pair<Res, MultivaluedMap<String,Object>> genericPostWithResponseHeader(Req requestBody, Class<Res> responseType, Class<Err> errorType, String path, Map<String, Object> templateParameters) throws RestConnectorException {
        //return genericPostWithResponseHeader(request, responseType, errorType, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON, templateParameters);
        return internalGenericRequest(HttpMethod.POST,requestBody, responseType, errorType, path, templateParameters,  DEFAULT_RETRY,DEFAULT_RETRY_MILLIS,MediaType.APPLICATION_JSON,MediaType.APPLICATION_JSON );
    }
    public <Res, Req, Err> Res genericPost(Req requestBody, Class<Res> responseType, Class<Err> errorType, String path, String acceptMediaType, String postMediaType) throws RestConnectorException {
        //return genericPostWithResponseHeader(request, responseType, errorType, path, acceptMediaType, postMediaType, null).getKey();
        return internalGenericRequest(HttpMethod.POST,requestBody, responseType, errorType, path, null,  DEFAULT_RETRY,DEFAULT_RETRY_MILLIS,acceptMediaType,postMediaType ).getKey();
    }
    public <Res, Req, Err> Res genericPost(Req requestBody, Class<Res> responseType, Class<Err> errorType, String path, String acceptMediaType, String postMediaType, Map<String, Object> templateParameters) throws RestConnectorException {
        //return genericPostWithResponseHeader(request, responseType, errorType, path, acceptMediaType, postMediaType, templateParameters).getKey();
        return internalGenericRequest(HttpMethod.POST,requestBody, responseType, errorType, path, templateParameters,  DEFAULT_RETRY,DEFAULT_RETRY_MILLIS,acceptMediaType,postMediaType ).getKey();

    }
    public <Res, Req, Err> Pair<Res, MultivaluedMap<String,Object>> genericPostWithResponseHeader(Req requestBody, Class<Res> responseType, Class<Err> errorType, String path, String acceptMediaType, String postMediaType) throws RestConnectorException {
        //return genericPostWithResponseHeader(request, responseType, errorType, path, acceptMediaType, postMediaType, null);
        return internalGenericRequest(HttpMethod.POST,requestBody, responseType, errorType, path, null,  DEFAULT_RETRY,DEFAULT_RETRY_MILLIS,acceptMediaType,postMediaType );

    }
    public <Res, Req, Err> Pair<Res, MultivaluedMap<String,Object>> genericPostWithResponseHeader(Req requestBody, Class<Res> responseType, Class<Err> errorType, String path, String acceptMediaType, String postMediaType, Map<String, Object> templateParameters) throws RestConnectorException {
        //return genericPostWithResponseHeader(request, responseType, errorType, path, acceptMediaType, postMediaType, templateParameters, 1, 0);
        return internalGenericRequest(HttpMethod.POST,requestBody, responseType, errorType, path, templateParameters,  DEFAULT_RETRY,DEFAULT_RETRY_MILLIS,acceptMediaType,postMediaType );

    }
    public <Res, Req, Err> Pair<Res, MultivaluedMap<String,Object>> genericPostWithResponseHeader(Req requestBody, Class<Res> responseType, Class<Err> errorType, String path, String acceptMediaType, String postMediaType, int retryCount, long retryWaitMillis) throws RestConnectorException {
        //return genericPostWithResponseHeader(request, responseType, errorType, path, acceptMediaType, postMediaType, null, retryCount, retryWaitMillis);
        return internalGenericRequest(HttpMethod.POST,requestBody, responseType, errorType, path, null,  retryCount,retryWaitMillis,acceptMediaType,postMediaType );
    }


    // -----------------------------------------------------------------------------------
    // -----------------------------------------------------------------------------------
    // --- Implementeaciones para PUT
    // -----------------------------------------------------------------------------------
    // -----------------------------------------------------------------------------------
    //Hay muchas variantes, probablemente la mayoria innecesarias
    //
    // Manejando JSON como contentType response
    //   put                    -> response
    //   put                    -> response, con error type
    //
    // Manejando ANY como contentType response
    //   put                    -> response, con error type
    //   put (map{key,value})   -> response, con error type
    //   put (map{key,value})   -> response, con error type + response headers
    //
    // Adicionalmente, todas, si falla la autorizacion hacen 1 reintento mas permitiendo la actualizacion de tokens
    //
    public <Res, Req> Res genericPut(Req requestBody, Class<Res> responseType, String path) throws RestConnectorException {
        //return genericPut(request, responseType, null, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON);
        return internalGenericRequest(HttpMethod.PUT,requestBody, responseType, null, path, null,  DEFAULT_RETRY,DEFAULT_RETRY_MILLIS,MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON ).getKey();
    }
    public <Res, Req, Err> Res genericPut(Req requestBody, Class<Res> responseType, Class<Err> errorType, String path) throws RestConnectorException {
        //return genericPut(request, responseType, errorType, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON);
        return internalGenericRequest(HttpMethod.PUT,requestBody, responseType, errorType, path, null,  DEFAULT_RETRY,DEFAULT_RETRY_MILLIS,MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON ).getKey();
    }

    public <Res, Req, Err> Res genericPut(Req requestBody, Class<Res> responseType, Class<Err> errorType, String path, String acceptMediaType, String postMediaType) throws RestConnectorException {
        //return genericPut(request, responseType, errorType, path, acceptMediaType, postMediaType, null);
        return internalGenericRequest(HttpMethod.PUT,requestBody, responseType, errorType, path, null,  DEFAULT_RETRY,DEFAULT_RETRY_MILLIS,acceptMediaType,postMediaType ).getKey();
    }
    public <Res, Req, Err> Res genericPut(Req requestBody, Class<Res> responseType, Class<Err> errorType, String path, String acceptMediaType, String postMediaType, Map<String, Object> templateParameters) throws RestConnectorException {
        //return genericPutWithResponseHeader(request, responseType, errorType, path, acceptMediaType, postMediaType, null).getKey();
        return internalGenericRequest(HttpMethod.PUT,requestBody, responseType, errorType, path, templateParameters,  DEFAULT_RETRY,DEFAULT_RETRY_MILLIS,acceptMediaType,postMediaType ).getKey();
    }
    public <Res, Req, Err> Pair<Res, MultivaluedMap<String,Object>> genericPutWithResponseHeader(Req requestBody, Class<Res> responseType, Class<Err> errorType, String path, String acceptMediaType, String postMediaType, Map<String, Object> templateParameters) throws RestConnectorException {
        //return genericPutWithResponseHeader(request, responseType, errorType, path, acceptMediaType, postMediaType, templateParameters,1,0);
        return internalGenericRequest(HttpMethod.PUT,requestBody, responseType, errorType, path, templateParameters,  DEFAULT_RETRY,DEFAULT_RETRY_MILLIS,acceptMediaType,postMediaType );
    }


    // -----------------------------------------------------------------------------------
    // -----------------------------------------------------------------------------------
    // --- Implementeaciones para DEL
    // -----------------------------------------------------------------------------------
    // -----------------------------------------------------------------------------------
    //Hay muchas varianres, probablemente la mayoria innecesarias
    //
    // Manejando JSON como contentType response
    //   del                    -> response
    //   del                    -> response, con error type
    //
    // Adicionalmente, todas, si falla la autorizacion hacen 1 reintento mas permitiendo la actualizacion de tokens
    //
    public <Res, Req> Res genericDelete(Req requestBody, Class<Res> responseType, String path, Map<String, Object> templateParameters) throws RestConnectorException {
        //return genericDelete(request, responseType, null, path, templateParameters);
        return internalGenericRequest(HttpMethod.DELETE,requestBody, responseType, null, path, templateParameters,  DEFAULT_RETRY,DEFAULT_RETRY_MILLIS,null,null ).getKey();
    }

    public <Res, Req, Err> Res genericDelete(Req requestBody, Class<Res> responseType, Class<Err> errorType, String path, Map<String, Object> templateParameters) throws RestConnectorException {
        //return genericDelete(request, responseType, errorType, path, templateParameters, 1,0);
        return internalGenericRequest(HttpMethod.DELETE,requestBody, responseType, errorType, path, templateParameters,  DEFAULT_RETRY,DEFAULT_RETRY_MILLIS,null,null ).getKey();
    }



    // -----------------------------------------------------------------------------------
    // -----------------------------------------------------------------------------------
    // --- Implementeacion generia del request, para todos los juegos de parametros
    // -----------------------------------------------------------------------------------
    // -----------------------------------------------------------------------------------

    //Wrapper para poder hacer el reintento si falla la autorizacion
    private  <Res, Req, Err> Pair<Res, MultivaluedMap<String,Object>> internalGenericRequest(String method,Req requestBody, Class<Res> responseType, Class<Err> errorType, String path, Map<String, Object> templateParameters, int retryCount,long retryWaitMillis, String acceptMediaType, String postMediaType) throws RestConnectorException {
            try {
            return internalGenericRequestrNotryOnAuth(method,requestBody, responseType, errorType, path, templateParameters, retryCount, retryWaitMillis,acceptMediaType,postMediaType);
        } catch (RestConnectorException e) {
            if (e.getStatus() == Response.Status.UNAUTHORIZED.getStatusCode() && securityManager.retryOnUnauthorized()) {
                log.error("Retry por autorizacional hacer GET para request [{}] con path [{}]. Se reconsulta", requestBody, path);
                return internalGenericRequestrNotryOnAuth(method,requestBody, responseType, errorType, path, templateParameters, retryCount, retryWaitMillis,acceptMediaType,postMediaType);
            } else {
                throw e;
        }
    }
    }
    // Esta es la que realmente hace las cosas a nivel comunicaciones
    private  <Res, Req, Err> Pair<Res, MultivaluedMap<String,Object>> internalGenericRequestrNotryOnAuth(String method,Req requestBody, Class<Res> responseType, Class<Err> errorType, String path, Map<String, Object> templateParameters, int retryCount, long retryWaitMillis, String acceptMediaType, String postMediaType) throws RestConnectorException {
        log.trace("[RestConnector] {} client request [{}}]",method,requestBody);
        RestConnectorException lastException = null;

        Invocation.Builder buildRequest = buildRequest(environment.baseUrl + path, templateParameters);

        Entity entity = null;
        //Esto solo tiene sentido si los metodos son POST o PUT, que es cuando deberia tenet un objeto request!=null;
        if (requestBody != null) {
            entity = Entity.entity(requestBody, postMediaType);
        }

        // Para soportar el modelo de firmas, necesito contar con el string del request a firmar, pero solo a veces
        // No me convence mucho esta forma de hacerlo, pero si funciona, lo mejoraremos
        if (securityManager != null) {
            if (securityManager instanceof RestSignSecurityManager) {
                if (requestBody != null) {
                    String stringRequest;
                    if (postMediaType.equals(MediaType.APPLICATION_JSON)) {
                        ObjectMapper objectMapper = new ObjectMapper();
                        try {
                            stringRequest = objectMapper.writeValueAsString(requestBody);
                        } catch (JsonProcessingException e) {
                            stringRequest = requestBody.toString();
                        }
                    } else {
                        stringRequest = requestBody.toString();
                    }
                    buildRequest = ((RestSignSecurityManager) securityManager).addHeaders(buildRequest, stringRequest);
                    entity = Entity.entity(stringRequest, postMediaType);
                } else {
                    buildRequest = securityManager.addHeaders(buildRequest);
                }
            } else {
                buildRequest = securityManager.addHeaders(buildRequest);
            }
        }
        Invocation.Builder buildResponse = buildRequest.accept(acceptMediaType);

        while (retryCount >=0) {
            Response response = null;
            try {
                switch (method) {
                    case HttpMethod.GET :response = buildResponse.get();
                                break;
                    case HttpMethod.POST:response = buildResponse.post(entity);
                                break;
                    case HttpMethod.PUT :response = buildResponse.put(entity);
                                break;
                    case HttpMethod.DELETE: response = buildResponse.delete();
                                break;
                    default: throw  new RestConnectorException("Invalid method:"+method);
    }
                log.trace("[RestConnector] {} client response [{}/{}]",method,response.getStatus(),response.getStatusInfo());
                checkSuccessfulOrException(response, errorType);
                return Pair.of(processResponse(response, responseType), response.getHeaders());

            } catch (ProcessingException e) {
                lastException = new RestConnectorException("Error de comunicaciones", e);

            } finally {
                if (response!=null) response.close();
            }

            retryCount--;
            try {
                Thread.sleep(retryWaitMillis);
            } catch (InterruptedException e) {
                lastException = new RestConnectorException("Error en wait", e);
            }
        }

        throw  lastException;


    }


    /**
     * Comprueba que venga una respuesta successful (200 o 2NN), y sino tira una excepcion. Si vino informado un errorType (el body de la respuesta erronea tiene un json),
     * este se sube a la excepcion.
     *
     * @param response
     * @param errorType
     * @param <Err>
     * @throws RestConnectorException
     */
    private <Err> void checkSuccessfulOrException(Response response, Class<Err> errorType) throws RestConnectorException {
        if (response.getStatusInfo().getFamily() != Response.Status.Family.SUCCESSFUL) {
            Object entityResponseError = null;

            if (errorType != null) {
                try {
                    //NEcesito esto porque readEntity consume el stream, y no puedo
                    //volver a llamarlo en caso de tener que manejarlo distinto en el manejo de excepcion
                    response.bufferEntity();
                    entityResponseError  = response.readEntity(errorType);
                    log.trace("[RestConnector] Entity responseError [{}]", entityResponseError);
                } catch ( Exception e) {
                    //Pordria haber tenido un problema de tupos al intentar leer la responde como un objeto
                    //Traro de leerlo de otra forma, pero igual es un error
                    entityResponseError = response.readEntity(String.class);
                    log.debug("La consulta no tiene tipo[{}]", errorType.getCanonicalName());
                }
            } else {
                entityResponseError = response.readEntity(String.class);
            }
            throw new RestConnectorException("La consulta no devolvio OK. Status: [" + response.getStatus() + "/" + response.getStatusInfo() + "]", response.getStatus(), response.getStatusInfo(), entityResponseError);
        }
    }

    /**
     * Devuelve la response con la entity correcta considerando el caso en que pueda venir vacia
     *
     * @param response
     * @param responseType
     * @param <Res>
     * @return
     */
    private <Res> Res processResponse(Response response, Class<Res> responseType) throws RestConnectorException{
        Res entityResponse;
        //Caso especial para cuando se invoca con Void y no se espera respuesta (solo que responda OK)
        if (response.getEntity() == null || responseType == Void.class) {
            entityResponse = null;
            log.trace("[RestConnector] Entity response [vacio]");
        } else {
            try {
                //NEcesito esto porque readEntity consume el stream, y no puedo
                //volver a llamarlo en caso de tener que manejarlo distinto en el manejo de excepcion
                response.bufferEntity();
                entityResponse = response.readEntity(responseType);
                log.trace("[RestConnector] Entity response [{}]", entityResponse);
            } catch (ProcessingException e) {
                // Si ocurre un problema de codificación, intentamos forzar la lectura en UTF-8
                log.warn("[RestConnector] Problema al leer la respuesta como [{}]. Intentando con UTF-8.", responseType.getCanonicalName());

                try {
                    // Leer el contenido bruto del InputStream y forzar UTF-8
                    InputStream inputStream = response.readEntity(InputStream.class);
                    String stringResponse = convertStreamToUTF8String(inputStream);

                    log.trace("[RestConnector] Raw response as UTF-8 String [{}]", stringResponse);

                    // Convertir el string a la clase esperada
                    entityResponse = new ObjectMapper().readValue(stringResponse, responseType);
                    log.trace("[RestConnector] Entity response despues de forzar UTF-8 [{}]", entityResponse);

                } catch (Exception utf8Exception) {
                    // Si falla nuevamente, lanzar una excepción final
                    String stringResponse = response.readEntity(String.class);
                    throw new RestConnectorException("Error al procesar respuesta como [" + responseType.getCanonicalName() + "]. Resultado: [" + stringResponse + "]", utf8Exception);
                }
            } catch ( Exception e) {
                //Pordria haber tenido un problema de tupos al intentar leer la responde como un objeto
                //Traro de leerlo de otra forma, pero igual es un error
                String stringResponse = response.readEntity(String.class);
                throw new RestConnectorException("La consulta no tiene tipo["+responseType.getCanonicalName()+"]. Resultado: [" + stringResponse+ "]", e);
            }
        }
        return entityResponse;
    }

    private String convertStreamToUTF8String(InputStream inputStream) throws IOException {
        // Leer el InputStream con la codificación forzada a UTF-8
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8))) {
            StringBuilder stringBuilder = new StringBuilder();
            String line;
            while ((line = reader.readLine()) != null) {
                stringBuilder.append(line).append("\n");
            }
            return stringBuilder.toString();
        }
    }

    public Invocation.Builder buildRequest(String url, Map<String, Object> templateParameters) throws RestConnectorException {

        ClientBuilder builder = ClientBuilder.newBuilder();

        if (keyStore != null) {
            builder.keyStore(keyStore, environment.keyStorePassword);
        }

        if (trustStore != null) {
            builder.trustStore(trustStore);
            builder.hostnameVerifier((String hostName, SSLSession session) -> true);
        }

        if (securityManager != null && securityManager.getDisableHTTPSErrors()) {
            /*
            // No esa funcionando. Es posible que haya que tomar otro SSLContext, ej SSL, o TLS, o TLSv1.2
            // Igual, da errores
            */
            SSLContext sslContext = null;
            TrustManager[] trustAllCerts = new X509TrustManager[]{new X509TrustManager() {
                @Override
                public java.security.cert.X509Certificate[] getAcceptedIssuers() {
                    return null;
                }
                @Override
                public void checkClientTrusted(X509Certificate[] certs, String authType) {
                }
                @Override
                public void checkServerTrusted(X509Certificate[] certs, String authType) {
                }
            }};
            try {
                sslContext = SSLContext.getInstance(SSLSocketFactoryGenerator.defaultProtocol) ; //Podria ser SSL, o TLSv1.2 o TLSv1.3

                String alg= KeyManagerFactory.getDefaultAlgorithm();
                KeyManagerFactory kmFact=KeyManagerFactory.getInstance(alg);
                //Init the key manager factory with the loaded key store
                kmFact.init(keyStore, environment.keyStorePassword!=null?environment.keyStorePassword.toCharArray():null);
                KeyManager[] kms=kmFact.getKeyManagers();
                sslContext.init(kms, trustAllCerts,  new java.security.SecureRandom());
            } catch (NoSuchAlgorithmException | KeyManagementException e) {
                //Ignore
            } catch (UnrecoverableKeyException | KeyStoreException e) {
                throw new RuntimeException(e);
            }
            builder.sslContext(sslContext);
            // Disable PKIX path validation errors when running tests using SSL
            builder.hostnameVerifier((String hostName, SSLSession session) -> true);
        }

        WebTarget resource = builder.build().target(url);
        resource.register(clientLoggingFilter);

        if (templateParameters != null) {
            resource = resource.resolveTemplates(templateParameters);
        }

        return resource.request();
    }

    private KeyStore loadStore(String storePath, String storePassword) throws Exception {
        KeyStore store = KeyStore.getInstance("JKS");
        store.load(new FileInputStream(storePath), storePassword!=null?storePassword.toCharArray():null);
        return store;
    }

}
