package ar.com.sdd.commons.rest.log;

import ar.com.sdd.commons.rest.util.RestConnectorUtil;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.annotation.Priority;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.client.ClientRequestContext;
import javax.ws.rs.client.ClientResponseContext;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.core.Response;
import javax.ws.rs.ext.WriterInterceptorContext;
import java.io.*;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.Date;

@Priority(Integer.MIN_VALUE)
public class ClientFileLoggingFilter implements ClientLoggingFilter {

    private static final Logger log = LogManager.getLogger(ClientFileLoggingFilter.class);
    private static QueuedLogWriter queuedLogWriterInstance;

    private static QueuedLogWriter getQueuedLogWriterInstance() {
        if (queuedLogWriterInstance==null) {
            QueuedLogWriterFileImpl.initialize(); //Medio raro, pero funciona
        }
        return queuedLogWriterInstance;
    }

    private static final String ENTITY_STREAM_PROPERTY = "ClientLoggingFilter.entityStream";

    /**
     * Request filter
     *
     * Parto el reporte en 2:
     *    - request:   timestamp46, Method, Header, URL
     *    - response:  timestamp46, Status, logFile
     *
     * @param requestContext
     */
    @Override
    public void filter(ClientRequestContext requestContext) {
        String timestamp36 = RestConnectorUtil.getCacheSafeValue();
        requestContext.setProperty("timestamp36", timestamp36);
        String method = requestContext.getMethod() != null ? requestContext.getMethod() : "N/A";
        String uri = requestContext.getUri() != null ? requestContext.getUri().toString() : "N/A";
        String headers = RestConnectorUtil.getHeadersStringObject(requestContext.getHeaders());
        String date = RestConnectorUtil.getFormattedDate(new Date());
        StringBuilder respuesta = new StringBuilder();
        respuesta.append("[").append(timestamp36).append("]:[RestConnector.clireq][").append(method).append("],H=[").append(headers).append("],URI=[").append(uri).append("]");
        if (requestContext.hasEntity()) {
            OutputStream stream = new LoggingStream(requestContext.getEntityStream(), date, timestamp36, respuesta.toString(), method, uri);
            requestContext.setEntityStream(stream);
            requestContext.setProperty(ENTITY_STREAM_PROPERTY, stream);
        } else {
            log.info(respuesta + "(vacio)");
        }
    }

    @Override
    public void aroundWriteTo(WriterInterceptorContext context) throws IOException, WebApplicationException {
        final LoggingStream stream = (LoggingStream) context.getProperty(ENTITY_STREAM_PROPERTY);
        context.proceed();
        if (stream != null) {
            stream.logStream();
        }
    }

    private static class LoggingStream extends FilterOutputStream {

        private final ByteArrayOutputStream baos = new ByteArrayOutputStream();
        private final String timestamp36;
        private final String date;
        private final String respuesta;
        private final String method;
        private final String uri;

        LoggingStream(OutputStream out, String date, String timestamp36, String respuesta, String method, String uri) {
            super(out);
            this.date = date;
            this.timestamp36 = timestamp36;
            this.respuesta = respuesta;
            this.method = method;
            this.uri = uri;
        }

        void logStream() throws IOException {
            File logFile = RestConnectorUtil.getLogFile(date, timestamp36, "cli", "req", RestConnectorUtil.createTag(method, uri));
            log.info(respuesta + ", logFile [" + logFile + "]");

            // write entity to the builder
            byte[] bytes = baos.toByteArray();

            getQueuedLogWriterInstance().add(new QueueLogHolderByteArray(bytes, logFile.getAbsolutePath(), respuesta, timestamp36));

        }

        @Override
        public void write(final int i) throws IOException {
            baos.write(i);
            out.write(i);
        }
    }


    /**
     * Response filter
     *
     * @param requestContext
     * @param responseContext
     * @throws IOException
     */
    @Override
    public void filter(ClientRequestContext requestContext, ClientResponseContext responseContext) throws IOException {
        String timestamp36 = (String) requestContext.getProperty("timestamp36");
        if (StringUtils.isEmpty(timestamp36)) {
            timestamp36 = RestConnectorUtil.getCacheSafeValue();
            log.warn("No se pudo recuperar el timestamp36, se computa uno nuevo [" + timestamp36 + "]");
        }
        Response.StatusType getStatusInfo = responseContext.getStatusInfo();
        String statusCode = getStatusInfo != null ? String.valueOf(getStatusInfo.getStatusCode()) : "N/A";
        String method = requestContext.getMethod();

        URI uriObject= requestContext.getUri();
        String uri =  uriObject!= null ? uriObject.toString() : "";

        MultivaluedMap<String, Object> headersMap = requestContext.getHeaders();
        String headers = RestConnectorUtil.getHeadersStringObject(headersMap);
        String date = RestConnectorUtil.getFormattedDate(new Date());
        //Hago mas compacto el log:
        //String respuesta = "[" + date + "][" + timestamp36 + "]:[RestConnector] clires - Method [" + method + "], Status [" + statusCode + "], Headers [" + headers + "], URI [" + uri + "]";
        //String respuesta = "[" + timestamp36 + "]:[RestConnector.clires][" + method + "],S=[" + statusCode + "], H=[" + headers + "], URI=[" + uri + "]";
        String respuesta = "[" + timestamp36 + "]:[RestConnector.clires],S=[" + statusCode + "]";

        //Analizo si la respuesta viene chunked
        boolean chunked = false;
        if (headersMap!=null) {
               String transferEncoding =  responseContext.getHeaderString("Transfer-Encoding");
               chunked = transferEncoding!=null && transferEncoding.equalsIgnoreCase("chunked");
        }

        File logFile = RestConnectorUtil.getLogFile(date, timestamp36, "cli", "res", RestConnectorUtil.createTag(method, uri));
        log.info(respuesta + ",logFile [" + logFile + "]");
        String entityStreamResult = null;

        ByteArrayOutputStream baos = new ByteArrayOutputStream();

        if (responseContext.hasEntity() && !chunked) {
            entityStreamResult = IOUtils.toString(responseContext.getEntityStream(), StandardCharsets.UTF_8);
            baos.write(entityStreamResult.getBytes(StandardCharsets.UTF_8));

        } else {
            if (chunked) {
                baos.write("(chunked)".getBytes(StandardCharsets.UTF_8));
            } else {
                baos.write("(vacio)".getBytes(StandardCharsets.UTF_8));
            }
        }

        baos.close();
        byte[] bytes = baos.toByteArray();
        getQueuedLogWriterInstance().add(new QueueLogHolderByteArray(bytes, logFile.getAbsolutePath(), respuesta, timestamp36));

        if (responseContext.hasEntity() && !chunked && entityStreamResult != null) {
            responseContext.setEntityStream(IOUtils.toInputStream(entityStreamResult, StandardCharsets.UTF_8));
        }
    }

    public static void  setQueuedLogWriterInstance(QueuedLogWriter queuedLogWriterInstance) {
        ClientFileLoggingFilter.queuedLogWriterInstance = queuedLogWriterInstance;
    }
}
