package ar.com.sdd.commons.rest.util;

import com.fasterxml.jackson.core.JsonGenerationException;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.ser.std.StdSerializer;
import com.fasterxml.jackson.datatype.joda.JodaModule;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.joda.time.format.ISODateTimeFormat;

import javax.ws.rs.ext.ContextResolver;
import javax.ws.rs.ext.Provider;
import java.io.IOException;

@Provider
public class RestMapperBuilder implements ContextResolver<ObjectMapper> {
    private final ObjectMapper objectMapper;

    public RestMapperBuilder() {
        objectMapper = getObjectMapper();
    }

    public ObjectMapper getContext(Class<?> type) {
        return objectMapper;
    }

    public static ObjectMapper getObjectMapper() {
        return new ObjectMapper()
                .setDateFormat(new RFC3339DateFormat())
                .configure(DeserializationFeature.READ_UNKNOWN_ENUM_VALUES_AS_NULL, true)
                .registerModule(new JodaModule() {
                    {
                        addSerializer(DateTime.class, new StdSerializer<DateTime>(DateTime.class) {
                            @Override
                            public void serialize(DateTime value, JsonGenerator jgen, SerializerProvider provider) throws IOException, JsonGenerationException {
                                jgen.writeString(ISODateTimeFormat.dateTimeNoMillis().print(value));
                            }
                        });
                        addSerializer(LocalDate.class, new StdSerializer<LocalDate>(LocalDate.class) {
                            @Override
                            public void serialize(LocalDate value, JsonGenerator jgen, SerializerProvider provider) throws IOException, JsonGenerationException {
                                jgen.writeString(ISODateTimeFormat.date().print(value));
                            }
                        });

                    }
                })
        ;
    }
}