package ar.com.sdd.commons.rest.log;

import org.apache.log4j.Logger;

import javax.annotation.Priority;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.client.ClientRequestContext;
import javax.ws.rs.client.ClientRequestFilter;
import javax.ws.rs.client.ClientResponseContext;
import javax.ws.rs.client.ClientResponseFilter;
import javax.ws.rs.core.Response;
import javax.ws.rs.ext.WriterInterceptor;
import javax.ws.rs.ext.WriterInterceptorContext;
import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;

@Priority(Integer.MIN_VALUE)
public class ClientLoggingFilter implements ClientRequestFilter, ClientResponseFilter, WriterInterceptor {

    private static final Logger log = Logger.getLogger(ClientLoggingFilter.class);

    private static final String ENTITY_STREAM_PROPERTY = "ClientLoggingFilter.entityStream";
    private static final Charset DEFAULT_CHARSET = StandardCharsets.UTF_8;
    private final int maxEntitySize = 1024 * 8;

    private void log(StringBuilder sb) {
        log.info(sb.toString());
    }


    //REQUEST

    @Override
    public void filter(ClientRequestContext requestContext) {
        if (requestContext.hasEntity()) {
            final OutputStream stream = new LoggingStream(requestContext.getEntityStream(), requestContext.getUri(), requestContext.getMethod());
            requestContext.setEntityStream(stream);
            requestContext.setProperty(ENTITY_STREAM_PROPERTY, stream);

        } else {
            final StringBuilder sb = new StringBuilder();
            String uri = requestContext.getUri() != null ? requestContext.getUri().toString() : "N/A";
            String method = requestContext.getMethod() != null ? requestContext.getMethod() : "N/A";
            sb.append("[RestConnector] Client Request - Method [").append(method).append("], URI [").append(uri).append("]:NO_ENTITY");
            log(sb);
        }
    }

    @Override
    public void aroundWriteTo(WriterInterceptorContext context) throws IOException, WebApplicationException {
        final LoggingStream stream = (LoggingStream) context.getProperty(ENTITY_STREAM_PROPERTY);
        context.proceed();
        if (stream != null) {
            log(stream.getStringBuilder(DEFAULT_CHARSET));
        }
    }

    private class LoggingStream extends FilterOutputStream {

        private final StringBuilder sb = new StringBuilder();
        private final ByteArrayOutputStream baos = new ByteArrayOutputStream();
        private URI uri;
        private String method;

        LoggingStream(OutputStream out, URI uri, String method) {
            super(out);
            this.uri = uri;
            this.method = method;
        }

        StringBuilder getStringBuilder(Charset charset) {
            // write entity to the builder
            final byte[] entity = baos.toByteArray();

            sb.append("[RestConnector] Client Request - Method [").append(method).append("], URI [").append(uri).append("]:");

            sb.append(new String(entity, 0, entity.length, charset));
            if (entity.length > maxEntitySize) {
                sb.append("...more...");
            }

            return sb;
        }

        @Override
        public void write(final int i) throws IOException {
            if (baos.size() <= maxEntitySize) {
                baos.write(i);
            }
            out.write(i);
        }
    }





    //RESPONSE

    @Override
    public void filter(ClientRequestContext requestContext, ClientResponseContext responseContext) throws IOException {
        final StringBuilder sb = new StringBuilder();

        //Analizo si la respuesta viene chunked
        boolean chunked = responseContext.getHeaders() != null
                && responseContext.getHeaderString("Transfer-Encoding") != null
                && responseContext.getHeaderString("Transfer-Encoding").equalsIgnoreCase("chunked");

        if (responseContext.hasEntity()) {
            responseContext.setEntityStream(logInboundEntity(sb, responseContext.getStatusInfo(), responseContext.getEntityStream(), DEFAULT_CHARSET, chunked));
        } else {
            String statusCode = responseContext.getStatusInfo() != null ? String.valueOf(responseContext.getStatusInfo().getStatusCode()) : "N/A";
            sb.append("[RestConnector] Client Response - Status [").append(statusCode).append("]:NO_ENTITY");
        }
        log(sb);

    }

    private InputStream logInboundEntity(final StringBuilder sb, Response.StatusType statusInfo, InputStream stream, final Charset charset, boolean chunked) throws IOException {
        if (!stream.markSupported()) {
            stream = new BufferedInputStream(stream);
        }
        stream.mark(maxEntitySize + 1);
        final byte[] entity = new byte[maxEntitySize + 1];
        final int entitySize = stream.read(entity);

        sb.append("[RestConnector] Client Response - Status [").append(statusInfo.getStatusCode()).append("]:");
        if (!chunked) {
            sb.append(new String(entity, 0, Math.min(entitySize, maxEntitySize), charset));
            if (entitySize > maxEntitySize) {
                sb.append("...more...");
            }
        } else {
            //FIXME encontrar la manera de loguear los bloques chunked
            sb.append("[chunked]");
        }
        stream.reset();
        return stream;
    }


}
