package ar.com.sdd.commons.util;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.Serializable;
import java.util.MissingResourceException;
import java.util.PropertyResourceBundle;
import java.util.ResourceBundle;


/**
 * Excepcion base de la cual deben descender todas las otras excepciones.
 * @author esteban@sdd
 * @date 11/03/2002
 */
public abstract class AbstractException extends Exception implements NestedException, Serializable {

    private static final Logger log = LogManager.getLogger(AbstractException.class);
    private static final long serialVersionUID = 5546497497565700547L;
    static ResourceBundle resource = null;

    /**
     * No habria que utilizarlo mas
     */
    @Deprecated
    static public void initialize(String errorFileName) {
        if (errorFileName != null) {
            try {
                resource = new PropertyResourceBundle(new FileInputStream(errorFileName));
            } catch (IOException e) {
                log.error("Ha ocurrido un error", e);
            }
        }
    }

    /**
     * Recursos de la exepcion (Codigos, mensajes y etc.)
     * Es recomendable que no se carge un ResourceBundle c/ vez que
     * se llama a este metodo, sino que devuelva un objeto previamente
     * instanciado.
     * @returns ResourceBundle
     */
    static public ResourceBundle getResourceBundle() {
        return resource;
    }

//---- Variables y metodos de instancia -------------------------------------------------//

    private Object[] params = null;
    private Throwable detail = null;


    public AbstractException() {
    }

    public AbstractException(String msg) {
        this(msg, null, null);
    }

    public AbstractException(String msg, Object[] params) {
        this(msg, params, null);
    }

    public AbstractException(Throwable detail) {
        this(detail.getMessage(), null, detail);
    }

    public AbstractException(String msg, Throwable detail) {
        this(msg, null, detail);
    }

    public AbstractException(String msg, Object[] params, Throwable detail) {
        super(msg);
        this.detail = detail;
        this.params = params;
        initCause(detail);

        if (getLoggeable()) {
            dump();
        }
    }

    /**
     * La excepcion debe generar un log?
     * @returns boolean True si la excepcion se logea
     */
    protected boolean getLoggeable() {
        return false;
    }

    /**
     * La excepcion es fatal (panic)
     * @returns boolean True si la excepcion es fatal
     */
    protected boolean getFatal() {
        return false;
    }

    /**
     * Detalle de la excepcion
     * @returns Throwable
     */
    public Throwable getDetail() {
        return detail;
    }

    /**
     * Detalle base de la excepcion (si tiene multiples detalles)
     * @returns Throwable
     */
    public Throwable getRootDetail() {
        Throwable ret = getDetail();
        Throwable aux = ret;
        while (aux != null) {
            if (aux instanceof NestedException) {
                aux = ((NestedException) aux).getDetail();
            }

            if (aux != null) {
                ret = aux;
            }
        }

        return ret;
    }

    /**
     * Devuelve el mensaje de error. Si hay un codigo establecido para el en
     * el archivo de resource, obtiene el String asociado.
     * NOTA: si no se desea la "traduccion", usar getMessage().
     * no se deberia utilizar mas este metodo
     */
    @Deprecated
    public String getLocalizedMessage() {
        String ret = null;
        String key = super.getMessage();

        ResourceBundle resource = AbstractException.getResourceBundle();
        if (resource != null) {
            try {

                if (key != null) {
                    String msg = resource.getString(key);
                    ret = (msg != null ? msg : key);
                } else {
                    ret = key;
                }

            } catch (MissingResourceException ex) {
                // Si no encuentra el resource, directamente devuelve la 'key'
                ret = key;
                log.error("Ha ocurrido un error", ex);

            }

        } else {
            ret = super.getMessage();
        }

        return ret;
    }

    private void dump() {
        Logger log = LogManager.getLogger(this.getClass());
        if (!getFatal()) {
            log.error(getMessage(), this);
            if (detail != null) {
                log.error(detail.getMessage(), this);
            }
        } else {
            log.fatal(getMessage(), this);
            if (detail != null) {
                log.fatal(detail.getMessage(), this);
            }
        }
    }
}