package ar.com.sdd.commons.util;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.io.BufferedInputStream;
import java.io.Serializable;
import java.util.Properties;

/**
 * Carga una serie de propiedades de un archivo de configuraci?n. Este archivo
 * debe tener un formato compatible con el que usa <code>java.util.Properties</code>, es
 * decir, pares <i>nombre_propiedad</i>=<i>valor_propiedad</i>
 * <p>
 * El constructor recibe el nombre del archivo del cual debera tomar las propiedades.
 * Este nombre de archivo no puede ser modificado posteriormente. Cada objeto
 * <code>Config</code> se asume relacionado con un ?nico archivo de configuraci?n.
 * Sin embargo, si el contenido del archivo fuera modificado, el m?todo <code>refresh</code>
 * permite volver a leer el contenido del mismo.
 * <p>
 * Las propiedades contenidas en el archivo de configuraci?n pueden ser
 * recuperadas usando los m?todos <code>getProperty</code> y <code>getProperties</code>,
 * que permite recuperar las propiedades de a una y todas juntas, respectivamente.
 *
 * @author esteban@sdd
 * @version     14/08/2001
 *
 * @author alejandra@sdd
 * @version     17/1/2002
 *
 * @see Properties
 */
public class Config implements Serializable {

    Logger log = LogManager.getLogger(Config.class);
    private static final long serialVersionUID = -4273437898806760790L;
	private Properties props;

    /** Cuenta la cantidad de refreshes para poder ver si se tiene una instancia acualizada del Config o no.
     * <p>
     * <u>Atenci?n</u>: Esta variable no mide cuantas veces cambio el archivo de configuraci?n sino simplemente
     *      cuantas veces se invoc? al m?todo <code>refresh</code>. Podr?a pasar que no se produjeran cambios
     *      en las propiedades al hacer un refresh....
     */
    private int generation = 0;

    /**
     * Construye una nueva instancia, asociandole el nombre del archivo
     * del cual deber? obtener las propiedades.
     * <p>
     * Nota: las propiedades se levantan bajo demanda, la primera vez que se
     * necesita una de ellas.
     */
    public Config() {
        this.props = null;
    }

    /**
     * Devuelve el valor de la propiedad identificada por la clave <b>key</b>.
     * Si esta no tiene un valor definido, se devuelve el valor default <b>def</b>.
     */
    public String getProperty(String key, String def) {
        if (props == null) {
            initialize();
        }
        return props.getProperty(key, def);
    }

    /**
     * Devuelve el valor de la propiedad identificada por la clave <b>key</b>.
     * Si ?sta no tiene un valor definido, se devuelve <b>null</b>.
     */
    public String getProperty(String key) {
        if (props == null) {
            initialize();
        }
        return props.getProperty(key);
    }

    /**
     * Devuelve un objeto <code>java.util.Properties</code> conteniendo todas
     * las propiedades levantadas del archivo de configuraci?n.
     * Nota: este m?todo podr?a devolver <b>null</b> ?nicamente en los casos
     * en los que no se haya podido levantar las propiedades del archivo de
     * configuraci?n.
     */
    public Properties getProperties() {
        if (props == null) {
            initialize();
        }
        return props;
    }


    /**
     * Vuelve a cargar el archivo de configuracion. Incrementa en 1 la <i>generaci?n</i> de la instancia.
     */
    public void refresh() {
        initialize();
        generation++;
    }

    /**
     * Da una estimaci?n <b>heur?stica</b> para determinar si se tiene una instancia vieja desactualizada, midiendo
     * la <i>generaci?n</i> de la instancia en cantidad de veces que se invoc? al m?todo <code>refresh</code>. 
     * <p>
     * <u>Atenci?n</u>: Este m?todo no dice cuantas veces cambio el archivo de configuraci?n sino simplemente
     *      cuantas veces se invoc? al m?todo <code>refresh</code>. Podr?a pasar que no se produjeran cambios
     *      en las propiedades al hacer un refresh....
     * 
     * @return int <i>Generaci?n</i> de la instancia, medida en cantidad de veces que se hizo refresh sobre la misma
     */
    public int getGeneration() {
        return generation;
    }

    

//--- Metodos privados ------------------------------------------------------------------------
    /**
     * Carga las propiedades desde el archivo <b>configFileName</b>. Si la operaci?n
     * tiene ?xito, se pisan las propiedades anteriores con las recientemente
     * cargadas. Si la operaci?n fracasa, las propiedades anterioes permanecen
     * intactas.
     */
    private void initialize() {
        try {
            Properties p = new Properties();
            //EN WF10 se carga de esta manera
            p.load(new BufferedInputStream(Config.class.getResourceAsStream("/ebfserver.conf")));
            
            //pisa las propiedades que hubiera seteadas con anterioridad
            this.props = p;
        } catch (Exception e) {
            log.error("No se pudo encontrar el archivo de configuracion [ebfserver.conf]", e);
        }
    }


}
