package ar.com.sdd.commons.util;

import java.lang.reflect.Method;
import java.util.Properties;

/**
 * Instancia objetos genericos usando reflection. Los objetos deben cumplir una interfaz determinada
 * para poder ser construidos usando este factory.
 * 
 * @see #create
 * @author aalfonso
 * @version Aug 13, 2002-2:54:59 PM
 */
public class GenericFactory {

    /**
     * Instancia un objeto usando el constructor sin parametros y luego lo inicializa invocando el 
     * metodo <code>init</code> al cual le pasa los argumentos recibidos como parametro, encapsulados
     * dentro del objeto <code>Properties</code>.
     * <p>
     * Requisitos que debe cumplir un objeto para poder ser creado de esta forma:<br>
     * 
     * <ul>
     *      <li> Debe tener un constructor <b>default</b> sin parametros de la forma: <br>
     *              <pre>public MiClase()</pre> </li>
     *      <li> Debe tener un metodo <code>init</code> publico con el siguiente encabezado: <br>
     *              <pre>public void init(java.util.Properties args) </pre> </li>
     * </ul>
     * 
     * Si el objeto no necesitara parametros de inicializacion, debe incluir igual el metodo <code>init</code>
     * con la interfaz descripta arriba. Sin embargo, para esos objetos se puede invocar a este metodo ( <code>create</code>)
     * con <b>args</b> = <b>null</b>.
     * 
     * @param className Nombre completo de la clase que se quiere instanciar. Debe cumplir los requisitos mencionados
     *                  arriba. No puede ser <b>null</b>.
     * @param args Propiedades de inicializacion para el metodo <code>init</code> del objeto a ser creado.
     *                  Puede ser <b>null</b> si no se necesitan parametros de inicializacion.
     * @return Object Instancia de la clase <b>className</b>.
     * @throws InitializationException (runtime) si surgiera algun error en la inicializacion del objeto, por
     *              ejemplo, si la clase no cumpliera con los requisitos mencionados arriba
     */
    public static Object create(String className, Properties args)
        throws InitializationException {
        if (className == null) {
            throw new InitializationException("Nombre de la clase a instanciar nulo.");
        }

        try {
            Class objectClass = Class.forName(className);
            Object newObject = objectClass.newInstance();

            Method initMethod =
                objectClass.getMethod("init", new Class[] { Properties.class });
            initMethod.invoke(newObject, new Object[] { args });
            return newObject;

        } catch (ClassNotFoundException ex) {
            throw new InitializationException(
                "No se encuentra la clase [" + className + "]",
                ex);
        } catch (IllegalAccessException ex) {
            throw new InitializationException(
                "No se puede acceder a los metodos de la clase ["
                    + className
                    + "]",
                ex);
        } catch (InstantiationException ex) {
            throw new InitializationException(
                "No se puede instanciar la clase [" + className + "]",
                ex);
        } catch (NoSuchMethodException ex) {
            throw new InitializationException(
                "No se existe el metodo [init] en la clase [" + className + "]",
                ex);
        } catch (java.lang.reflect.InvocationTargetException ex) {
            throw new InitializationException(
                "Error al inicializar objeto de la clase [" + className + "]",
                ex.getTargetException());
        }

    }

}
