package ar.com.sdd.commons.util;

import ar.com.sdd.commons.util.mapper.DecodeMapper;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.Properties;

public class PropertiesUtil {

	private static final Logger log = LogManager.getLogger(PropertiesUtil.class);

	public static final String DEFAULT_SEPARATOR = "" + (char) 13 + (char) 10;

	Properties prop;
	
	public PropertiesUtil() {
		prop = new Properties();
	}
	
	public PropertiesUtil(String serializedProperties) {
		prop = new Properties();
		readProperties(serializedProperties);
	}

    //Obtiene una property (por default, string)
	public String getProperty(String name) {
		return prop.getProperty(name);
	}

	//Obtiene una property String
	public String getStringProperty(String name) {
		return getProperty(name);
	}

	//Obtiene una property boolean
	public boolean getBooleanProperty(String name) {
		String value = getProperty(name);
		return !StringUtil.isEmpty(value) && (value.equalsIgnoreCase("true") || value.equals("1") || value.equalsIgnoreCase("verdadero"));
	}

	//Obtiene una property boolean, conun default si no esta definida
	public boolean getBooleanProperty(String name, boolean defaultIfUndefined) {
		String value = getProperty(name);
		if (StringUtil.isEmptyNull(value)) return defaultIfUndefined;
		return value.equalsIgnoreCase("true") || value.equals("1") || value.equalsIgnoreCase("verdadero");
	}

	//EBF
    // Obtiene una property buscandola en los resources
	//public String getResourceProperty(long instanceId, String language, String keyStr) {
	//	return StringResources.getStringResource(instanceId, language, getProperty(keyStr));
	//}

	public void setProperty(String name, String value) {
		prop.setProperty(name, value);
		storeProperties(); //commitea
	}

	public void readProperties(String serializedProperties) {
		if (serializedProperties!=null && !serializedProperties.equals("")) {
			ByteArrayInputStream bais = new ByteArrayInputStream(serializedProperties.getBytes());
			try {
				prop.load(bais);
			} catch (IOException ex) {
				log.debug("EXCEP  ---- PropertiesUtil.getProperties: IOException");
				throw new ApplicationException(ex);
			}		
		}
	}

	public String storeProperties() {
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		try {
			prop.store(baos, "");
		} catch (Exception e) {
			log.error("Ocurrio un error al intentar convertir las properties en un string", e);
		}
		String result;
		result = baos.toString();
		try {
			baos.close();
		} catch (IOException e) {
			log.error ("Error al cerrar el stream", e);
		}
		return result;
	}

	public static String getProperty(String properties, String propertyKey) {
    	Collection<String> propertiesVec = StringUtil.parseString(properties, ",");
		for (String prop : propertiesVec) {
			ArrayList components = (ArrayList) StringUtil.parseString(prop, "=");
			String key = (String) components.get(0);
			if (key.equals(propertyKey)) {
				if (components.size() > 1) return (String) components.get(1);
			}
		}
    	return null;
	}

	public static Collection<String> getPropertiesName(String properties) {
		Collection<String> propertiesName = new ArrayList<>();
		Collection<String> propertiesVec = StringUtil.parseString(properties, ",");
		for (String prop : propertiesVec) {
			Collection<String> components = StringUtil.parseString(prop, "=");
			propertiesName.add((String) components.toArray()[0]);
		}
    	return propertiesName;
	}
	
	public boolean isEmpty() {
		return (prop == null) || prop.isEmpty(); 
	}
	
	public Collection<String> keys() {
		ArrayList<String> ret = new ArrayList<>();
		if (prop!=null) {
			for (Enumeration<Object> e = prop.keys();e.hasMoreElements();) {
				String value = (String) e.nextElement();
				ret.add(value);
			}
		}
		return ret;
	}

	public void merge(Properties propToMerge) {
		//Evito usar el putAll por: https://bugs.openjdk.java.net/browse/JDK-8043219
		for (Enumeration<Object> keys = propToMerge.keys(); keys.hasMoreElements(); ) {
			String key = (String)keys.nextElement();
			prop.setProperty(key, propToMerge.getProperty(key));
		}
	}

    /**
     * Para codificar los enters del CDATA por _ESCPENTER_
     * @param propertyValue
     * @return
     */
    public static String encodeEnters(String propertyValue) {
        if (StringUtil.isEmptyNull(propertyValue)) {
            return propertyValue;
        }
        return propertyValue.replaceAll( "\\n", DecodeMapper.ESCAPE_ENTER).replaceAll( "\\r", "");
    }

    /**
     * Para decodificar los enters del CDATA por los correctos \n
     * @param propertyValue
     * @return
     */
    public static String decodeEnters(String propertyValue) {
        if (StringUtil.isEmptyNull(propertyValue)) {
            return propertyValue;
        }
        return propertyValue.replaceAll(DecodeMapper.ESCAPE_ENTER, "\n");
    }

	public Properties getProps() {
		return prop;
	}
}