package ar.com.sdd.commons.util;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * Implementa un manager de caches muy sencillo
 * <br>
 * Vive en un singleton por lo que todas las instancias/usuarios pueden acceder a la misma informacion, asi que los datos a persistir
 * en estas caches tienen que ser cross-usuario (una alternativa es que en nombre de la cache tenga un hash del usuario)
 * <br>
 * Ejemplo:
 * 
	<pre>
    //Para una cache comun (SimpleCacheLevel.WEAK):
	SimpleCache cache = SimpleCacheManager.getInstance().getCache(SelectBankAjax.class.getName());
    //Para una cache que sobreviva el ReinicializarServer:
	SimpleCache cache = SimpleCacheManager.getInstance().getCache(SelectBankAjax.class.getName(), SimpleCacheManager.SimpleCacheLevel.HARD);
	String key = SimpleCacheManager.buildKey(type, billerIdByReasonType, billerIdTag, countryId, filter, sort, objectId);
	if (cache.containsKey(key)) {
		return (String)cache.get(key);
		
	} else {
		... cosas para calcular el resultado
		cache.put(key, result);
	}
	</pre>
 *
 */
public class SimpleCacheManager {

	private static SimpleCacheManager instance;
	private Map<String, SimpleCache> caches;
	private static final Logger log = LogManager.getLogger(SimpleCacheManager.class);

	public enum SimpleCacheLevel {
		WEAK(1),	//Se borrar con el Reinicializar Server. Es el default
		HARD(10),  	//No se borra con el Reinicializar Server, y hay que ir a a la pantalla del server
		NEVER(Integer.MAX_VALUE), //No se borra hasta reinicializar el WF
		;

		private int level;
		SimpleCacheLevel(int level) {
			this.level = level;
		}

		public int getLevel() {
			return level;
		}

		public boolean isCleareable(SimpleCacheLevel minimumLevel) {
			if (minimumLevel.level == NEVER.level) return false;
			else return level <= minimumLevel.level;
		}
	}
	
	private SimpleCacheManager() {
		caches = new HashMap<>();
	}
	
	public synchronized static SimpleCacheManager getInstance() {
		if (instance == null) {
			instance = new SimpleCacheManager();
		}
		return instance;
	}

	/**
	 * Devuelve una cache por su nombre. De no estar, la crea
	 *
	 * Crear una cache de level WEAK por default
	 *
	 * @param cacheName el nombre de la cache a obtener
	 * @return
	 */
	public SimpleCache getCache(String cacheName) {
		return getCache(cacheName, SimpleCacheLevel.WEAK);
	}

	/**
	 * Devuelve una cache por su nombre. De no estar, la crea
	 *
	 * @param cacheName el nombre de la cache a obtener
	 * @param cacheLevel El nivel de cache
	 * @return
	 */
	public SimpleCache getCache(String cacheName, SimpleCacheLevel cacheLevel) {
		if (caches.containsKey(cacheName)) {
			return caches.get(cacheName);
		} else {
			//Si no tengo el name, creo una nueva cache con ese name
			SimpleCache cache = new SimpleCache(cacheName, cacheLevel);
			caches.put(cacheName, cache);
			return cache;
		}
	}
	
	/**
	 * Invalida los datos de una cache, cuando su level es menor o igual al minimumLevel pasado
	 * @param cacheName el nombre de la cache
	 */
	public void invalidate(String cacheName, SimpleCacheLevel minimumLevel) {
		if (!caches.containsKey(cacheName)) {
			log.error("No se puede invalidar la cache [" + cacheName + "] porque no existe");
			return;
		}
		SimpleCache cache = caches.get(cacheName);
		if (cache.getCacheLevel().isCleareable(minimumLevel)) {
			cache.clear();
		}
	}

	/**
	 * Invalida todas las caches creadas, cuando su level es menor o igual al minimumLevel pasado
	 */
	public void invalidateAllCaches(SimpleCacheLevel minimumLevel) {

		Iterator<Map.Entry<String, SimpleCache>> cachesIterator = caches.entrySet().iterator();

		while (cachesIterator.hasNext()) {
			Map.Entry<String, SimpleCache> entry  = cachesIterator.next();
			invalidate(entry.getKey(), minimumLevel);

			//Remuevo la cache de todas las caches
			SimpleCache cache = entry.getValue();
			if (cache.getCacheLevel().isCleareable(minimumLevel)) {
				cachesIterator.remove();
			}
		}

	}
	

	/**
	 * Matodo de ayuda que arma una key separada por pipes
	 * 
	 * @param objects el conjunto de objects a usar como key. Si alguno es null, se pone el string "null"
	 * @return
	 */
	public static String buildKey(Object... objects) {
		StringBuilder key = new StringBuilder();
		boolean first = true;
		for (Object object : objects) {
			if (first) {
				first = false;
			} else {
				key.append("|");
			}
			key.append(object);
		}
		return key.toString();
	}

}
