package ar.com.sdd.commons.util;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.transaction.Status;
import javax.transaction.SystemException;
import javax.transaction.UserTransaction;

/**
 * Ayuda de pattern de transacciones en procesos iterativos.
 *
 * Esta clase provee soporte para procesos iterativos que procesan una determinada
 * cantidad de iteraciones por transaccion.
 *
 * El pattern de uso es:
 *
 * TxBatchProcessorHelper txBatch = new TxBatchProcessorHelper( batchSize );
 * try {
 *     while ( continueProcess ) {
 *	     	txBatch.beginBatchIteration()
 * 	    
 * 	    			...hace cosas a commitear en la tx, despues de batchSize veces
 * 	    
 *  	   	if (error) {
 *     			txBatch.setItemsOk(false)
 *     		}
 *     	
 *     		boolean necesitoReatachear = txBatch.endBatchIteration()
 *     		if (necesitoReatachear) {
 *     			PersistenceHelper.reAttachObject(em, objeto)
 *     		} 
 *     }
 * } finally {
 *  	txBatch.endBatchProcess()   
 * }
 *
 * @author dachcar
 *
 */
public class TxBatchProcessorHelper implements TxBatchProcessor {

	public static final long DEFAULT_BATCH_SIZE = 100;
	private UserTransaction tx;
	private static final Logger log = LogManager.getLogger(TxBatchProcessorHelper.class);
	private long batchSize;
	private long processedItems = 0;
	private boolean itemsOk = true;
	private boolean inTransaction = false;
	private boolean inIteration = false;
	private boolean handleExceptions = false;
	
	public static final String USER_TRANSACTION_JNDI = "java:/jboss/UserTransaction";


	public TxBatchProcessorHelper(){
		batchSize = DEFAULT_BATCH_SIZE;
	}

	public TxBatchProcessorHelper(long batchSize){
		this.batchSize = batchSize;
	}

	public long getBatchSize() {
		return batchSize;
	}

	public void setBatchSize(long batchSize) {
		this.batchSize = batchSize;
	}

	public boolean getItemsOk() {
		return itemsOk;
	}

	public void setItemsOk(boolean itemOk) {
		this.itemsOk = itemOk;
	}

	public boolean getHandleExceptions() {
		return handleExceptions;
	}

	public void setHandleExceptions(boolean handleExceptions) {
		this.handleExceptions = handleExceptions;
	}

	
	/*
	 * Un workaround para cuando no se si tengo que cerrar 
	 * Deberia usarse:
	 * 	
	 *  ....
	 *  endTxBatchIterationIfNeeded();
	 *  ...
	 *  endTxBatchIterationIfNeeded(); 
	 */
	
	public boolean endTxBatchIterationIfNeeded() throws TxBatchProcessorException{
		if (!inIteration) return false;
		endTxBatchIteration();
		return true;
	}

	public void beginTxBatchIteration() throws TxBatchProcessorException {
		if ( inIteration )
			reportTxBatchPatternViolation( "beginTxBatchIteration llamado sin llamar al endTxBatchIteration de la iteracion anterior" );
		try {
			if ( !inTransaction ){
				//log.info("begin: "+processedItems);
				//TransactionManager tm = (TransactionManager)getInitialContext().lookup("java:/TransactionManager");
				tx = (UserTransaction)getInitialContext().lookup(TxBatchProcessorHelper.USER_TRANSACTION_JNDI);
				int txs = tx.getStatus();
				tx.begin();
				inTransaction=true;
				itemsOk=true;
			}
			inIteration = true;
		}
		catch(Throwable ex){
			log.error("problemas al arrancar la Transaccion", ex);
			throw new TxBatchProcessorException(ex);
		}
	}

	public boolean endTxBatchIteration() throws TxBatchProcessorException{
		return endTxBatchIteration(false);
	}
	public boolean endTxBatchIteration(boolean forceCommitOnOk) throws TxBatchProcessorException{
		boolean necesitoReatachear = false;
		if ( !inIteration )
			reportTxBatchPatternViolation( "endTxBatchIteration llamado sin su correspondiente beginTxBatchIteration" );
		processedItems++; 	// Se hace si ok o no, para reportar correctamente la
							// iteracion inconclusa si hay error
		try {
			if (itemsOk) {
				if (forceCommitOnOk || shouldCommitTransaction()) {
					commitTransaction();
					necesitoReatachear = true;
				}
			} else {
				rollbackTransaction();
				necesitoReatachear = true;
			}
		} finally {
			inIteration = false;
		}
		
		return necesitoReatachear;
	}

	public void endTxBatchProcess() throws TxBatchProcessorException {
		if ( inIteration ) {
			reportTxBatchPatternViolation( "endTxBatchProcess llamado sin llamar al endTxBatchIteration de la iteracion anterior" );
			endTxBatchIteration();
		}
		// Aqui se llega solamente luego de endTxBatchIteration, con lo que solo hay
		// una transaccion activa si no hay error, en cuyo caso hay que forzar el commit
		if (inActiveTransaction())
			commitTransaction();
	}

	private void commitTransaction() throws TxBatchProcessorException{

		try{
			if (inActiveTransaction())
				tx.commit();
		}
		catch(Exception ex){
			log.error("Problemas al comitear la transaccion", ex);
			rollbackTransaction();
			if ( !handleExceptions )
				throw new TxBatchProcessorException("Problemas al realizar el commit de la transaccion", ex);
		}
		finally {
			processedItems = 0;
			inTransaction = false;
			inIteration = false;
			itemsOk = true; //arrancamos de nuevo un batch
		}
	}
	
	/**
	 * 
	 * ATENCION!!
	 * 
	 * SOLO DEBE USARSE ESTE METODO PARA CASOS EXCEPCIONALES
	 * DONDE SE REQUIERA HACER UN COMMIT INTERMEDIO
	 * 
	 * EN EL CICLO NORMAL DEBE USARSE 
	 * O BIEN endTxBatchIteration
	 * O BIEN endTxBatchProcess
	 *  
	 */
	public void exceptionalCommit() throws TxBatchProcessorException{
		log.debug("EJECUTANDO COMMIT EN FORMA EXCEPCIONAL");
		try {
			if (inActiveTransaction()) {
				tx.commit();
			}
		}
		catch(Exception ex){
			log.error("Problemas al comitear la transaccion", ex);
			rollbackTransaction();
			if ( !handleExceptions )
				throw new TxBatchProcessorException("Problemas al realizar el commit de la transaccion", ex);
		}		
	}

	private void rollbackTransaction() throws TxBatchProcessorException{
		try{
			if (inActiveTransaction())
				tx.rollback();
			log.error("Se hizo rollback de la transaccion, podrian no haberse procesado "+processedItems+" elementos. El batchSize es de: " + batchSize + " elementos");
		}
		catch(Exception ex){
			log.error("Problemas al hacer rollback de la transaccion, podrian no haberse procesado "+processedItems+" elementos. El batchSize es de: " + batchSize + " elementos", ex);
			if ( !handleExceptions )
				throw new TxBatchProcessorException("Problemas al realizar el rollback de la transaccion", ex);
		}
		finally {
			processedItems = 0;
			inTransaction = false;
			inIteration = false;
			itemsOk = true; //arrancamos de nuevo un batch
		}
	}

	private void reportTxBatchPatternViolation( String mensaje ) {
		log.error(mensaje);
		log.error(ExceptionUtil.dumpStackTace());
	}

	private InitialContext getInitialContext(){

		 InitialContext ctx = null;
		 try{
			ctx = new InitialContext();
		 }
		 catch(NamingException ex){
			log.error("Problemas al obtener el contexto JNDI",ex);
		 }
		 return ctx;

	 }

	 private boolean shouldCommitTransaction(){

		 return (processedItems % batchSize == 0);
	 }

	private boolean inActiveTransaction() throws TxBatchProcessorException {
		try {
			return inTransaction && tx != null && tx.getStatus() != Status.STATUS_NO_TRANSACTION;
		} catch (SystemException e) {
			throw new TxBatchProcessorException( e );
		}
	}

	
	
	/**
	 * Chequea si esta o no dentro de una transaccion activa.
	 * Esto sirve particularmente para el DataProcessor ya que se llama
	 * desde contextos donde ya puede haber una transaccion activa (como un jsp)
	 * 
	 * @return
	 */
	public static boolean checkIfAlreadyInsideAnActiveTransaction() {
		boolean alreadyInActiveTransaction = false;
		try {
			UserTransaction actualTransaction = (UserTransaction) new InitialContext().lookup(TxBatchProcessorHelper.USER_TRANSACTION_JNDI);
			alreadyInActiveTransaction = actualTransaction != null && actualTransaction.getStatus() == Status.STATUS_ACTIVE; 
		} catch (Exception e) {
			//No hago nada
		}
		return alreadyInActiveTransaction;
	}
}
