package ar.com.sdd.commons.util.id;

import java.io.Serializable;


/**
 * Base para las clases que representen identificadores numericos. Tiene
 * metodos para crear el id a partir de un <code>String</code>, un <code>long</code>,
 * <code>Long</code> o <code>BigDecimal</code>.
 * Implementa metodos como <code>hashCode</code>, <code>toString</code> e
 * <code>equals</code> de forma estandard, siendo siempre posible sobreescribirlos.
 * Ademas tiene los proyectores necesarios para ver al id como cualquiera de los
 * tipos descriptos mas arriba
 *
 */
public class NumberId implements Serializable {
//public abstract class NumberId implements Serializable {

    private static final long serialVersionUID = 3666355064160613242L;
	/**
     * Contiene el valor del id.
     */
    protected long value;

//--- Constructores ------------------------------------------------------------

    /**
     * Construye un NumberId a partir de un String, intentando parsearlo a long.
     * <p>
     * Se intara parsear un long a partir del string. Si este no es el caso, se
     * arrojara una excepcion <code>IllegalArgumentException</code> indicando
     * que no se puede construir un NumberId a partir del string recibido.
     *
     * @param id String que representa el valor del NumberId (no puede ser <b>null</b>).
     */
     public NumberId(String id) {
        try {
            value = Long.parseLong(id);
        } catch (NumberFormatException ex) {
            throw new IllegalArgumentException("No se puede construir un "+ this.getClass().getName() +" a partir de [" + id + "]");
        } catch (NullPointerException ex) {
            throw new IllegalArgumentException("No se puede construir un "+ this.getClass().getName() +" a partir de [" + id + "]");
        }
     }

    /**
    * Construye un NumberId a partir de un Long, tomando el valor del mismo como su
    * valor.
    * <p>
    * Si el Long es <b>null</b>, se arroja una excepcion <code>IllegalArgumentException</code>
    * indicando que no se puede construir un NumberId a partir de <b>null</b>.
    *
    * @param id Long con el valor que debe tener el NumberId (no puede ser <b>null</b>)
    */
    public NumberId(Long id) {
        try {
            value = id.longValue();
        } catch (NullPointerException ex) {
            throw new IllegalArgumentException("No se puede construir un "+ this.getClass().getName() +" a partir de [" + id + "]");
        }
    }


    /**
     * Construye un NumberId a partir de un long, tomando el valor del mismo como su
     * valor.<br>
     */
    public NumberId(long id) {
        value = id;
    }


//--- Metodos publicos ------------------------------------------------------------------------

    /**
     * Convierte el NumberId a String
     */
    public String toString() {
        return String.valueOf(value);
    }


    /**
    * Determina cuando dos NumberIds son iguales.
    * @return <b>true</b> cuando el <b>id</b> es un <code>NumberId</code> y
    *           el valor coincide con el de esta instancia.
    */
    public boolean equals(Object id) {
        return  (id != null) &&
                (id instanceof NumberId) &&
                (value == ((NumberId)id).value);
    }

    /**
     * Debe devolver algun codigo de hash adecuado
     */
    public int hashCode() {
        return Long.valueOf(value).hashCode();
    }

    /**
     * Devuelve el valor del NumberId
     *
     */
     public long longValue() {
        return value;
     }

//Cambio por creacion de modulo sddutil-java
	public Long getLong() {
        //return new Long(this.longValue());
        return Long.valueOf(value);
	}

	public long getValue(){
		return value;
	}

}
