package ar.com.sdd.commons.util.lineparser;

import java.util.HashMap;
import java.util.Map;


/**
 * Parsea una linea en formato texto plano. Hace falta describir el formato
 * del registro (tipo de registro, nombre de los campos, etc.).
 * Considera que todos los campos son de tipo String.
 *
 * Subclases: CSVLineParser, FixedLineParser.
 *
 * @author Andres Ferrari.
 */
public abstract class LineParser {

	// Tipos de parsing
	public static int PARSE_NONE	= -1;	// Sin inicializar
	public static int PARSE_CSV  	= 0;	// Registros con campos separados por comas
	public static int PARSE_FIXED	= 1;	// Registros de longitud fija

	/**
	 * Cantidad de campos en el registro.
	 */
	protected int numOfFields;

	/**
	 * Nombres de los campos del registro.
	 */
	protected String[] fieldNames;
	
	/**
	 * Valores de cada campo.
	 */
	protected String[] fieldValues;

	/**
	 * Indice numerico de cada campo indexado por el nombre del campo.
	 */
	protected Map fieldIndexesByName;
	
	
	//-----------------------------------------------------------------------------//
	
	/**
	 * Constructor.
	 * @param fieldNames Determina los nombres de los campos asi como su cantidad.
	 */
	public LineParser(String[] fieldNames) {
		
		fieldNames = standarizeFieldName(fieldNames);
		this.numOfFields = fieldNames.length;
		this.fieldNames  = fieldNames;
		this.fieldValues = new String[fieldNames.length];
		
		// Indexa indices de campos por nombre de campo,
		// de forma de acelerar la busqueda de valores por nombres de campo
		this.fieldIndexesByName = new HashMap();
		for (int i = 0; i < numOfFields; i++) {
			fieldIndexesByName.put(fieldNames[i], Integer.valueOf(i));
		}
	}	
	

	/**
	 * Devuelve la cantidad de campos.
	 */
	public int getNumOfFields() { return numOfFields; }


	/**
	 * Obtiene el valor del i-esimo campo.
	 */
	public String getFieldValue(int i) {
		
		String fieldValue = (i >= 0 && i < fieldValues.length) ? fieldValues[i] : null;
		return fieldValue;
	}


	/**
	 * Obtiene el valor del campo cuyo nombre es 'fieldName'.
	 */
	public String getFieldValue(String fieldName) {
		
		// Obtiene el indice a partir del nombre del campo
		int idx = getFieldIndex(fieldName);
		// Si el indice existe, obtengo el valor del campo; si no devuelvo null
		String retValue = (idx != -1) ? fieldValues[idx] : null;
		
		return retValue;
	}


	/**
	 * Obtiene el nombre del i-esimo campo.
	 */
	public String getFieldName(int i) {
		
		String fieldName = (i >= 0 && i < fieldNames.length) ? fieldNames[i] : null;
		return fieldName;
	}
	

	/**
	 * Obtiene el indice de un campo a partir de su nombre. Devuelve -1 si
	 * el campo no existe.
	 */
	public int getFieldIndex(String fieldName) {

		// Obtiene el indice a partir del nombre del campo
		Integer idx = (Integer) fieldIndexesByName.get(standarizeFieldName(fieldName));
		// Si el indice existe, lo casteo a int; si no devuelvo -1
		int retValue = (idx != null) ? idx.intValue() : -1;
		
		return retValue;
	}
	
	
	/**
	 * Es responsabilidad de cada subclase dar una implementacion
	 * adecuada al mecanismo de parsing.
	 *
	 * @param lineBuffer linea de texto a parsear.
	 */
	public abstract void parse(String lineBuffer) throws LineParserException;
	
	
	//-----------------------------------------------------------------------------//
	
	/**
	 * Pone los valores de todos los fields en null.
	 */
	protected void resetFieldValues() {
		
		for (int i = 0; i < getNumOfFields(); i++)
			fieldValues[i] = null;
	}

	public static String standarizeFieldName(String field) {
		return field.toLowerCase().replaceAll("[. _-]", "");
	}
	public static String[] standarizeFieldName(String fields[]) {
		for (int i =0; i<  fields.length; i++) {
			fields[i] = standarizeFieldName(fields[i]);
		}
		return fields;
	}
	
}