package ar.com.sdd.commons.util.mapper;

import ar.com.sdd.commons.util.StringUtil;
import groovy.lang.Binding;
import groovy.lang.GroovyShell;
import groovy.lang.Script;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.util.HashMap;
import java.util.Map;

/**
 * Mapper que evalua expresiones usando un shell de Groovy
 * En el decode acepta un conjunto de variables y una expresion, todo separado por punto y coma
 * Cada variable hay que prefijarla con 'i' para enteros, 'l' para longs, 'd' para doubles y 's' para strings
 *  
 * Ejemplo 1:
 *   decode="d1=0:15;d2=15:30;exp=(d2!=0.0) ? (d1/d2).doubleValue().round(2) : 0.0"
 *   
 *   En este caso hay dos variables double, d1 (que toma el valor de lo que haya entre las posiciones 0 y 15) y d2
 *   (que toma el valor de lo que haya entre las posiciones 15 y 30).
 *   Si el valor de d2 es distinto de cero, devuelve la division de d1 sobre d2, redondeada a dos decimales.
 *   Si el valor de d2 es cero, devuelve cero.
 *   
 *   Si viene: "000000019271.95        4014.99"
 *   Devuelve: "4.80"
 *   
 *   Si viene "000000019271.95        0000.00"
 *   Devuelve: "0.0" (porque d2 vale 0)
 *   
 * Ejemplo 2:
 *   decode="s1=0:10;s2=10:20;exp=(s1[0]+s2[0]).toUpperCase()"
 *   
 *   Aqui hay dos variables string, s1 y s2, y evalua la expresion tomando la primer letra de s1, concatenandola
 *   con la primer letra de s2 y pasando todo a mayuscula
 * 
 *   Si viene: "un perro  otro gato "
 *   Devuelve: "UO"
 * 
 * @author eviera
 *
 */
public class GroovyMapper extends ValueMapper {

	private static Logger log = LogManager.getLogger(GroovyMapper.class);
	
	private Map<String, String> variables = new HashMap<String, String>();
	private String expression = "";
	private Script compiledScript = null;
	
	public GroovyMapper(String decode) {
		try {
			String tokens[] = decode.split(";");
			if (tokens.length > 0) {
				for (String token : tokens) {
					//Parto el token en dos por el primer igual. No uso split, porque la exp podria tener el simbolo igual
					String name = token.substring(0, token.indexOf('='));
					String value = token.substring(name.length()+1);
					
					if (name.startsWith("exp")) {
						expression = value;
					//Todo lo que no sea la expresion, son variables
					} else {
						variables.put(name, value);
					}
				}
				if (expression != null && !expression.equals("")) {
					GroovyShell groovyShell = new GroovyShell();
				    compiledScript = groovyShell.parse(expression);
				}
			}
		} catch (Exception e) {
			log.error("Error interpretando el decode:"+decode, e);
		}
		
	}
	
	
	@Override
	public String map(String source) {
		if (compiledScript != null) {
			Binding binding = new Binding();
			//Recorro las variables y las voy atando al contexto de groovy
			for (String variable : variables.keySet()) {
				
				//Obtengo el valor como string, parseando el source en la posicion que indica la variable
				String value = StringUtil.fromToParser(source, variables.get(variable));
				
				//Depende de la letra con la que empieza la variable es el tipo de la misma
				switch (variable.charAt(0)) {
					case 's': 
						binding.setVariable(variable, value);
						break;
					case 'd':
						//Caso especial. Soporto que venga el simbolo menos como ultimo caracter (al sacarle los espacios)
						if (value.indexOf('-') > 0 && value.charAt(0) != '-') {
							value = value.trim();
							//Le pego el menos adelante
							value = "-" + value.substring(0,value.length()-1).trim();
						}
						binding.setVariable(variable, Double.parseDouble(StringUtil.nonEmpty(value, "0")));
						break;
					case 'l':
						binding.setVariable(variable, Long.parseLong(StringUtil.nonEmpty(value, "0")));
						break;
					case 'i':
						binding.setVariable(variable, Integer.parseInt(StringUtil.nonEmpty(value, "0")));
						break;
					default:
						log.error("Indicador de tipo invalido ["+variable.charAt(0)+"] para "+variable + "/" + value);
				}
				
			}
			
			//Ejecuto la expresion
			compiledScript.setBinding(binding);
			Object result = compiledScript.run();
			return String.valueOf(result);
		} else {
			return null;
		}
	}

}
