package ar.com.sdd.commons.util.security;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.bouncycastle.bcpg.ArmoredOutputStream;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openpgp.*;

import java.io.*;
import java.security.NoSuchProviderException;
import java.security.SecureRandom;
import java.security.Security;
import java.security.SignatureException;
import java.util.Date;
import java.util.Iterator;


/**
 * @author dachcar
 * Clase que brinda soporte para encriptar / desencriptar streams
 * Se utiliza PGP como motor de encriptacion, en particular se usa BouncyCastle como provider
 */
public class PGPEncryptor {
   
private static Logger log = LogManager.getLogger(PGPEncryptor.class);
//solo se declara el provider para que se inicialice una vez
private static int jceProvider = Security.addProvider(new BouncyCastleProvider()); 

	/**
	 * Firma / Encripta un stream
	 * Notar que ademas comprime.
	 * @param sourceIn El stream a firmar / encriptar
	 * @param sourceInLength Tama#o del stream a firmar / encriptar
	 * @param sourceFileName Nombre del archivo de origen(necesario para descencriptar luego)
	 * @param targetOut Stream de salida (Ahi se guarda el stream de entrada firmado y encriptado)
	 * @param privateKeyIn Stream con la clave privada. 
	 * @param privateKeyPass Contrase#a de la clave privada.
	 * @param publicKeyIn Stream con la clave publica
	 */
	public static void encryptStream(InputStream sourceIn,
								     long sourceInLength,
								     String sourceFileName,
								     OutputStream targetOut,
								     InputStream privateKeyIn,
								     char[] privateKeyPass,
								     InputStream publicKeyIn){
			
		try{
		
			int DEFAULT_BUFFER_SIZE=2048;
			PGPSecretKey secretKey = readSecretKey(privateKeyIn);
			PGPPrivateKey signingKey = secretKey.extractPrivateKey(privateKeyPass, "BC");
			OutputStream compressedOut = null;
			OutputStream literalOut = null;
			BufferedInputStream in = null;
			File literalDataFile = null;
			int bytesRead = 0;
			byte[] buffer = null;
			
			ByteArrayOutputStream tmpOut = new ByteArrayOutputStream();
			//Se genera la firma
			PGPSignatureGenerator signGen = new PGPSignatureGenerator(secretKey.getPublicKey().getAlgorithm(), PGPUtil.SHA1, "BC");
			signGen.initSign(PGPSignature.BINARY_DOCUMENT, signingKey);
			// comprime
			PGPCompressedDataGenerator compressedDataGenerator = new PGPCompressedDataGenerator(PGPCompressedData.ZIP);
			compressedOut = compressedDataGenerator.open(tmpOut);
			PGPOnePassSignature onePassSignature = signGen.generateOnePassVersion(false);
			onePassSignature.encode(compressedOut);
			PGPLiteralDataGenerator literalDataGenerator = new PGPLiteralDataGenerator();
			//el 3er parametro decide el nombre del archivo al desencriptar
			literalOut = literalDataGenerator.open(compressedOut, PGPLiteralData.BINARY, sourceFileName,sourceInLength,new Date());
			
			in = new BufferedInputStream(sourceIn, DEFAULT_BUFFER_SIZE);
			buffer = new byte[DEFAULT_BUFFER_SIZE];
			bytesRead = 0;
			while ((bytesRead = in.read(buffer)) != -1) {
				literalOut.write(buffer, 0, bytesRead);
				signGen.update(buffer, 0, bytesRead);
			}
			signGen.generate().encode(compressedOut);
			literalDataGenerator.close();
			compressedDataGenerator.close();
			
			// se cierran los streams
			if (compressedOut != null) 
				compressedOut.close();
			if (tmpOut != null) 
				tmpOut.close();
			if (literalOut != null) 
				literalOut.close();	
			if (in != null) 
				in.close();
		
			OutputStream out = null;
			InputStream literalIn = null;
		
			// se encripta el texto ya firmado
			PGPEncryptedDataGenerator encryptedDataGenerator = new PGPEncryptedDataGenerator(PGPEncryptedData.CAST5, true, new SecureRandom(), "BC");
	
			encryptedDataGenerator.addMethod(readPublicKey(publicKeyIn));
	
			out = new BufferedOutputStream(targetOut, DEFAULT_BUFFER_SIZE);
			out = new ArmoredOutputStream(out);
	
			byte[] tmpBytes = tmpOut.toByteArray();
			out = encryptedDataGenerator.open(out, tmpBytes.length);
			literalIn = new BufferedInputStream(new ByteArrayInputStream(tmpBytes), DEFAULT_BUFFER_SIZE);
			bytesRead = 0;
			while ((bytesRead = literalIn.read(buffer)) != -1) {
				out.write(buffer, 0, bytesRead);
			}
			encryptedDataGenerator.close();
		 
			if (out != null)
				out.close();	
			if (literalIn != null) 
				literalIn.close();
			if (literalDataFile != null) 
				literalDataFile.delete();
				
			log.debug("Firma / Encriptacion del stream finalizada.");
		}
		catch(Exception e){
			log.error("Ha ocurrido un error", e);
		}
	}
	
	
	/**
	 * desencripta / verfica una firma sobre un stream
	 * @param in El stream a desencriptar/verificar
	 * @param privateKeyIn Clave privada para desencriptar
	 * @param privateKeyPass Contrase#a de la clave Privada
	 * @param publicKeyPass Clave publica para verificar la firma
	 * @throws Exception 
	 */
	
	public static void decryptStream(InputStream in, 
									  InputStream privateKeyIn, 
									  char[] privateKeyPass,
									  InputStream publicKeyIn) throws Exception {
		
		in = PGPUtil.getDecoderStream(in);
		
		try {
			PGPObjectFactory pgpF = new PGPObjectFactory(in);
			PGPEncryptedDataList enc;

			Object o = pgpF.nextObject();
			// El primer objeto podria ser un PGP marker packet.
			if (o instanceof PGPEncryptedDataList) {
				enc = (PGPEncryptedDataList) o;
			} else {
				enc = (PGPEncryptedDataList) pgpF.nextObject();
			}

			// Se obtiene el objeto clave privada
			Iterator it = enc.getEncryptedDataObjects();
			PGPPrivateKey sKey = null;
			PGPPublicKeyEncryptedData pbe = null;

			while (sKey == null && it.hasNext()) {
				pbe = (PGPPublicKeyEncryptedData) it.next();
				sKey = findSecretKey(privateKeyIn, pbe.getKeyID(), privateKeyPass);
			}

			if (sKey == null) {
				throw new IllegalArgumentException("No se encontro la clave privada para el mensaje.");
			}

			InputStream clear = pbe.getDataStream(sKey, "BC");
			PGPObjectFactory plainFact = new PGPObjectFactory(clear);
			Object message = plainFact.nextObject();

			PGPObjectFactory pgpFact = null;

			if (message instanceof PGPCompressedData) {
				
				PGPCompressedData cData = (PGPCompressedData) message;
				/*PGPObjectFactory*/pgpFact = new PGPObjectFactory(cData.getDataStream());
				message = pgpFact.nextObject();

			}

			//manejo de los distintos tipos de mensaje: Literal o firmado
			if (message instanceof PGPLiteralData) {
				
				PGPLiteralData ld = (PGPLiteralData) message;
				OutputStream fOut = new BufferedOutputStream(new FileOutputStream(ld.getFileName()));
				InputStream unc = ld.getInputStream();
				int ch;
				while ((ch = unc.read()) >= 0) {
					fOut.write(ch);
				}

			} else if (message instanceof PGPOnePassSignatureList) {
				// contiene una firma, hay que verificarla
			
				// esta es la lista de firmas
				PGPOnePassSignatureList sigList = (PGPOnePassSignatureList) message;
				PGPOnePassSignature ops = sigList.get(0);

				PGPLiteralData p2 = (PGPLiteralData) pgpFact.nextObject();

				InputStream dIn = p2.getInputStream();

				
				String newFileName = p2.getFileName();
				if (newFileName==null || newFileName.equals(""))
					newFileName="salida.out";
				
				OutputStream fOut = new BufferedOutputStream(new FileOutputStream(newFileName));
				
				// se obtiene el objeto clave publica
				PGPPublicKeyRingCollection pgpRing = new PGPPublicKeyRingCollection(PGPUtil.getDecoderStream(publicKeyIn));
				PGPPublicKey pubKey = pgpRing.getPublicKey(ops.getKeyID());
				ops.initVerify(pubKey, "BC");

				int ch;
				while ((ch = dIn.read()) >= 0) {
					ops.update((byte) ch);
					fOut.write((byte) ch);
				}

				PGPSignatureList p3 = (PGPSignatureList) pgpFact.nextObject();

				if (!ops.verify(p3.get(0))) {
					throw new SignatureException("La firma es incorrecta.");
				} else {
					log.debug("Firma verificada con exito.");
				}
				
			} else {
				throw new PGPException("El mensaje no contiene un tipo PGP conocido.");
			}

			if (pbe.isIntegrityProtected()) {
				if (!pbe.verify()) {
					log.debug("Chequeo de integridad fallido!");
				} else {
					log.debug("Chequeo de integridad exitoso");
				}
			} else {
				log.debug("No se chequea integridad");
			}
		} catch (PGPException e) {
			log.error("Ha ocurrido un error", e);
			if (e.getUnderlyingException() != null) {
				log.error(e.getUnderlyingException());
			}
		}
	}
	
	
	/**
	 * Obtiene una clave publica a traves de un keyring
	 * @param in Stream que contiene el keyring
	 * @return la clave publica
	 * @throws IOException
	 * @throws PGPException
	 */
	private static PGPPublicKey readPublicKey(InputStream in)
	throws IOException, PGPException {
		
		in = PGPUtil.getDecoderStream(in);
		PGPPublicKeyRingCollection pgpPub = new PGPPublicKeyRingCollection(in);
		PGPPublicKey key = null;
		Iterator rIt = pgpPub.getKeyRings();
		while (key == null && rIt.hasNext()) {
			PGPPublicKeyRing kRing = (PGPPublicKeyRing) rIt.next();
			Iterator kIt = kRing.getPublicKeys();
			while (key == null && kIt.hasNext()) {
				PGPPublicKey k = (PGPPublicKey) kIt.next();
				if (k.isEncryptionKey()) {
					key = k;
				}
			}
		}

		if (key == null) {
			throw new IllegalArgumentException("No se encontro la clave publica en el key ring.");
		}

		return key;
	}
	
	
	
	/**
	 * Obtiene la clave privada de un keyRing 
	 * 
	 * @param in Stream que contiene el keyRing
	 * @return La clave privada
	 * @throws IOException
	 * @throws PGPException
	 */
	private static PGPSecretKey readSecretKey(InputStream in)
			throws IOException, PGPException {
		in = PGPUtil.getDecoderStream(in);

		PGPSecretKeyRingCollection pgpSec = new PGPSecretKeyRingCollection(in);

		PGPSecretKey key = null;
		Iterator rIt = pgpSec.getKeyRings();
		while (key == null && rIt.hasNext()) {
			PGPSecretKeyRing kRing = (PGPSecretKeyRing) rIt.next();
			Iterator kIt = kRing.getSecretKeys();
			while (key == null && kIt.hasNext()) {
				PGPSecretKey k = (PGPSecretKey) kIt.next();
				if (k.isSigningKey()) {
					key = k;
				}
			}
		}

		if (key == null) {
			throw new IllegalArgumentException("No se encontro la clave privada en el keyring.");
		}

		return key;
	}
	

	public static PGPPrivateKey findSecretKey(InputStream keyIn, 
											   long keyID,
											   char[] pass) throws IOException, PGPException,NoSuchProviderException {
		
		PGPSecretKeyRingCollection pgpSec = new PGPSecretKeyRingCollection(
		PGPUtil.getDecoderStream(keyIn));
		PGPSecretKey pgpSecKey = pgpSec.getSecretKey(keyID);
		if (pgpSecKey == null) {
			return null;
		}
		return pgpSecKey.extractPrivateKey(pass, "BC");
	}
	
	// main para hacer pruebas
	/*public static void main(String[] args) throws Exception {
		
		Security.addProvider(new BouncyCastleProvider());

		// completar con los datos que correspondan
		String clearFileName      = "texto.txt";
		String encryptedFileName  = "texto.txt.asc";
		String publicKeyFileName  = "xxx.pub";
		String privateKeyFileName = "xxx.sec";
		String privateKeyPass     = "xxxx xxxx xxxx";
		
		if (args[0].equals("-e")) {

			FileInputStream privateKeyIn = new FileInputStream(privateKeyFileName);
			FileInputStream publicKeyIn = new FileInputStream(publicKeyFileName);
			FileOutputStream out = new FileOutputStream(encryptedFileName);

			File f = new File(clearFileName);
			InputStream streamToEncrypt = new FileInputStream(clearFileName);
			
			encryptStream(streamToEncrypt,f.length(),clearFileName,out,privateKeyIn,privateKeyPass.toCharArray(),publicKeyIn);
		} else if (args[0].equals("-d")) {

			FileInputStream in = new FileInputStream(encryptedFileName);
			FileInputStream privateKeyIn = new FileInputStream(privateKeyFileName);
			FileInputStream publicKeyIn  = new FileInputStream(publicKeyFileName);
			decryptStream(in, privateKeyIn, privateKeyPass.toCharArray(),publicKeyIn);
		
		} else {
			log.debug("uso: PGPEncryptor -e|-d ");
		}
	}*/
	
		
}