package ar.com.sdd.commons.util.security;

import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openpgp.*;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.security.NoSuchProviderException;
import java.security.Security;
import java.util.Iterator;

public class PGPHelper {
    
    static {
        Security.addProvider( new BouncyCastleProvider() );
    }

    /**
     * Obtiene la primer clave privada de un keyRing 
     * 
     * @param keyFileName Nombre del archivo que contiene el keyRing
     * @return La primer clave privada
     * @throws IOException
     * @throws PGPException
     */
    public PGPSecretKey readSecretKey( String keyFileName ) throws IOException, PGPException {
        return readSecretKey( keyFileName, null );
    }

    /**
     * Obtiene la clave privada de un keyRing 
     * 
     * @param keyFileName Nombre del archivo que contiene el keyRing
     * @param publicKeyData item a desencriptar para el que se requiere la clave privada, o null para buscar la primer clave privada
     * @return La clave privada
     * @throws IOException
     * @throws PGPException
     */
    public PGPSecretKey readSecretKey( String keyFileName, PGPPublicKeyEncryptedData publicKeyData ) throws IOException, PGPException {
        FileInputStream keyStream = new FileInputStream( keyFileName );
        try {
            return readSecretKey( keyStream, publicKeyData );
        } catch ( FileNotFoundException fnfe ) {
            throw new IOException( "No se encontro el archivo de clave privada: " + keyFileName );
        } finally {
            keyStream.close();
        }
    }

    /**
     * Obtiene la clave privada de un keyRing 
     * 
     * @param inKey Stream que contiene el keyRing
     * @param publicKeyData item a desencriptar para el que se requiere la clave privada, o null para buscar la primer clave privada
     * @return La clave privada
     * @throws IOException
     * @throws PGPException
     */
    private PGPSecretKey readSecretKey( InputStream inKey, PGPPublicKeyEncryptedData publicKeyData ) throws IOException, PGPException {
        PGPSecretKey key = null;
        InputStream in = PGPUtil.getDecoderStream( inKey );
        try {
            PGPSecretKeyRingCollection pgpSec = new PGPSecretKeyRingCollection( in );
            if ( publicKeyData == null ) {
                Iterator rIt = pgpSec.getKeyRings();
                while (key == null && rIt.hasNext())
                {
                    PGPSecretKeyRing kRing = (PGPSecretKeyRing) rIt.next();
                    Iterator kIt = kRing.getSecretKeys();
                    while (key == null && kIt.hasNext())
                    {
                        PGPSecretKey k = (PGPSecretKey) kIt.next();
                        if ( k.isSigningKey() )
                        {
                            key = k;
                        }
                    }
                }
            } else {
                key = pgpSec.getSecretKey( publicKeyData.getKeyID() );
            }
        } finally {
            in.close();
        }
        if (key == null)
        {
            throw new IllegalArgumentException("No se encontro la clave privada en el keyring.");
        }
        return key;
    }

    /**
     * Obtiene la clave publica de un keyRing 
     * 
     * @param keyFileName Nombre del archivo que contiene el keyRing
     * @return La clave publica
     * @throws IOException
     * @throws PGPException
     */
    public PGPPublicKey readPublicKey( String keyFileName ) throws IOException, PGPException {
        return readPublicKey( keyFileName, null );
    }
    
    /**
     * Obtiene la clave publica de un keyRing 
     * 
     * @param keyFileName Nombre del archivo que contiene el keyRing
     * @return La clave publica
     * @throws IOException
     * @throws PGPException
     */
    public PGPPublicKey readPublicKey( String keyFileName, PGPOnePassSignature signature ) throws IOException, PGPException {
        FileInputStream keyStream = new FileInputStream( keyFileName );
        try {
            return readPublicKey( keyStream, signature );
        } catch ( FileNotFoundException fnfe ) {
            throw new IOException( "No se encontro el archivo de clave publica: " + keyFileName );
        } finally {
            keyStream.close();
        }
    }
    
    /**
     * Obtiene la clave publica de un keyRing 
     * 
     * @param inKey Stream que contiene el keyRing
     * @return La clave privada
     * @throws IOException
     * @throws PGPException
     */
    public PGPPublicKey readPublicKey( InputStream inKey, PGPOnePassSignature signature ) throws IOException, PGPException {
        PGPPublicKey key = null;
        InputStream in = PGPUtil.getDecoderStream( inKey );
        try {
            PGPPublicKeyRingCollection pgpPub = new PGPPublicKeyRingCollection( in );
            if ( signature == null ) {
                Iterator rIt = pgpPub.getKeyRings();
                while (key == null && rIt.hasNext())
                {
                    PGPPublicKeyRing kRing = (PGPPublicKeyRing) rIt.next();
                    Iterator kIt = kRing.getPublicKeys();
                    while (key == null && kIt.hasNext())
                    {
                        PGPPublicKey k = (PGPPublicKey) kIt.next();
                        if (k.isEncryptionKey())
                        {
                            key = k;
                        }
                    }
                }
            } else
                key = pgpPub.getPublicKey( signature.getKeyID() );
        } finally {
            in.close();
        }
        if (key == null)
        {
            throw new IllegalArgumentException("No se encontro la clave publica en el keyring.");
        }
        return key;
    }
    
    public PGPPrivateKey getPrivateKey( PGPSecretKey key, char passphrase[] ) throws PGPException {
        try {
            return key.extractPrivateKey( passphrase, "BC" );
        } catch ( NoSuchProviderException e ) {
            throw new PGPException( "Faltan agregar en el classpath los JARs de BouncyCastle", e );
        }
    }
    
}
