package ar.com.sdd.commons.util.xml;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;


/**
 * Variante de XMLData que no esta hardcodeado para construir un arbol en memoria.
 * En conjuncion con un derivado adecuado de XMLDataHandler adecuado, puede usarse 
 * para procesar archivos muy grandes, cosa que XMLData no puede.
 *
 * @author Andres Ferrari
 */
public class XMLDataNoTree {
	
	// Handler que maneja los eventos que se disparan durante el parsing
	private XMLDataHandler xmlDataHandler;


	//----------------------------------------------------------------------------//

	/**
	 * Constructor default
	 */
	public XMLDataNoTree(XMLDataHandler xmlDataHandler) {
		this.xmlDataHandler = xmlDataHandler;
	}
	
	
	/**
	 * Recibe el nombre del archivo con el XML, lo parsea, procesando por separado
	 * cada conjunto de items englobados por uno de primer nivel (i.e. procesa de
	 * a un documento, una relacion, etc.)
	 */
	public void parse(InputStream xmlDocument) throws XMLDataException {
		
		try {
		
			SAXParserFactory spf = SAXParserFactory.newInstance();
			SAXParser saxParser;
			saxParser = spf.newSAXParser();
		
			try {
				
				saxParser.parse(xmlDocument, xmlDataHandler);
				
			} catch (SAXException ex) {
			
				throw new XMLDataException(
								"Error al parsear el documento [" + xmlDocument + "]",
								ex,
								false);
								
			} catch (IOException ex) {
				
				throw new XMLDataException(
								"Error al leer el documento [" + xmlDocument + "]",
								ex,
								false);
								
			}
		
		} catch (SAXException ex) {
			
			throw new XMLDataException(
								"Error al inicializar el parser para el documento ["
								+ xmlDocument
								+ "]",
								ex,
								true);
		
		} catch (ParserConfigurationException ex) {
			
			throw new XMLDataException(
								"Error de configuracion en el parser para el documento ["
								+ xmlDocument
								+ "]",
								ex,
								true);
								
		}
	}
	
	
	/**
	 * Recibe el nombre del archivo con el XML, lo parsea, procesando por separado
	 * cada conjunto de items englobados por uno de primer nivel (i.e. procesa de
	 * a un documento, una relacion, etc.)
	 */
	public void parse(String xmlDocument) throws XMLDataException {
		
		try {
		
			SAXParserFactory spf = SAXParserFactory.newInstance();
			SAXParser saxParser;
			saxParser = spf.newSAXParser();
		
			try {
				
				saxParser.parse(new File(xmlDocument), xmlDataHandler);
				
			} catch (SAXException ex) {
			
				throw new XMLDataException(
								"Error al parsear el documento [" + xmlDocument + "]",
								ex,
								false);
								
			} catch (IOException ex) {
				
				throw new XMLDataException(
								"Error al leer el documento [" + xmlDocument + "]",
								ex,
								false);
								
			}
		
		} catch (SAXException ex) {
			
			throw new XMLDataException(
								"Error al inicializar el parser para el documento ["
								+ xmlDocument
								+ "]",
								ex,
								true);
		
		} catch (ParserConfigurationException ex) {
			
			throw new XMLDataException(
								"Error de configuracion en el parser para el documento ["
								+ xmlDocument
								+ "]",
								ex,
								true);
								
		}
	}
		

	//-----CLASES AUXILIARES----------------------------------------------------------//

	/***********************************************************************************
	 * XMLErrorHandler.
	 */
	class XMLErrorHandler implements ErrorHandler {

		public void error(SAXParseException e) {

			Logger log = LogManager.getLogger(XMLDataNoTree.class);
			log.error("Error en XML parser en "
					+ e.getPublicId()
					+ " ("
					+ e.getLineNumber()
					+ ", "
					+ e.getColumnNumber()
					+ ")", e);
		}
		
		
		public void fatalError(SAXParseException e) {
		
			Logger log = LogManager.getLogger(XMLDataNoTree.class);
			log.fatal("Error en XML parser en "
					+ e.getPublicId()
					+ " ("
					+ e.getLineNumber()
					+ ", "
					+ e.getColumnNumber()
					+ ")", e);
		}
		
		
		public void warning(SAXParseException e) {

			Logger log = LogManager.getLogger(XMLDataNoTree.class);
			log.warn("Error en XML parser en "
					+ e.getPublicId()
					+ " ("
					+ e.getLineNumber()
					+ ", "
					+ e.getColumnNumber()
					+ ")");
		}
		
    }

}
