package ar.com.sdd.commons.util.xml;

import org.apache.commons.lang3.StringUtils;

import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.TreeNode;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.util.*;


/**
 * Item de TAG xml.
 */
public class XMLTag extends DefaultMutableTreeNode implements java.io.Serializable {

    private static final long serialVersionUID = 1L;

    private String nsUri 		  = null;
	private String name 		  = null;
	private String data 		  = null;
	// declaraciones de namespaces
	private List<String> nsDeclarations = new ArrayList<String>();
	private Properties properties = new Properties();
	private Properties propertiesNsUri = new Properties();


	//----------------------------------------------------------------------------//

	public XMLTag(String name) {
		this(name, new Properties(), null);
	}


	public XMLTag(String name, Properties properties) {
		this(name, properties, null);
	}


	public XMLTag(String name, String data) {
		this(name, new Properties(), data);
	}


	public XMLTag(String name, Properties properties, String data) {

		setName(name);
		setProperties(properties);
		setData(data);
	}

	// Clonar un nodo
	public XMLTag( XMLTag copyFrom ) {
		// Clonar el tag sobre un nuevo objeto XMLTag
		this(copyFrom.getName());
        // Clonar Tag
        this.setNsUri( copyFrom.getNsUri() );

        // Clonar Data
        this.setData( copyFrom.getData() );

        // Clonar Atributos
        Iterator attrs = copyFrom.getProperties().keySet().iterator();
        while ( attrs.hasNext() ) {
            String attr = (String) attrs.next();
            this.setProperty( copyFrom.getPropertyNsUri(attr), attr, copyFrom.getProperty(attr) );
        }

        // Clonar Hijos
        Enumeration<TreeNode> children = copyFrom.children();
        while ( children.hasMoreElements() ) {
            XMLTag child = (XMLTag) children.nextElement();
            XMLTag clonedChild = new XMLTag(child);
            this.add( clonedChild );
        }
	}


	public String getName() { return name; }
	public void setName(String name) { this.name = name; }


	public Properties getProperties() { return properties; }
	public void setProperties(Properties properties) {
		this.properties = properties;
		this.propertiesNsUri = new Properties();
	}


	public String getProperty(String name) {
		return properties.getProperty(name);
	}

	public String getProperty(String name, String def) {
		return properties.getProperty(name, def);
	}

	public String getPropertyNsUri(String name) {
		return propertiesNsUri.getProperty(name);
	}

	public void setProperty(String name, String value) {
		setProperty("", name, value);
	}

	public void setProperty(String nsUri, String name, String value) {
		properties.setProperty(name, value);
		propertiesNsUri.setProperty(name, nsUri);
	}

	public void removeProperty(String name) {
		properties.remove(name);
		propertiesNsUri.remove(name);
	}


	public String getData(String dflt) {
		return ( (data == null || "".equals(data)) ? dflt : data );
	}

	public String getData() { return data; }
	public void setData(String data) { this.data = data; }


	/**
	 * Este caso especial se usa para agregar un pedacito a los datos ya existentes.
	 * Es necesario debido al modo en que funcionan algunos parsers SAX, para los cuales
	 * no esta garantizado que los datos lleguen en un solo "chunk".
	 */
	public void appendToData(String moreData) {
		//this.data = (this.data != null) ? (this.data + moreData) : moreData;
		if (this.data != null)
			this.data = this.data + "" + moreData;
		else
			this.data = moreData;
	}

	public void saveToStringBuilder(StringBuilder builder) {
		// Genero la cantidad de tabs necesarios acorde al level en el que este, le resto 1 porque el layout esta dentro del source pero no lo quiero tener en cuenta
		String tabs = "\t".repeat(Math.max(0, getLevel() - 1));

		builder.append(tabs).append("<").append(getName());
		if (getProperties().size() > 0) {
			Enumeration<Object> names = getProperties().keys();
			while (names.hasMoreElements()) {
				String name = (String) names.nextElement();
				builder.append(" ").append(name).append(" =\"").append(getProperty(name)).append("\"");
			}
		}

		for (String ns : nsDeclarations) {
			builder.append(ns);
		}

		if (getChildCount() > 0) {
			builder.append(">\n");

			Enumeration<TreeNode> childs = children();
			while (childs.hasMoreElements()) {
				XMLTag tag = (XMLTag) childs.nextElement();
				tag.saveToStringBuilder(builder);
			}

			builder.append(tabs).append("</").append(getName()).append(">\n");
		} else if (StringUtils.isNotEmpty(getData())) {
			builder.append(">").append(getData()).append("</").append(getName()).append(">\n");
		} else {
			builder.append("/>\n");
		}
	}

	public void saveToStream(PrintStream out){
		saveToStream(new PrintWriter(out));
	}


	public void saveToStream(PrintWriter out) {

		String tabs = "";

		for (int i = 0; i < getLevel(); i++) {
			tabs += "\t";
		}

		out.print(tabs + "<" + getName());
		if (getProperties().size() > 0) {

			Enumeration names = getProperties().keys();

			while(names.hasMoreElements()) {
				String name = (String)names.nextElement();
				out.print(" " + name + "=\"" + getProperty(name) + "\"");
			}
		}
		for (String ns : nsDeclarations) { 
			out.print(ns);
		}

		if (getChildCount() > 0) {

			out.println(">");
			Enumeration<TreeNode> childs = children();

			while(childs.hasMoreElements()) {
				XMLTag tag = (XMLTag) childs.nextElement();
				tag.saveToStream(out);
			}

			out.println(tabs + "</" + getName() + ">");

		} else if((getData() != null) && (getData().length() > 0)) {

			out.println(">" + getData() + "</" + getName() + ">");

		} else
			out.println("/>");
	}


	/**
	 * Devuelve un unico elemento con el nombre recibido. Asume que por lo menos
	 * existe uno (sino arroja una <code>NullPointerException</code>) y devuelve
	 * el primero que encuentra.
	 * (Con busqueda recursiva por defecto)
	 */
	public XMLTag getItem(String name) {

		ArrayList<XMLTag> v = getItems(name, true);
		return ( (v.size() > 0) ? ( v.get(0)) : null );
	}


	/**
	 * Devuelve un unico elemento con el nombre recibido. Asume que por lo menos
	 * existe uno (sino arroja una <code>NullPointerException</code>) y devuelve
	 * el primero que encuentra.
	 */
	public XMLTag getItem(String name, boolean recursive) {

		ArrayList<XMLTag> v = getItems(name, recursive);
		return ( (v.size() > 0) ? (v.get(0)) : null );
	}


	/**
	 * Devuelve un unico elemento con el nombre recibido. Puede existir uno o mas,
	 * en cuyo caso devuelve primero que encuentra o ninguno, en cuyo caso devuelve
	 * <b>null</b>
	 */
	public XMLTag getOptionalItem(String name) {
		ArrayList<XMLTag> v = getItems(name);
		return  ( v!= null && v.size() > 0 ? v.get(0) : null) ;
	}


	/**
 	 * Devuelve los hijos directos del tag cuyo nombre coincide con <b>name</b>.
 	 * Si el name es null devuelve todos los hijos
	 * @return ArrayList con los tags que matchean
	 */
	public ArrayList<XMLTag> getFirstLevelItems(String name) {

		ArrayList<XMLTag> items = new ArrayList<XMLTag>();

		if (getChildCount() > 0) {

			Enumeration<TreeNode> childs = children();

			while (childs.hasMoreElements()) {
				XMLTag tag = (XMLTag) childs.nextElement();
				if(name == null || tag.getName().equalsIgnoreCase(name)) items.add(tag);
			}
		}

		return items;
	}


	/**
	 * Devuelve todos los elementos con un nombre especifico
	 * (Con busqueda recursiva por defecto)
	 */
	public ArrayList<XMLTag> getItems(String name) {
		return getItems(name, true);
	}


	public ArrayList<XMLTag> getItems(String name, boolean recursive) {
		return getItems( null, name, recursive );
	}


	public ArrayList<XMLTag> getItems(String nsUri, String name, boolean recursive) {

		ArrayList<XMLTag> items = new ArrayList<XMLTag>();

		if (getChildCount() > 0) {

			Enumeration<TreeNode> childs = children();
			while (childs.hasMoreElements()) {

				XMLTag tag = (XMLTag) childs.nextElement();
				if (name.equalsIgnoreCase(tag.getName())) {
                    if (nsUri == null || tag.getNsUri() != null && tag.getNsUri().equals(nsUri)) {
                        items.add(tag);
                    }
                }
				if (recursive) {
				    items.addAll(tag.getItems(nsUri, name, recursive));
                }
			}
		}

		return items;
	}

	public List<XMLTag> getFirstLevelChildren() {
		List<XMLTag> result = new ArrayList<XMLTag>(); 
		Enumeration<TreeNode> childs = children();
		if (childs != null) {
			while (childs.hasMoreElements()) {
				XMLTag tag = (XMLTag) childs.nextElement();
				result.add(tag);
			}
		}
		return result;
	}
	public XMLTag getFirstLevelChild() {
		XMLTag result = null; 
		Enumeration<TreeNode> childs = children();
		if (childs != null) {
			if (childs.hasMoreElements()) {
				result = (XMLTag) childs.nextElement();
			}
		}
		return result;
	}
	
	public XMLTag getParentTag()
	{
		return
		    (getParent() != null && getParent() instanceof XMLTag)?
		    		(XMLTag)getParent(): null;
	}

	public String getNamePath()
	{
		String namePath = getName()+"[";
		for(Object  key: this.getProperties().keySet() ) {
		    namePath+=(String)key+"="+getProperty((String)key)+';';
		}
		namePath+="]";
		XMLTag parentTag = getParentTag();
		if (parentTag!=null){
			namePath = parentTag.getNamePath()+"/"+namePath;
		}
		return  namePath;
	}

	public String toString() {
	    return toString('\n');
    }

	public String toString(char sep) {
		StringBuilder retr = new StringBuilder();
		retr.append("<").append(this.getName()).append(sep);
		Enumeration<Object> keys = this.getProperties().keys();
		while (keys.hasMoreElements()) {
			String key = (String)keys.nextElement();
			retr.append(key).append("=").append(this.getProperty(key)).append(sep);
		}
		retr.append(">\n");
		retr.append(this.getData()).append("\n");
		retr.append("</").append(this.getName()).append(">");
		return retr.toString();
	}

	public String getNsUri() {
		return nsUri;
	}

	public void setNsUri(String nsUri) {
		this.nsUri = nsUri;
	}


    public XMLTag deepClone() {
        // Clonar Tag
        XMLTag result = new XMLTag( getName() );
        result.setNsUri( getNsUri() );

        // Clonar Data
        result.setData( getData() );

        // Clonar Atributos
        Iterator attrs = getProperties().keySet().iterator();
        while ( attrs.hasNext() ) {
            String attr = (String) attrs.next();
            result.setProperty( getPropertyNsUri(attr), attr, getProperty(attr) );
        }

        // Clonar Hijos
        Enumeration<TreeNode> children = children();
        while ( children.hasMoreElements() ) {
            XMLTag child = (XMLTag) children.nextElement();
            XMLTag clonedChild = child.deepClone();
            result.add( clonedChild );
        }

        return result;
    }


	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		XMLTag other = (XMLTag) obj;
		if (data == null) {
			if (other.data != null)
				return false;
		} else if (!data.equals(other.data))
			return false;
		if (name == null) {
			if (other.name != null)
				return false;
		} else if (!name.equals(other.name))
			return false;
		if (nsDeclarations == null) {
			if (other.nsDeclarations != null)
				return false;
		} else if (!nsDeclarations.equals(other.nsDeclarations))
			return false;
		if (nsUri == null) {
			if (other.nsUri != null)
				return false;
		} else if (!nsUri.equals(other.nsUri))
			return false;
		if (properties == null) {
			if (other.properties != null)
				return false;
		} else if (!properties.equals(other.properties))
			return false;
		if (propertiesNsUri == null) {
			if (other.propertiesNsUri != null)
				return false;
		} else if (!propertiesNsUri.equals(other.propertiesNsUri))
			return false;
		return true;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((data == null) ? 0 : data.hashCode());
		result = prime * result + ((name == null) ? 0 : name.hashCode());
		result = prime * result + ((nsDeclarations == null) ? 0 : nsDeclarations.hashCode());
		result = prime * result + ((nsUri == null) ? 0 : nsUri.hashCode());
		result = prime * result + ((properties == null) ? 0 : properties.hashCode());
		result = prime * result + ((propertiesNsUri == null) ? 0 : propertiesNsUri.hashCode());
		return result;
	}


	public List<String> getNsDeclarations() {
		return nsDeclarations;
	}
	
	

}