package ar.com.sdd.commons.util;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.naming.Context;
import javax.naming.NamingException;
import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Types;

/**
 * Ejecuta prepared statements en forma batch<br/>
 * Para usarlo es necesario hacer codigo como el siguiente:
 *
 * @author sfreue@sdd
 */
public class BatchProcessor {
	
	private static Logger log = LogManager.getLogger(BatchProcessor.class);

    private boolean oldAutoCommit;
    private String dataSourceName;
    private Context jndiContext;
    private DataSource dataSource = null;
    private Connection connection = null;
    private PreparedStatement preparedStmt = null;
    private String sqlQuery = null;
    
    public BatchProcessor(
            Context jndiContext,
            String dataSourceName) {
            this.jndiContext = jndiContext;
            this.dataSourceName = dataSourceName;
    }
        
    public BatchProcessor( DataSource dataSource ) {
            this.dataSource = dataSource;
    }
        
    public void beginBatch( String sql ) {
    	if ( sqlQuery != null && !sqlQuery.equals( sql ) )
    		cleanUp();
    	sqlQuery = sql;
    }
    
    public void commitBatch() {
    	if ( connection != null )
	    	try {
	    		if ( preparedStmt != null )
	    			preparedStmt.executeBatch();
				connection.commit();
			} catch (SQLException e) {
	            throw new BatchProcessorException("Problemas al realizar el commit del DS ["+dataSourceName+"]: "+e, e);
			}
    }
    
    public void rollbackBatch() {
    	if ( connection != null )
	    	try {
				connection.rollback();
			} catch (SQLException e) {
	            throw new BatchProcessorException("Problemas al realizar el rollback del DS ["+dataSourceName+"]: "+e, e);
			}
    }
    
    public void close()
    {
    	rollbackBatch();
    	cleanUp();
    }

    public void execute( Object params[] ) {
    	PreparedStatement stmt = getPreparedStatement();
    	try {
			int nullType = Types.VARCHAR;
			for ( int i = 0; i < params.length; i++ )
				if ( params[ i ] == null )
					stmt.setNull( i + 1, nullType );
				else
					stmt.setObject( i + 1, params[ i ] );
			stmt.addBatch();
		} catch (SQLException e) {
		//} catch (Throwable e) {
            throw new BatchProcessorException("Problemas al ejecutar batch sobre el datasource ["+dataSourceName+"]: "+e, e);
		}
    }
    
    public Context getJndiContext() {
        return jndiContext;
    }

    public String getDataSourceName() {
        return dataSourceName;
    }

    //--- Metodos privados ------------------------------------------------------------------------
    private PreparedStatement getPreparedStatement() throws BatchProcessorException {
    	if ( preparedStmt == null )
			try {
				preparedStmt = getConnection().prepareStatement( sqlQuery );
            } catch (SQLException e) {
                throw new BatchProcessorException("Problemas al preparar el statement datasource ["+dataSourceName+"]: "+e, e);
			}
    	return preparedStmt;
    }
        
    private DataSource getDataSource() throws BatchProcessorException {
        if (dataSource == null) {
            try {
            	String fullDataSourceName = 
                    dataSourceName.startsWith( "java:" )? 
                            dataSourceName:
                            "java:comp/env/jdbc/" + dataSourceName;
                dataSource = (DataSource) jndiContext.lookup( fullDataSourceName );
            } catch (NamingException e) {
                throw new BatchProcessorException("Problemas al localizar el datasource ["+dataSourceName+"]: "+e, e);
            }
        }
        return dataSource;
    }

    private Connection getConnection() throws BatchProcessorException
    {
    	if ( connection == null )
			try {
				connection = getDataSource().getConnection();
				oldAutoCommit = connection.getAutoCommit();
				if ( oldAutoCommit )
					connection.setAutoCommit( false );
			} catch (SQLException e) {
                throw new BatchProcessorException("Problemas al obtener la conexion del DS ["+dataSourceName+"]: "+e, e);
			}
    	return connection;
    }
    
    private void cleanUp() {
    	if ( preparedStmt != null )
			try {
				preparedStmt.close();
			} catch (SQLException e) {
				log.error("Ha ocurrido un error", e);
			}
		preparedStmt = null;
    	if ( connection != null )
			try {
				if ( oldAutoCommit )
					connection.setAutoCommit( oldAutoCommit );
				connection.close();
			} catch (SQLException e) { 
				log.error("Ha ocurrido un error", e);
			}
		connection = null;
    }
    
}