package ar.com.sdd.commons.util.reflection;

import ar.com.sdd.commons.util.ObjectUtil;
import ar.com.sdd.commons.util.StringUtil;
import ar.com.sdd.commons.util.xml.XMLTag;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class ObjectBuilder {

    private static Logger log = LogManager.getLogger(ObjectBuilder.class);

    public static final String TAG_ID = "id";        //El ID provee un orden numerico de argumentos
    public static final String TAG_NAME = "name";
    public static final String TAG_TYPE = "type";
    public static final String TAG_CLASS = "class";
    public static final String TAG_NULL = "null";
    public static final String TAG_VALUE = "value";
    public static final String TAG_CONTEXTVALUE = "contextValue";
    public static final String TAG_ARGUMENTS = "arguments"; //@Deprecated
    public static final String TAG_ARGUMENT = "argument";
    public static final String TYPE_STRING = "String";
	public static final String TYPE_DATE = "Date";
	public static final String TYPE_COMPANY_EBF_API = "CompanyEbfApi";
    public static final String TYPE_BANK_EBF_API = "BankEbfApi";
    public static final String TYPE_BIG_DECIMAL = "BigDecimal";
    public static final String TYPE_DOUBLE_PRIMITIVE = "double";
    public static final String TYPE_MAP = "java.util.HashMap";
    private String name;
    private String id;
    private String objectClassName;
    private Object value;
    private boolean contextValue;
    private boolean isObjectValue;
    private boolean nullValue;
    private HashMap contextMap;

	private ArrayList<ObjectBuilder> arguments = null;

    public ObjectBuilder(XMLTag configuration) throws ObjectBuilderException {
        this(configuration, null);
    }

    public ObjectBuilder(XMLTag configuration, Map contextMap) throws ObjectBuilderException {
        this(configuration, new HashMap(contextMap));
    }

    public ObjectBuilder(XMLTag configuration, HashMap contextMap) throws ObjectBuilderException {
        // Como parte del refactory soporto dos tipos de tag.
        //   < streamconfig id=""  name = "" ....>     ->  como se usa cuando estaba el tag <StreamConfiguration> con los datos en el primero nivel
        // o
        //   <streamconfig >
        //      <argument id=""  name = "" ....>  ->  con la sintaxis nueva
        //  </streamconfig >
        // Decido uno u otro, segun si tengo info
        if (!describesAnyValue( configuration ) ) {
            //simplemente, me meto un nivel para adentro:
            configuration = configuration.getFirstLevelChild();
            if (configuration==null) {
                throw new ObjectBuilderException("ObjectBuilder: configuracion no describe AnyValue, y contenido null");
            }
        }
        setContextMap(contextMap);
        this.setId(configuration.getProperty(TAG_ID));
        this.setName(configuration.getProperty(TAG_NAME));
        this.setContextValue( getBooleanAttribute( configuration, TAG_CONTEXTVALUE, false ) );
        this.setNullValue( getBooleanAttribute( configuration, TAG_NULL, false ) );
        this.isObjectValue = !describesPrimitiveValue( configuration );
        if ( this.isObjectValue  )
            this.initializeObject( configuration );
        else
            this.initializePrimitive( configuration );
    }

    public static boolean describesPrimitiveValue( XMLTag argumentTag ) {
        return argumentTag.getProperty(TAG_TYPE) != null;
    }

    public static boolean describesObjectValue( XMLTag argumentTag ) {
        return argumentTag.getProperty(TAG_CLASS) != null;
    }

    public static boolean describesNullObjectValue( XMLTag argumentTag ) {
        return getBooleanAttribute( argumentTag, TAG_NULL, false );
    }

    public static boolean describesAnyValue( XMLTag argumentTag ) {
        return describesObjectValue( argumentTag )
            || describesPrimitiveValue(argumentTag ) 
            || describesNullObjectValue( argumentTag )
            ;
    }

    private static boolean getBooleanAttribute( XMLTag tag, String attrName, boolean defaultValue ) {
        String boolString = tag.getProperty(attrName);
        if(boolString != null)
            return (Boolean.valueOf( boolString ));
        return defaultValue;
    }

    private void initializePrimitive( XMLTag argumentTag ) throws ObjectBuilderException {
        this.setObjectClassName(argumentTag.getProperty(TAG_TYPE));
        if (getObjectClassName().equals( "xml" ) || 
       		getObjectClassName().equals( "XMLTag" ) ||	
        	getObjectClassName().equals( XMLTag.class.getName() )) {
        		this.setValue(argumentTag);
        
        //Manejo el caso de que venga un CDATA. Lo cambio al tipo String, con los datos del CDATA adentro
        } else if (getObjectClassName().equalsIgnoreCase("cdata")) {
        	this.setObjectClassName(TYPE_STRING);
        	this.setValue(argumentTag.getData());
        } else if (getObjectClassName().equalsIgnoreCase("context")) {
        	this.setObjectClassName(TYPE_MAP);
        	this.setValue(getContextMap());
        //Sino, le pongo el valor seteado en el atributo value
    	} else {
        	this.setValue(argumentTag.getProperty(TAG_VALUE));
        }
        
    }

    private void initializeObject( XMLTag argumentTag ) throws ObjectBuilderException {
        this.setObjectClassName(argumentTag.getProperty(TAG_CLASS));
        if ( !this.isNullValue() ) {
            this.initializeObjectArguments( argumentTag );
        }
    }

    private void initializeObjectArguments(XMLTag argumentsTag) throws ObjectBuilderException	{
        this.arguments = new ArrayList<ObjectBuilder>();
        if ( argumentsTag != null ) {
            List<XMLTag> argumentTags = argumentsTag.getItems(TAG_ARGUMENT,false);
        	//Cambio la implementacion de recibir 
        	//    argumentTag<Arguments><argument/>...</arguments>
        	// a  directamente
        	//    argumentTag<argument/>...</arguments>
            //@Deprecated: Mantengo esto por compatibilidad con el viejo esquema de <arguments>
            if (argumentTags==null || argumentTags.isEmpty()) {
            	argumentsTag = argumentsTag. getItem(TAG_ARGUMENTS);
            	if ( argumentsTag != null ) {
            		argumentTags = argumentsTag.getItems(TAG_ARGUMENT,false);
            	}
            }

            if ( argumentTags != null ) {
            	//Por performance, seteo un tamano razonable para arguments
            	this.arguments.ensureCapacity(argumentTags.size());
                for(XMLTag argumentTag:  argumentTags){
                	try{
                	ObjectBuilder argument = new ObjectBuilder(argumentTag, getContextMap());
                	ObjectUtil.arraySet(this.arguments,argument.getId(),argument);
                	} catch (Exception e) {
                		log.error("Ha ocurrido un error", e);
                	}
                }
            }
        }
	}

    public Object createObject() throws ObjectBuilderException {
        return createObject( null );
    }

    public Object createObject( ObjectBuilderContext context ) throws ObjectBuilderException {
        if ( this.isNullValue() )
            return null;

        if ( this.isContextValue() )
            return this.getValue();

		try {
            InstanceCreator instanciator = new InstanceCreator();
			instanciator.setInstanceClassName(this.getObjectClassName());
            if ( this.isObjectValue() ) {
                this.setInstanciatorObjectArgs( instanciator, context );
            } else
                this.setInstanciatorPrimitiveArgs( instanciator, context );
			return instanciator.createInstance();
		}
		catch(Exception e) {
			throw new ObjectBuilderException("Ocurrio una excepcion al crear la instancia de " + this.getObjectClassName(), e);
		}
	}

    private void setInstanciatorPrimitiveArgs( InstanceCreator instanciator, ObjectBuilderContext context ) throws InstanceCreatorException {
       	if (getValue() instanceof XMLTag) {
    		instanciator.addArgument(XMLTag.class.getName(), getValue());
       	} else if (getValue() instanceof HashMap) {
        	instanciator.addArgument(HashMap.class.getName(), getValue());
    	} else {
    		String theValue = (String) getValue();
            if ( context != null ) {
                theValue = contextReplace(context, theValue);
            }
            instanciator.addArgument( "String", theValue );
    	}
    }

    // Reemplaza los textos ${xxx}  con el valor asociado a xxxx en el context
    // Si el xxx no esta en el context, mantiene el texto original ${xxxx}
    // Opciones dentro de los parametos
    //     ${xxxx,padLength,padType,padChar}  permite hacer padding
    //              padLength:nro
    //              padType  :  N pad left, con 0 por default
    //                          A pad tight
    //              padChr   :  pr default '0'
    //     ${xxx:Format}
    //              Format: valores de Strkng.format  (no parece usarse)

	public static String contextReplace(ObjectBuilderContext context, String theValue) {
		int substFrom=0;
		while ( ( substFrom = theValue.indexOf( "${",substFrom ) ) >= 0 ) {
		    int substTo = theValue.indexOf( "}", substFrom );
            String tag = theValue.substring( substFrom + 2, substTo ); //texto completo entre {}
            String key = tag; //El que voy a ir actualizando
		    boolean padd = false;
		    int lengthPadd = 0;
		    String typePadd = null;
		    char charPadd = ' ';
		    if (key.indexOf(",") > 0){ //soporta padding
		    	padd = true;
		    	String[] params = key.split("[,]");
		    	String value = params[0];
		    	lengthPadd = Integer.parseInt(params[1]);
		    	typePadd = params[2].equalsIgnoreCase("N")?StringUtil.PADD_LEFT:StringUtil.PADD_RIGHT;
		    	charPadd = params[2].equalsIgnoreCase("N")?'0':' ';
		    	key = value;
		    }
		    String substFmt = null;
		    if (key.indexOf(":")>0) { //soporta formato
		    	int fmtPos = key.indexOf(":");
		    	substFmt = key.substring(fmtPos+1); //primero tomo el resto
		    	key = key.substring(0,fmtPos); //me quedo con la key
		    }

		    boolean keyFounded = true;
		    String value = ObjectUtil.safeToString(context.getContextValue(key), null );
            if (value==null) {
                //No encontre la key, dejo todo como estaba
                value = "${"+tag+"}";
                keyFounded = false;
            } else {
                if (substFmt != null) {
                    value = String.format(substFmt, value);
                }
                if (padd) {
                    value = StringUtil.padd(value, lengthPadd, charPadd, typePadd);
                }
            }
		    theValue =
		        theValue.substring( 0, substFrom ) +
		        value +
		        theValue.substring( substTo + 1 );
            if (!keyFounded) {
                substFrom++; //me muevo uno para encontrar algo nuevo
            }
		}
		return theValue;
	}

	private void setInstanciatorObjectArgs(InstanceCreator instanciator, ObjectBuilderContext context) throws ObjectBuilderException {
		try {
	 		for (ObjectBuilder argument : arguments) {
	 			if (argument != null) {
	 				instanciator.addArgument(argument.getObjectClassName(),argument.createObject(context));
	 			}
			}
		} catch(InstanceCreatorException e) {
			throw new ObjectBuilderException("Error al definir agrumentos al crear una instancia de " + this.getObjectClassName(), e);
		}
	}

	public void setContextArgumentValue(String name, Object value) {
		//Si me corresponde, seteo el valor
		if (isContextValue() && getName().equals(name)) {
			setValue(value);
		}
		//Propago para el resto de los argumentos
		if (this.arguments != null) {
			for (ObjectBuilder argument : this.arguments) {
				if (argument != null) {
					argument.setContextArgumentValue(name, value);
				}
			}
		}
	}

	public ArrayList<ObjectBuilder> getArguments() {
		return arguments;
	}

	public void setArguments(ArrayList<ObjectBuilder> arguments) {
		this.arguments = arguments;
	}

	public String getObjectClassName() {
		return objectClassName;
	}

	public void setObjectClassName(String objectClassName) {
		this.objectClassName = objectClassName;
	}

    public boolean isContextValue() {
        return contextValue;
    }

    public void setContextValue( boolean contextValue ) {
        this.contextValue = contextValue;
    }

    public boolean isObjectValue() {
        return isObjectValue;
    }

    public void setObjectValue( boolean isObjectValue ) {
        this.isObjectValue = isObjectValue;
    }

    public String getName() {
        return name;
    }

    public void setName( String name ) {
        this.name = name;
    }

    public String getId() {
        return id;
    }

    public void setId( String id ) {
        this.id=id ;
    }

    public Object getValue() {
        return value;
    }

    public void setValue( Object value ) {
        this.value = value;
    }

    public boolean isNullValue() {
        return nullValue;
    }

    public void setNullValue( boolean nullValue ) {
        this.nullValue = nullValue;
    }

	public void setContextMap(Map contextMap) {
		this.contextMap = new HashMap();
		this.contextMap.putAll(contextMap);
	}

	public void setContextMap(HashMap contextMap) {
		this.contextMap = contextMap;		
	}
	
	public HashMap getContextMap() {
		if (contextMap != null) {
			return contextMap;
		} else {
			return new HashMap();
		}
	}
}
