package ar.com.sdd.commons.util;

import ar.com.sdd.commons.util.converter.DateToTimestampConverter;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.logging.log4j.*;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * Conjunto de constantes y metodos utiles para el tratamiento de fechas.
 *
 * @author Andres Ferrari
 */
public class DateUtil implements java.io.Serializable {
	
    /**
	 * Representa una fecha minima en formato timestamp.
	 */
	public static final Timestamp LOWER_TIMESTAMP = Timestamp.valueOf("1900-01-01 00:00:00.0");
	/**
	 * Representa una fecha maxima (suficientemente grande) en formato timestamp.
	 */
	public static final Timestamp UPPER_TIMESTAMP = Timestamp.valueOf("3000-01-01 00:00:00.0");

	/**
	 * Para conversion de fechas (HEREDADO DE CRONO).
	 */
	public static final String DATE_FORMAT_STRING = "dd/MM/yyyy";

	private static final String ORACLE_DATE_FORMAT_STRING = "dd/MM/yyyy";

    private static final String DATE_FORMAT_WITH_WEEKDAY_STRING = "EEEE dd/MM/yyyy";

	private static final String HOUR_FORMAT_STRING = "HH:mm:ss";

	private static final String ORACLE_HOUR_FORMAT_STRING = "HH24:MI:SS";
	public static final String ORACLE_DATEHOUR_FORMAT_STRING = ORACLE_DATE_FORMAT_STRING + " " + ORACLE_HOUR_FORMAT_STRING;

	private static final Logger log = LogManager.getLogger(DateUtil.class);	// Crono
	
	public static final String FORMAT_TIMESTAMP = "dd/MM/yyyy HH:mm:ss";
	public static final String FORMAT_TIMESTAMP_NO_SECONDS = "dd/MM/yyyy HH:mm";
	public static final String FORMAT_TIMESTAMP_NOSPACES = "yyyyMMddhhmmss";
	public static final String FORMAT_DATE_NOSPACES = "yyyyMMdd";

	public static final String DIAS_STRING     = "dias";
	public static final String HORAS_STRING    = "horas";
	public static final String HORASACUM_STRING    = "horasAcum";
	public static final String MINUTOS_STRING  = "minutos";
	public static final String MINUTOSACUM_STRING  = "minutosAcum";
	public static final String SEGUNDOS_STRING = "segundos";
	public static final String SEGUNDOSACUM_STRING = "segundosAcum";
	//Swift Date Format
	public static final SimpleDateFormat sdf8601OnlyMinutes = new SimpleDateFormat("yyyy-MM-dd'T'HH:mmZ");
	public static final SimpleDateFormat sdf8601            = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSXXX"); //Z genera TZ en formato -0300, XXX en formato -03:00
	private static final String DATE_JSON_FORMAT_STRING = "yyyy-MM-dd";
	private static final String DATE_JSON_FORMAT_STRING_NODASH = "yyyyMMdd";

	public final static String DRT_ISSUE_DATE_RANGE_DAYS_MTD = "01/MM";
    public final static String DRT_ISSUE_DATE_RANGE_DAYS_YTD = "01/YYYY";
    public final static String DRT_ISSUE_DATE_RANGE_DAYS_MONTHS = "M:";

	/**
	 * Incrementa la fecha en la cantidad especificada de dias.
	 */
	public static Date incrDateDays(Date d, int numDays) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(d);
		calendar.add(Calendar.DAY_OF_MONTH, numDays);
		return calendar.getTime();
	}

    /**
	 * Incrementa la fecha en la cantidad especificada de dias, solo de LaV
	 */
	public static Date incrDateDaysLaV(Date d, int numDays) {

		Calendar calend = Calendar.getInstance();
		calend.setTime(d);
		
		int delta = 1;
		if (numDays < 0) {
			delta = -1;
			numDays = -numDays;
		}
		int dayCounter = 0; // contador de cuantos dias REALMENTE incremente
		while (numDays > dayCounter) {
			calend.add(Calendar.DATE, delta);
			if(!(calend.get(Calendar.DAY_OF_WEEK) == Calendar.SATURDAY) && !(calend.get(Calendar.DAY_OF_WEEK) == Calendar.SUNDAY)) { 
				dayCounter++;
			}
		}	
		return calend.getTime();

	}

    /**
     * Versiones String de lo mismo
     */
    public static Date incrDateDays(String date, String numDays) {
        if (StringUtil.isEmpty(date)) return null;
        Date d = parseDate(date);
        if (StringUtil.isEmpty(numDays)) return d;
        Integer  days = Integer.parseInt(numDays);
        return incrDateDays(d, days);
    }
    public static Date subsDateDays(String date, String numDays) {
        if (StringUtil.isEmpty(date)) return null;
        Date d = parseDate(date);
        if (StringUtil.isEmpty(numDays)) return d;
        Integer  days = Integer.parseInt(numDays);
        return incrDateDays(d, -days);
    }

    /**
	 * Trunca la hora:min:seg de una fecha, dejandola en 0:0:0
	 */
	public static Date truncDateTime(Date d) {

		if (d == null) return null;

		Calendar calendar = Calendar.getInstance();
		calendar.setTime(d);
		calendar.set(Calendar.HOUR_OF_DAY, 0);
		calendar.set(Calendar.MINUTE, 0);
		calendar.set(Calendar.SECOND, 0);
		calendar.set(Calendar.MILLISECOND, 0);
		return calendar.getTime();
	}

	/**
	 * Trunca la min:seg de una fecha, dejandola en hora_original:0:0
	 */
	public static Date truncDateTimeToHour(Date d) {
		if (d == null) return null;

		Calendar calendar = Calendar.getInstance();
		calendar.setTime(d);
		calendar.set(Calendar.MINUTE, 0);
		calendar.set(Calendar.SECOND, 0);
		calendar.set(Calendar.MILLISECOND, 0);
		return calendar.getTime();
	}

	public static Date lastSecondOf(Date d) {

		if (d == null) return null;

		Calendar calendar = Calendar.getInstance();
		calendar.setTime(d);
		calendar.set(Calendar.HOUR_OF_DAY, 23);
		calendar.set(Calendar.MINUTE, 59);
		calendar.set(Calendar.SECOND, 59);
		calendar.set(Calendar.MILLISECOND, 0);
		return calendar.getTime();

	}

	/**
	 * Trunca la hora:min:seg de una fecha, dejandola en 0:0:0
	 */
	public static Date truncDateTime(Timestamp t) {
		if (t == null) return null;
		return truncDateTime(DateToTimestampConverter.convertFrom(t));
	}

	public static Date truncDateToQuarter(Date date) {
	    Calendar cal = Calendar.getInstance();
	    cal.setTime(date);
	    cal.set(Calendar.MONTH, cal.get(Calendar.MONTH)/3 * 3);
	    cal.set(Calendar.DAY_OF_MONTH, 1);
	    return cal.getTime();
	}

	public static Date truncDateToSemester(Date date) {
		Calendar cal = Calendar.getInstance();
		cal.setTime(date);
		cal.set(Calendar.MONTH, cal.get(Calendar.MONTH)/6 * 6);
		cal.set(Calendar.DAY_OF_MONTH, 1);
		return cal.getTime();
	}

	//----HEREDADO DE CRONO-------------------------------------------------------//

	/**
	 * Chequea si un Date es null
	 * @param o
	 * @return true 
	 */
	public static boolean isEmpty(Date o){
		return o==null;
	}
	
	public static String getFormat() { return DATE_FORMAT_STRING; }
	public static String getOracleDateFormat() { return ORACLE_DATE_FORMAT_STRING; }
	public static String getHourFormat() { return HOUR_FORMAT_STRING; }
	public static String getOracleHourFormat() { return ORACLE_HOUR_FORMAT_STRING; }


    public static String getDateFormatWithWeekday() {
        return DATE_FORMAT_WITH_WEEKDAY_STRING;
    }

    /**
	 * Convierte una java.sql.Date a String.
	 */
	static public String dateToString(java.sql.Date d) {
		
		// En este uso no hay problemas de threading, ya que se
		// instancia un nuevo objeto en cada llamada a este metodo.
		SynchronizedSimpleDateFormat formatDate = new SynchronizedSimpleDateFormat(DATE_FORMAT_STRING);

		String strDate = null;
		if (d != null) {
    		strDate = formatDate.format(d);
    	}
    	return strDate;
    }
    
    
   	/**
	 * Convierte una java.util.Date a String.
	 */
	static public String dateToString(Date d) {

		// En este uso no hay problemas de threading, ya que se
		// instancia un nuevo objeto en cada llamada a este metodo.
		SynchronizedSimpleDateFormat formatDate = new SynchronizedSimpleDateFormat(DATE_FORMAT_STRING);
		String strDate = (d != null) ? formatDate.format(d) : null;
    	return strDate;
    }


    static public String dateToString(Date d, String dateFormat) {
	    return dateToString(d, dateFormat, null);
    }

   	/**
	 * Convierte una java.util.Date a String. 
	 * Recibe como parametro el formato del output.
	 */
	static public String dateToString(Date d, String dateFormat, Locale locale) {

		// En este uso no hay problemas de threading, ya que se
		// instancia un nuevo objeto en cada llamada a este metodo.
        SynchronizedSimpleDateFormat df;
        if (locale != null) {
            df = new SynchronizedSimpleDateFormat(dateFormat, locale);
        } else {
            df = new SynchronizedSimpleDateFormat(dateFormat);
        }
		return (d != null) ? df.format(d) : null;
    }

	public static String format(Date date, String format) {
		return dateToString(date, format);
	}

    public static String format(Date date) {
        return dateToString(date, getFormat());
    }

    public static String format(Date date, String format, Locale locale) {
        return dateToString(date, format, locale);
    }

    /**
	 * Parsea una fecha en formato String y devuelve
	 * una java.sql.Date.
	 */
	static public java.sql.Date parseDate(String strDate) {
	    if (StringUtil.isEmpty(strDate)) {
	        return null;
        }
		// reemplazo todos los puntos a barra para contemplar el caso de una fecha del tipo dd.MM.yyyy
		strDate= strDate.replace('.','/').replace('-', '/');
		if (strDate.length()-strDate.lastIndexOf('/')==3) { // Viene xx/xx/xx, la ultima barra esta en la pos 5	
			strDate=strDate.substring(0,strDate.lastIndexOf('/'))+'/'+"20"+strDate.substring(strDate.lastIndexOf('/')+1); 
		}

		// En este uso no hay problemas de threading, ya que se
		// instancia un nuevo objeto en cada llamada a este metodo.
		SynchronizedSimpleDateFormat formatDate = new SynchronizedSimpleDateFormat(DATE_FORMAT_STRING);
		
		java.sql.Date d = null;
		
		try {
			
            Date auxDate = formatDate.parse(strDate);
            d = new java.sql.Date(auxDate.getTime());

		} catch (ParseException pe) {
			log.error("parseDate() - ParseException: " + pe.getMessage());
		} catch (Exception e) {
			log.error("parseDate() - Exception: " + e.getMessage());
		}
				
		return d;
	}

	/**
	 * Parsea una fecha en formato String para devolver un java.util.date
	 * @param strDate string a parsear
	 * @param pattern formato del string, si es null o vacio se usa el parttern por default
	 */
	
	static public Date parseDate(String strDate, String pattern) {
		return parseDate(strDate, pattern, null);
	}
	
	static public Date parseDate(String strDate, String pattern, Locale locale) {
		if (strDate == null || strDate.trim().equals("")) {
			return null;
		}
		if (StringUtil.isEmpty(pattern)) {
			pattern=DATE_FORMAT_STRING;
		}
		
		Date result=null;
		try {
			//Fix para permitir el parsing del string de dias con acento
			if (pattern.equals("E dd/MM/yyyy")) {
				pattern = "dd/MM/yyyy";
				strDate = StringUtil.fromToParser(strDate, "-10:0");
			}
			
			SynchronizedSimpleDateFormat sdf = null;
			if (locale != null) {
				sdf = new SynchronizedSimpleDateFormat(pattern, locale);
			} else {
				sdf = new SynchronizedSimpleDateFormat(pattern);
			}
			
			result= sdf.parse(strDate);
		} catch (ParseException e) {
			log.error("Ha ocurrido un error", e);
		}
		return result;
	}
	
	
	/**
	 * Devuelve un String con una fecha en formato aceptable por un query.
	 */
	static public String toQueryDate(Date d) {
		
		return toQueryDate(new java.sql.Date(d.getTime()));
	}


	/**
	 * Devuelve un String con una fecha en formato aceptable por un query.
	 * NOTA: usa funciones especificas de Oracle.
	 */
	static public String toQueryDate(java.sql.Date d) {
		
		String dateStr = "TO_DATE('" 
						+ DateUtil.dateToString(d) + "', '"
						+ DateUtil.getFormat() + "')";
		
		return dateStr;
	}
	
	
	/**
	 * Dada una fecha, le suma (o resta, dependiendo del signo) una cantidad de dias. 
	 * El comportamiento es similar a Calendar.add().
	 *
	 * NOTA: en el caso de saltearse fines de semana, la cantidad de dias desplazados
	 * puede ser mayor que la pasada como parametro.
	 *
	 * @param d Fecha a la cual se le sumaran los dias.
	 * @param days Cantidad de dias a sumar.
	 * @param skipSaturday  Si es true, se saltea sabado
	 * @param skipSunday  Si es tru, se saltea el domingo
	 */

	
	static public Date add(Date d, int days, boolean skipSaturday, boolean skipSunday) {
		
		Calendar calend = Calendar.getInstance();
		calend.setTime(d);
		
		if (skipSaturday || skipSunday) {
			
			int delta = 1;
			if (days < 0) {
				delta = -1;
				days = -days;
			}

			while (days > 0) {
				calend.add(Calendar.DATE, delta);
				
				// Solo decrece la cuenta si no es un fin de semana
				if (( !skipSaturday || calend.get(Calendar.DAY_OF_WEEK) != Calendar.SATURDAY ) &&
					( !skipSunday || calend.get(Calendar.DAY_OF_WEEK) != Calendar.SUNDAY) ) {
					days--;
				}
			}	
			
		} else 
			calend.add(Calendar.DATE, days);
		
		Date newDate = calend.getTime();
		
		return newDate;
	}

	/**
	 * Dada una fecha, le suma (o resta, dependiendo del signo) una cantidad de dias. 
	 * El comportamiento es similar a Calendar.add().
	 *
	 * NOTA: en el caso de saltearse fines de semana, la cantidad de dias desplazados
	 * puede ser mayor que la pasada como parametro.
	 *
	 * @param d Fecha a la cual se le sumaran los dias.
	 * @param days Cantidad de dias a sumar.
	 * @param skipWeekends Si es true, se saltea los fines de semana.
	 */
	static public Date add(Date d, int days, boolean skipWeekends) {
		return add( d, days, skipWeekends, skipWeekends );
	}
	
	 static public Date add(Date d, int days) {
		return add( d, days, false, false );
	}

	public static Date addMonth(Date d, int months) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(d);
        calendar.add(Calendar.MONTH, months);
        return calendar.getTime();
    }
	 
	public static Date addYear(Date d, int years) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(d);
        calendar.add(Calendar.YEAR, years);
        return calendar.getTime();
    }

	static public Date addHours(Date d, int hours) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(d);
		calendar.add(Calendar.HOUR_OF_DAY, hours);
		return calendar.getTime();
	}

	 static public Date addMinutes(Date d, int minutes) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(d);
		calendar.add(Calendar.MINUTE, minutes);
		return calendar.getTime();
	}

	static public Date addSeconds(Date d, int seconds) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(d);
        calendar.add(Calendar.SECOND, seconds);
        return calendar.getTime();
    }

    static public int daysDiference(Date d1, Date d2){
    	if (d1 == null || d2 == null) {
    		throw new ApplicationException("Alguna de las fechas es nula d1=" + d1 + " d2="+d2);
    	}
    	long fechaInicialMs = d1.getTime();
    	long fechaFinalMs = d2.getTime();
    	long diferencia =   fechaInicialMs - fechaFinalMs;
    	double dias = Math.floor(diferencia / (1000 * 60 * 60 * 24));
    	return ( (int) dias);
       
    }
    static public int daysBetween(Date d1, Date d2,  boolean skipSaturday, boolean skipSunday ) {
        if ( d1.after( d2 ) )
            return -daysBetween( d2, d1, skipSaturday, skipSunday );

		int result = 0;
		if (true) {
			if (skipSunday || skipSaturday) {
				//Esta version 'cuenta los dias'
				while (d1.before(d2)) {
					d1 = add(d1, 1, skipSaturday, skipSunday);
					result++;
				}
			} else {
				//no tengo que esquivar nada, hago la cuenta
				result =  daysDiference(d2, d1);
			}
		} else {
			//no anda! Buscar una buena forma de hacer esto:
			int days = daysDiference(d2, d1);
			if (skipSaturday && skipSunday) {
				result = 1 + (days * 5 - (d1.getDay() - d2.getDay()) * 2) / 7;

				if (d2.getTime() == 6) result--;
				if (d1.getTime() == 0) result--;
			} else if (skipSunday || skipSaturday) {

			} else {
				result = days;
			}

/*
			//Esta version ' calcula los dias'
			int days = daysDiference(d2, d1);

			//semanas completas: 0 si son de la misma semana
			int fullWeeks = days / 7;
			if (fullWeeks==0) {
				int week1 = days;
				int sat1  = d2.getDay() == 6 ? 1 : 0;
				int sun1  = d1.getDay() == 0 ? 1 : 0;
				result =  week1
						- (skipSaturday?sat1: 0)
						- (skipSunday?sun1:0)
				;

			} else {
				//dias de la semana1, mas dias de la semana ultima
				//day ranges from 0 SUnday to 6 Saturday
				//  En la semana inicial tengo que sumar dias entre el d y el 5:
				//    O sea, esta funcion:  0->5, 1->5, 2->4, 3->3, 4->2 5->1 6->0
				//	En la final, quiero:   0->0 , 1->0, 2->1, 3->2, 4->3, 5->4, 6->0
				int week1 = d1.getDay() == 0 ? 5 : (6 - d1.getDay());
				int sat1  = 1;
				int sun1  = d1.getDay() == 0 ? 1 : 0;
				int weekn = (d2.getDay() == 0||d2.getDay() == 6) ? 0 : (d2.getDay()-1);
				int satn  = 0;
				int sunn  = d2.getDay() == 0 ? 0 : 1;
				result = fullWeeks * 5 + week1 + weekn
						+ (skipSaturday?0: (fullWeeks+sat1+satn))
						+ (skipSunday?0: (fullWeeks+sun1+sunn))
				;

			}
*/
		}

        return result;
    }

	/*
	*  Dias transcurridos entre ambas fechas:  d2-d1
	*/
    static public int daysBetween(Date d1, Date d2 ) {
        return daysBetween( d1, d2,  false, false );
    }

	/*
	 *  Dias transcurridos entre ambas fechas:  d2-d1, solo si d2 es posterior a d1
	 */
	static public int daysDue(Date d1, Date d2 ) {
		if ( d1.after( d2 ) ) return 0;
		else return daysBetween( d1, d2,  false, false );
	}


	/**
	* convierte un date de un timezone a otro.
	* @param source objeto Date a convertir
	* @param sourceTzId timezone de origen expresado como offset de GMT. Ejemplo GMT-4    
	* @param sourceTzId timezone de destino expresado como offset de GMT. 
	*/
	public static Date applyTimeZone(Date source, String sourceTzId,String targetTzId){	
		
		Calendar sourceCal= Calendar.getInstance(TimeZone.getTimeZone(sourceTzId));
		GregorianCalendar targetCal = new GregorianCalendar(TimeZone.getTimeZone(targetTzId));
		targetCal.setTime(source);
		long offsetDif = sourceCal.getTimeZone().getRawOffset() - targetCal.getTimeZone().getRawOffset();
		targetCal.setTimeInMillis(targetCal.getTimeInMillis()-offsetDif);
		return targetCal.getTime();
		
	}
	
	/**
	* devuelve un date formateado como un string y aplicado a un cierto timezone
	* @param source objeto Date a convertir
	* @param sourceTzId timezone de origen expresado como offset de GMT. Ejemplo GMT-4    
	* @param sourceTzId timezone de destino expresado como offset de GMT.
	* @param pattern formato del string
	*/
	
	public static String formatByTimeZone(Date source, String sourceTzId,String targetTzId,String pattern){
		
		Date result = applyTimeZone(source,sourceTzId,targetTzId);
		return dateToString(result,pattern);
		
	}
	
	/**
	* devuelve un date formateado como un string y aplicado a un cierto timezone
	* @param String que representa un momento del tiempo
	* @param sourceTzId timezone de origen expresado como offset de GMT. Ejemplo GMT-4    
	* @param sourceTzId timezone de destino expresado como offset de GMT.
	* @param pattern formato del string
	*/
	
	public static String formatByTimeZone(String source, String sourceTzId,String targetTzId,String pattern){
		
		Date result = applyTimeZone(parseDate(source,pattern),sourceTzId,targetTzId);
		return dateToString(result,pattern);
		
	}
	
	
	/**
	 * Retorna un Map con la diferencia entre dos fechas.
	 * @param fechaMayor
	 * @param fechaMenor
	 * @return
	 * @throws Exception
	 */
	public static Map<String, Long> getDiferencia(Date fechaMayor, Date fechaMenor) {
		Map<String, Long> resultadoMap = new HashMap<>();
		long diferenciaMils;
		long segundos;
		long minutos;
		long horas;
		long dias = 0;
		
		//Verificamos que fechaActual sea mayor.
		if (fechaMayor.before(fechaMenor)) {
		    //Los doy vuelta
		    var temp = new Date(fechaMenor.getTime());
		    fechaMenor = new Date(fechaMayor.getTime());
		    fechaMayor = new Date(temp.getTime());
        }

        // los milisegundos
        diferenciaMils = fechaMayor.getTime() - fechaMenor.getTime();

        // obtenemos los segundos
        segundos = diferenciaMils / 1000;
		resultadoMap.put(SEGUNDOSACUM_STRING, Long.valueOf(segundos));

		minutos  =  segundos / 60;
		resultadoMap.put(MINUTOSACUM_STRING, Long.valueOf(minutos));

		// obtenemos las horas
        horas    = segundos / 3600;
		resultadoMap.put(HORASACUM_STRING, Long.valueOf(horas));

        // restamos las horas para continuar con segundos remanentes en la hora
        segundos -= horas * 3600;

        // igual que el paso anterior, para quedarme con minutos remanentes y segundos remantens
        minutos  =  segundos / 60;
        segundos -= minutos * 60;

        //Si transcurrio mas de 24 hs calculamos los dias.
        if (horas > 24) {
            dias   =  (int) Math.floor(horas/24);
            horas -= dias * 24;
        }

        // ponemos los resultados en un mapa.
		resultadoMap.put(DIAS_STRING,Long.valueOf(dias));

        resultadoMap.put(HORAS_STRING,    Long.valueOf(horas));
        resultadoMap.put(MINUTOS_STRING,  Long.valueOf(minutos));
        resultadoMap.put(SEGUNDOS_STRING, Long.valueOf(segundos));

		return resultadoMap;	
	}

	//Esto estaba retornarn las diferencia de la parte de minutos, no el acumuldo en minutos: si la diferencia era 2hs:45, devolvia 45
	//Se modifica al acumulado, expresado en minutos
    public static Long getDiferenciaMinutos(Date fechaMayor, Date fechaMenor) {
	    return getDiferencia(fechaMayor, fechaMenor).get(MINUTOSACUM_STRING);
    }

	//Esto estaba retornarn las diferencia de la parte de segundos, no el acumuldo en minutos: si la diferencia era 2hs:45, devolvia 45
	//Se modifica al acumulado, expresado en segundos
    public static Long getDiferenciaSegundos(Date fechaMenor) {
	    return getDiferencia(new Date(), fechaMenor).get(SEGUNDOSACUM_STRING);
    }

    public static Long getElapsedMillis(Date fechaReferencia) {
        return  (now().getTime() - fechaReferencia.getTime());
    }
    public static Long getElapsedSegundos(Date fechaReferencia) {
        return  getElapsedMillis(fechaReferencia)/1000;
    }
    public static Long getElapsedMinutos(Date fechaReferencia) {
        return  getElapsedMillis(fechaReferencia)/1000/60;
    }

    public static Date getPrimerDiaDelMes(Date date) {
		 return getPrimerDiaDelMes(date,false);
	}
	public static Date getPrimerDiaDelMes(Date date, boolean onlyHabil) {
		Calendar cal = GregorianCalendar.getInstance();
		cal.setTime(truncDateTime(date));
		cal.set((Calendar.DAY_OF_MONTH),1);
		cal.set(Calendar.HOUR_OF_DAY, 0); // Trunco el resto, por las dudas
		cal.set(Calendar.MINUTE, 0);
		cal.set(Calendar.SECOND, 0);
		cal.set(Calendar.MILLISECOND, 0);
		if (onlyHabil) {
			int dow = cal.get(Calendar.DAY_OF_WEEK);
			if(dow==Calendar.SATURDAY) cal.set(Calendar.DAY_OF_MONTH,3);     //1 es sabado ->3 es lunes
			else if (dow==Calendar.SUNDAY) cal.set(Calendar.DAY_OF_MONTH,2); //1 es domingo->2 es lunes
		}
		return cal.getTime();
	}

	public static Date getPrimerDiaDelMesActual() {
		 return getPrimerDiaDelMes(new Date());
	}
	public static Date getPrimerDiaDelMesActual(boolean onlyHabil) {
		return getPrimerDiaDelMes(new Date(),onlyHabil);
	}

    public static Date getPrimerDiaDelMes(int cantMeses) {
        return getPrimerDiaDelMes(addMonth(new Date(), cantMeses));
    }
	public static Date getPrimerDiaDelMes(int cantMeses,boolean onlyHabil) {
		return getPrimerDiaDelMes(addMonth(new Date(), cantMeses),onlyHabil);
	}

	public static Date getUltimoDiaDelMes(Date date) {
		Calendar cal = GregorianCalendar.getInstance();
		cal.setTime(date);
		cal.set(cal.get(Calendar.YEAR),cal.get(Calendar.MONTH),cal.getActualMaximum(Calendar.DAY_OF_MONTH));
		cal.set(Calendar.HOUR_OF_DAY, 0); // Trunco el resto, por las dudas
		cal.set(Calendar.MINUTE, 0);
		cal.set(Calendar.SECOND, 0);
		cal.set(Calendar.MILLISECOND, 0);
		return cal.getTime();
	}

    public static Date getUltimoDiaDelMesActual() {
	    return getUltimoDiaDelMes(new Date());
    }

    public static Date getUltimoDiaDelMes(int cantMeses) {
        return getUltimoDiaDelMes(addMonth(new Date(), cantMeses));
    }

	public static Date getUltimoDiaDelMesPrevio(Date date) {
		Calendar cal = GregorianCalendar.getInstance();
		cal.setTime(date);

		cal.add(Calendar.MONTH, -1);
		cal.set(Calendar.DAY_OF_MONTH, cal.getActualMaximum(Calendar.DAY_OF_MONTH));

		cal.set(Calendar.HOUR_OF_DAY, 0); // Trunco el resto, por las dudas
		cal.set(Calendar.MINUTE, 0);
		cal.set(Calendar.SECOND, 0);
		cal.set(Calendar.MILLISECOND, 0);
		return cal.getTime();
	}

	public static Date getUltimoSegundoDelDia() {

		Calendar calendar = Calendar.getInstance();
		calendar.setTime(DateUtil.today());
		//Uso 23:59 para no confundir las 00:00 de hoy con las de ayer. Despues le sumo 1 minuto al final
		calendar.set(Calendar.HOUR_OF_DAY, 23);
		calendar.set(Calendar.MINUTE, 59);
		calendar.set(Calendar.SECOND, 0);
		calendar.set(Calendar.MILLISECOND, 0);
		return calendar.getTime();
	}

	public static Date getUltimoDiaDelMesPrevio() {
		return getUltimoDiaDelMesPrevio(new Date());
	}

	public static int getMes(Date date){
		Calendar cal = Calendar.getInstance();
		if (date!=null)
			cal.setTime(date);
		return cal.get(Calendar.MONTH);
	}
	public static int getMesActual(){
		Calendar cal = Calendar.getInstance();
		return cal.get(Calendar.MONTH);
	}
	public static int getYearActual(){
		Calendar cal = Calendar.getInstance();
		return cal.get(Calendar.YEAR);
	}

	public static Date getPrimerDiaDelAnio(Date date) {
        Calendar cal = GregorianCalendar.getInstance();
        cal.setTime(truncDateTime(date));
        cal.set(Calendar.DAY_OF_MONTH, 1);
        cal.set(Calendar.MONTH, 0);
        cal.set(Calendar.HOUR_OF_DAY, 0); // Trunco el resto, por las dudas
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.SECOND, 0);
        cal.set(Calendar.MILLISECOND, 0);
        return cal.getTime();
    }
	public static Date getUltimoDiaDelAnio(Date date) {
		Calendar cal = GregorianCalendar.getInstance();
		cal.setTime(truncDateTime(date));
		cal.set(Calendar.DAY_OF_MONTH, 31);
		cal.set(Calendar.MONTH, 11);
		cal.set(Calendar.HOUR_OF_DAY, 0); // Trunco el resto, por las dudas
		cal.set(Calendar.MINUTE, 0);
		cal.set(Calendar.SECOND, 0);
		cal.set(Calendar.MILLISECOND, 0);
		return cal.getTime();
	}

    public static Date getPrimerDiaDelAnioActual() {
        return getPrimerDiaDelAnio(new Date());
    }
	public static Date getUltimoDiaDelAnioActual() {
		return getUltimoDiaDelAnio(new Date());
	}
	
	/**
	 * Compara dos Dates, considerandolos null o vacio 'menor' que el resto
	 * @param s1 Primer Date
	 * @param s2 Segundo Date
	 */
	public static int safeCompareTo(Date d1 , Date d2) {
		if (d1==null) {
			return 1;
		}
		else if (d2==null) {
			return -1;
		} else {
			return d1.compareTo(d2);
		}
	}

    public static boolean between(Date dateToEval, Date dateFrom, Date dateTo) {
        return after(dateToEval, dateFrom) && before(dateToEval, dateTo);
    }

    public static boolean betweenOrEquals(Date dateToEval, Date dateFrom, Date dateTo) {
        return afterOrEquals(dateToEval, dateFrom) && beforeOrEquals(dateToEval, dateTo);
    }
	
	/**
	 * reemplaza al before de Date. 
	 * si fecha1 es anterior a fecha2 devuelve TRUE, sino devuelve FALSE.
	 * si alguna fecha pasada por parametros es null, revuelve false.
	 * @param fecha1
	 * @param fecha2
	 * @return
	 */
	public static boolean before(Date fecha1, Date fecha2){
		boolean result = false;
		  if (fecha1 != null && fecha2 != null){
			  result = fecha1.before(fecha2);
		  } 		
		return result;
	}

	public static boolean before(Date fecha1, String fecha2_DATE_FORMAT_STRING){
        return before(fecha1, parseDate(fecha2_DATE_FORMAT_STRING));
    }

	public static boolean beforeOrEquals(Date fecha1, Date fecha2){
		boolean result = false;
		  if (fecha1 != null && fecha2 != null){
			  result = fecha1.before(fecha2) || fecha1.equals(fecha2);
		  } 		
		return result;
	}

    public static boolean beforeOrEquals(Date fecha1, String fecha2_DATE_FORMAT_STRING){
        return beforeOrEquals(fecha1, parseDate(fecha2_DATE_FORMAT_STRING));
	}

    public static boolean equals(Date fecha1, Date fecha2) {
        boolean result = false;
        if (fecha1 != null && fecha2 != null) {
            result = DateUtils.isSameDay(fecha1, fecha2);
        }
        return result;
    }

	/*
	*    fecha1 > fecha2 => true
	 *    fecha1 =null    => false
	 *    fecha2 =null    => false (ojo)
	*/
	public static boolean after(Date fecha1, Date fecha2){
        boolean result = false;
        if (fecha1 != null && fecha2 != null){
            result = fecha1.after(fecha2);
        }
        return result; //una o ambas son null
    }
	/*
	 *    fecha1 > fecha2 => true
	 *    fecha1 =null    => false
	 *    fecha2 =null    => true
	 */

	public static boolean afterN(Date fecha1, Date fecha2){
		boolean result = false;
		if (fecha1 != null && fecha2 != null){
			result = fecha1.after(fecha2);
		} else if (fecha1!=null) {
			return true;
		}
		return result; //ambas son null
	}
    public static boolean after(Date fecha1, String fecha2_DATE_FORMAT_STRING){
        return after(fecha1, parseDate(fecha2_DATE_FORMAT_STRING));
    }

    public static boolean afterOrEquals(Date fecha1, Date fecha2){
		boolean result = false;
		  if (fecha1 != null && fecha2 != null){
			  result = fecha1.after(fecha2) || fecha1.equals(fecha2);
		  } 		
		return result;
	}

    public static boolean afterOrEquals(Date fecha1, String fecha2_DATE_FORMAT_STRING){
        return afterOrEquals(fecha1, parseDate(fecha2_DATE_FORMAT_STRING));
    }

	
	public static Date menorFecha(Date... fechas) {
		Date menor = null;
		if (fechas != null) {
			for (Date fecha : fechas) {
				if(fecha != null && (menor==null || menor.after(fecha))){
					menor=fecha;
				}
			}
		}
		return menor;
	}
	public static Date notAfterToday(Date  fecha) {
		return menorFecha(fecha, today());
	}
	public static Date notBeforeToday(Date  fecha) {
		return mayorFecha(fecha, today());
	}

	public static Date mayorFecha(Date... fechas) {
		Date mayor = null;
		if (fechas != null) {
			for (Date fecha : fechas) {
				if(fecha != null && (mayor==null || mayor.before(fecha))){
					mayor=fecha;
				}
			}
		}
		return mayor;
	}

	public static Date nvl(Date... fechas) {
	    if (fechas != null) {
	        for (Date fecha : fechas) {
	            if (fecha != null) {
	                return fecha;
                }
            }
        }

	    return null;
    }

	public static String secondsToTime(long seconds) {
		return millisecondsToTime(seconds * 1000);
	}
	
	public static String millisecondsToTime( long milliseconds ) {
		final SynchronizedSimpleDateFormat sdf = new SynchronizedSimpleDateFormat(HOUR_FORMAT_STRING);
	    try {
			Date baseTime = sdf.parse( "00:00:00" );
		    Date time = new Date( baseTime.getTime() + milliseconds );
		    return sdf.format( time );
	    } catch ( ParseException e ) {
	        return "00:00:00";
	    }
	}
	
	public static String getTimeDifferenceBucket(Date startDate, Date endDate) {
		if (startDate == null || endDate == null) {
			return null;
		}
		long diffMillis = Math.abs(endDate.getTime() - startDate.getTime());
		
		//Menos de 60 min, devuelvo minutos
		if (diffMillis < 60 * 60 * 1000) {
			long mi=TimeUnit.MILLISECONDS.toMinutes(diffMillis); 
			return mi + " min.";
		//Menos de 24 horas, devuelvo horas
		} else if (diffMillis < 24 * 60 * 60 * 1000) {
			long hs=TimeUnit.MILLISECONDS.toHours(diffMillis);
			return hs + (hs==1?"h.":" hs.");
		//Menos de una semana, devuelvo dias
		} else if (diffMillis < 7 * 24 * 60 * 60 * 1000) {
			long dd=TimeUnit.MILLISECONDS.toDays(diffMillis);
			return dd + (dd==1?" dia":" dias");
		//Mas de una semana, devuelvo la fecha inicio
		} else {
			SynchronizedSimpleDateFormat formatDate = new SynchronizedSimpleDateFormat(DATE_FORMAT_STRING);
			return formatDate.format(startDate);
		}
	}
	
	public static Date buildDate(int year, int month, int date) {
		Calendar cal = GregorianCalendar.getInstance();
		cal.set(year, month, date, 0, 0, 0);
		return cal.getTime();
	}

	public static Date someFutureDate() {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(new Date());
		calendar.set(Calendar.DAY_OF_MONTH,31);
		calendar.set(Calendar.MONTH,11);
		calendar.add(Calendar.YEAR, 1);
		calendar.set(Calendar.HOUR_OF_DAY, 0);
		calendar.set(Calendar.MINUTE, 0);
		calendar.set(Calendar.SECOND, 0);
		calendar.set(Calendar.MILLISECOND, 0);
		return calendar.getTime();
	}
	public static Date someFarFutureDate() {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(new Date());
		calendar.set(Calendar.DAY_OF_MONTH,31);
		calendar.set(Calendar.MONTH,11);
		calendar.add(Calendar.YEAR, 10);
		calendar.set(Calendar.HOUR_OF_DAY, 0);
		calendar.set(Calendar.MINUTE, 0);
		calendar.set(Calendar.SECOND, 0);
		calendar.set(Calendar.MILLISECOND, 0);
		return calendar.getTime();
	}
	public static Date someFarPastDate() {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(new Date());
		calendar.set(Calendar.DAY_OF_MONTH,1);
		calendar.set(Calendar.MONTH,0);
		calendar.add(Calendar.YEAR, -10);
		calendar.set(Calendar.HOUR_OF_DAY, 0);
		calendar.set(Calendar.MINUTE, 0);
		calendar.set(Calendar.SECOND, 0);
		calendar.set(Calendar.MILLISECOND, 0);
		return calendar.getTime();
	}


	/**
	* devueld la dueDate si es anterior a hoy, y sino el primer dia del proximo mes
	*/

    public static Date dueDateOrNextMonth(Date dueDate) {

	    if (dueDate==null || before(dueDate, today())) {
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(today());
            calendar.add(Calendar.MONTH, 1);
            return calendar.getTime();
        } else {
			//Le quito a la dueDate la informacion de la hora
			return truncDateTime(dueDate);
        }
    }

	public static Date yesterday() {
		return truncDateTime(add(new Date(), -1));
	}
    public static Date today() {
        return truncDateTime(new Date());
    }
	public static Date tomorrow() {
		return truncDateTime(add(new Date(),1));
	}
    public static Date now() {
        return new Date();
    }

    public static Date parseDateSimple(String date) {
        return parseDateSimple(date, DATE_FORMAT_STRING);
    }

    public static Date parseDateSimple(String date, String format) {
        SynchronizedSimpleDateFormat sdf = new SynchronizedSimpleDateFormat(format);
        Date result = null;
        try {
            result = sdf.parse(date);
        } catch (ParseException e) {
            log.error("Error al parsear la fecha [" + date + "]", e);
        }
        return result;
    }

    public static String formatDateSimple(Date date) {
	    return format(date, DATE_FORMAT_STRING);
    }


	public static Date iso8601OnlyMinutesToDate(final String iso8601string) throws ParseException {
		String s = iso8601string.replace("Z", "+0000");
		return sdf8601OnlyMinutes.parse(s);
	}
	public static Date iso8601OnlyMinutesToDate(final String iso8601string, String gmtHHshift) throws ParseException {
		String s = iso8601string.replace("Z", gmtHHshift+"00");
		return sdf8601OnlyMinutes.parse(s);
	}

	public static boolean isValidDate(String date) {
		return isValidDate(date, DATE_FORMAT_STRING);
	}

	public static boolean isValidDate(String date, String format) {
		SimpleDateFormat sdf = new SimpleDateFormat(format);
		sdf.setLenient(false); // Para que sea estrictamente del formato especificado
		try {
			sdf.parse(date);
		} catch (ParseException e) {
			return false;
		}
		return true;
	}

    public static Date getValidDate(String date) {
	    return getValidDate(date, DATE_FORMAT_STRING);
    }

    public static Date getValidDate(String date, String format) {
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        sdf.setLenient(false); // Para que sea estrictamente del formato especificado
        try {
            return sdf.parse(date);
        } catch (ParseException e) {
            return null;
        }
    }

    public static String formatJsonDate(Date date) {
		return format(date, DATE_JSON_FORMAT_STRING);
	}

	//Manejo algun formato mas, para los despistados que no lo manden correctamente
	public static Date parseJsonDate(String date) {
		if (date==null) return null;
		if (date.length()==8) return parseDate(date,DATE_JSON_FORMAT_STRING_NODASH);
		else return parseDate(date, DATE_JSON_FORMAT_STRING);
	}

	public static int monthsBetweenDates(Date startDate, Date endDate) {
		Calendar start = Calendar.getInstance();
		start.setTime(startDate);

		Calendar end = Calendar.getInstance();
		end.setTime(endDate);

		int monthsBetween = 0;
		int dateDiff = end.get(Calendar.DAY_OF_MONTH) - start.get(Calendar.DAY_OF_MONTH);

		if (dateDiff < 0) {
			int borrrow = end.getActualMaximum(Calendar.DAY_OF_MONTH);
			dateDiff = (end.get(Calendar.DAY_OF_MONTH) + borrrow) - start.get(Calendar.DAY_OF_MONTH);
			monthsBetween--;

			if (dateDiff > 0) {
				monthsBetween++;
			}
		} else {
			monthsBetween++;
		}
		monthsBetween += end.get(Calendar.MONTH)-start.get(Calendar.MONTH);
		monthsBetween  += (end.get(Calendar.YEAR)-start.get(Calendar.YEAR))*12;
		return monthsBetween;
	}

	public static int yearsBetweenDates(Date startDate, Date endDate) {
		Calendar start = Calendar.getInstance();
		start.setTime(startDate);

		Calendar end = Calendar.getInstance();
		end.setTime(endDate);

		int diff = end.get(Calendar.YEAR) - start.get(Calendar.YEAR);
		if (start.get(Calendar.MONTH) > end.get(Calendar.MONTH) || (start.get(Calendar.MONTH) == end.get(Calendar.MONTH) && start.get(Calendar.DATE) > end.get(Calendar.DATE))) {
			diff--;
		}

		return diff;
	}

    public static LocalDate parseToLocalDate(Date dateToConvert) {
        return dateToConvert.toInstant()
                .atZone(ZoneId.systemDefault())
                .toLocalDate();
    }

    public LocalDate parseToLocalDateViaSqlDate(Date dateToConvert) {
        return new java.sql.Date(dateToConvert.getTime()).toLocalDate();
    }

    public static String formatLocalDate(LocalDate localDate, String pattern) {
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern(pattern);
        return localDate.format(formatter);
    }

    public static String formatLocalDate(LocalDate localDate) {
	    return formatLocalDate(localDate, DATE_FORMAT_STRING);
    }

    public static LocalDateTime toLocalDateTime(Date dateToConvert) {
	   return  dateToConvert.toInstant().atZone(ZoneId.systemDefault())
                .toLocalDateTime();
    }

    public static Date convertLocalDateToDate(LocalDate dateToConvert) {
        return Date.from(dateToConvert.atStartOfDay(ZoneId.systemDefault()).toInstant());
    }

	/**
	 * Devuelve una lista de fechas desde/hasta, separadas por batchSizeDays dias
	 * Ej:
	 *   Se pide con fromDate 1/1/1997, dueDate 31/12/1999 y batchSizeDays 365
	 *   Se devuelven tres buckets de un año cada uno [(1/1/1997,31/12/1997) ; (1/1/1998,31/12/1998) ; (1/1/1999,31/12/1999]
	 *
	 * @param fromDate
	 * @param toDate
	 * @param batchSize
	 * @return
	 */
	public static List<Pair<Date, Date>> getDateInBatches(Date fromDate, Date toDate, Integer batchSizeDays) {
		List<Pair<Date, Date>> result = new ArrayList<>();

		//Computo la cantidad de dias que se pide para ver si la tengo que separar en batches
		if (batchSizeDays != null && batchSizeDays > 0 && fromDate != null && toDate != null) {
			int daysBetween = DateUtil.daysBetween(fromDate, toDate);
			if (daysBetween > batchSizeDays) {

				Date startDate = new Date(fromDate.getTime());

				while(daysBetween > batchSizeDays) {
					result.add(new ImmutablePair<>(startDate, DateUtils.addDays(startDate, batchSizeDays - 1)));

					daysBetween -= batchSizeDays;
					startDate = DateUtils.addDays(startDate, batchSizeDays);
				}
				//Agrego los que me quedaron
				result.add(new ImmutablePair<>(startDate, toDate));

			} else {
				//Si el batchSize es menor a los dias transcurridos, devuelvo todo el rango
				result.add(new ImmutablePair<>(fromDate, toDate));
			}

		} else {
			//Si no hay batchSize o falta alguna fecha, devuelvo todo el rango
			result.add(new ImmutablePair<>(fromDate, toDate));
		}

		return result;
	}

    /**
     * Devuelve el mes convertido en letra, donde A es Enero, B es Febrero, etc.
     *
     * @param date
     * @return
     */
	public static String getMesComoLetra(Date date) {
	    if (date == null) {
	        return "";
        }
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        int month = calendar.get(Calendar.MONTH);
        return Character.toString(((int)'A' + month));

    }

    /**
     * Devuelve el mes de Enero a Septiembre como numero 1 a 9
     * y Octubre, Noviembre y Diciembre como letra A, B y C respectivamente
     *
     * @param date
     * @return
     */
	public static String getMesComoNumeroYLetra(Date date) {
	    if (date == null) {
	        return "";
        }
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        int month = calendar.get(Calendar.MONTH) +  1;
        if (month <= 9) {
            return month + "";
        } else {
            return Character.toString(((int)'A' + (month - 10)));
        }
    }

    /**
     * Compara si la hora actual es posterior a la indicada
     *
     * @param time Texto con formato hh:mm:ss
     * @return
     */
    public static boolean nowIsAfter(String time) {
        return !StringUtil.isEmptyNull(time) && LocalTime.now().isAfter(LocalTime.parse(time));
    }

    /**
     * Compara si la hora actual es anterior a la indicada
     *
     * @param time Texto con formato hh:mm:ss
     * @return
     */
    public static boolean nowIsBefore(String time) {
        return !StringUtil.isEmptyNull(time) && LocalTime.now().isBefore(LocalTime.parse(time));
    }

    public static Date fixToPeriodoSAP(Date date) {
        return fixToPeriodoSAP(date, true);
    }

    public static Date fixToPeriodoSAP(Date date, boolean useOnlyDiasHabiles) {
        if (date == null) {
            return null;
        }

        Date today = today();
        Calendar calendar = Calendar.getInstance();
        int currentHour = calendar.get(Calendar.HOUR_OF_DAY);

        if (DateUtil.afterOrEquals(getPrimerDiaDelMes(date, false), getPrimerDiaDelMesActual(false))) {
            // Si el mes de la fecha de contabilizaciones es de este mes o posterior, mando esa fecha de contabilizacion
            return date;
        } else if (DateUtil.beforeOrEquals(today, getPrimerDiaDelMesActual(useOnlyDiasHabiles)) && currentHour < 12) {
            // Ahora bien, aca estoy exportando un doc que se contabilizo antes del mes actual, entonces si hoy es el primer dia del mes o anterior, exporto con fecha del mes anterior porque SAP aun no cerro periodo
            // Y además, solo si la hora actual es antes de las 12 del mediodía
            return getUltimoDiaDelMesPrevio(new Date());
        } else {
            // Sino, devuelvo el primer dia del mes actual
            return getPrimerDiaDelMesActual(useOnlyDiasHabiles);
        }
    }
}