package ar.com.sdd.commons.util;

import ar.com.sdd.commons.util.id.Id;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.NumberFormat;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Locale;

/**
 * Utilidades varias para numeros
 * @author eviera@sdd
 */
public class NumberUtil {
	
	public static double CASI_CERO = 1E-6;
	public static final int BIGDECIMAL_SCALE     = 2;
	public static final int BIGDECIMAL_SCALETC   = 5;
	public static final BigDecimal ZERO_SCALED   = BigDecimal.ZERO.setScale(NumberUtil.BIGDECIMAL_SCALE);
	public static final BigDecimal ZERO_SCALEDTC = BigDecimal.ZERO.setScale(NumberUtil.BIGDECIMAL_SCALETC);
	public static final BigDecimal ONE_SCALED    = BigDecimal.ONE.setScale(NumberUtil.BIGDECIMAL_SCALE);
	public static final BigDecimal ONE_SCALEDTC  = BigDecimal.ONE.setScale(NumberUtil.BIGDECIMAL_SCALETC);
	public static final BigDecimal MINUS_ONE_SCALED = BigDecimal.ONE.setScale(NumberUtil.BIGDECIMAL_SCALE).negate();
	public static final BigDecimal CIEN_SCALED = newBigDecimal("100");

	//Evito que se instancie
	public NumberUtil() {}

	/**
	* Compara dos float para determinar si son igules 
	*/
	public static boolean floatIgual (float a, float b) {
		//El == puede fallar, pero es mas rapido. Si son ==, me alcanza:
		if (a==b) return true;
		NumberParser npa = new NumberParser (a,100);
		NumberParser npb = new NumberParser (b,100);
		return npa.equals(npb);
	}

	/**
	* Compara dos float para determinar si uno es mayor que otro 
	*/
	public static boolean floatMayor(float a, float b) {
		return !floatIgual(a,b) && (a>b);
	}
	/**
	* Compara dos float para determinar si uno es mayor o igual que otro 
	*/
	public static boolean floatMayorOIgual(float a, float b) {
		return floatIgual(a,b) || (a>b);
	}
	/**
	* Compara dos float para determinar si uno es menor que otro 
	*/
	public static boolean floatMenor(float a, float b) {
		return !floatIgual(a,b) && (a<b);
	}
	/**
	* Compara dos float para determinar si uno es menor o igual que otro 
	*/
	public static boolean floatMenorOIgual(float a, float b) {
		return floatIgual(a,b) || (a<b);
	}

	/**
	* Compara dos double para determinar si son igules 
	*/
	public static boolean doubleIgual (double a, double b) {
		//El == puede fallar, pero es mas rapido. Si son ==, me alcanza:
		if (a==b) return true;
		//Sino, veamos si son suficientemente parecidos:
		NumberParser npa = new NumberParser (a,100);
		NumberParser npb = new NumberParser (b,100);
		return npa.equals(npb);
	}

    public static boolean doubleDistinto (double a, double b) {
	    return !doubleIgual(a, b);
    }

	public static boolean isZero(double a) {
	    return doubleIgual(a, 0.0);
    }

    public static boolean isNotZero(double a) {
        return !doubleIgual(a, 0.0);
    }

    /**
	* Compara dos double para determinar si uno es mayor que otro 
	*/
	public static boolean doubleMayor(double a, double b) {
		return !doubleIgual(a,b) && (a>b);
	}
	/**
	* Compara dos double para determinar si uno es mayor o igual que otro 
	*/
	public static boolean doubleMayorOIgual(double a, double b) {
		return doubleIgual(a,b) || (a>b);
	}
	/**
	* Compara dos double para determinar si uno es menor que otro 
	*/
	public static boolean doubleMenor(double a, double b) {
		return !doubleIgual(a,b) && (a<b);
	}
	/**
	* Compara dos double para determinar si uno es menor o igual que otro 
	*/
	public static boolean doubleMenorOIgual(double a, double b) {
		return doubleIgual(a,b) || (a<b);
	}

	public static int doubleSign(double a) {
		if (doubleIgual(a,0)) return 0;
		if (a>0) return 1;
		return -1;
	}

	public static int doubleSignCeroPositive(double a) {
		int sign = doubleSign(a);
		if (sign == 0) {
			return 1;
		}
		return sign;
	}

	public static boolean bigDecimalIgual(BigDecimal a, BigDecimal b) {
		return bigDecimalIgual(a, b, CASI_CERO);
	}

	public static boolean bigDecimalIgual(BigDecimal a, BigDecimal b, double tolerance) {
		return (a.subtract(b)).abs().doubleValue() <= tolerance;
	}

	public static boolean isZero(BigDecimal a) {
	    return bigDecimalIgual(a, NumberUtil.ZERO_SCALED);
    }

    public static boolean isNotZero(BigDecimal a) {
	    return !isZero(a);
    }

	public static boolean bigDecimalZero(BigDecimal a) {
		return a.abs().doubleValue() <= CASI_CERO;
	}

	/**
	* Compara dos double para determinar si uno es mayor que otro 
	*/
	public static boolean bigDecimalMayor(BigDecimal a, BigDecimal b) {
		return !bigDecimalIgual(a,b) && (a.compareTo(b)>0) ;
	}
	public static boolean bigDecimalMayor(BigDecimal a, BigDecimal b,double tolerance) {
		return !bigDecimalIgual(a,b,tolerance) && (a.compareTo(b)>0) ;
	}

	/**
	* Compara dos double para determinar si uno es mayor o igual que otro 
	*/
	public static boolean bigDecimalMayorOIgual(BigDecimal a, BigDecimal b) {
		return bigDecimalIgual(a,b) || (a.compareTo(b)>0) ;
	}
	public static boolean bigDecimalMayorOIgual(BigDecimal a, BigDecimal b, double tolerance) {
		return bigDecimalIgual(a,b,tolerance) || (a.compareTo(b)>0) ;
	}
	/**
	* Compara dos double para determinar si uno es menor que otro 
	*/
	public static boolean bigDecimalMenor(BigDecimal a, BigDecimal b) { 
		return !bigDecimalIgual(a,b) && (a.compareTo(b)<0) ;
	}
	public static boolean bigDecimalMenor(BigDecimal a, BigDecimal b, double tolerance) {
		return !bigDecimalIgual(a,b,tolerance) && (a.compareTo(b)<0) ;
	}
	/**
	 * Helper para BigDecimal, Integer 
	 */
	public static boolean bigDecimalMenor(BigDecimal a, Integer b) {
		return bigDecimalMenor(a, new BigDecimal(b));
	}
	
	/**
	* Compara dos double para determinar si uno es menor o igual que otro 
	*/
	public static boolean bigDecimalMenorOIgual(BigDecimal a, BigDecimal b) {
		return bigDecimalIgual(a,b) || (a.compareTo(b)<0);
	}
	public static boolean bigDecimalMenorOIgual(BigDecimal a, BigDecimal b,double tolerance) {
		return bigDecimalIgual(a,b,tolerance) || (a.compareTo(b)<0);
	}

	public static int bigDecimalSign(BigDecimal a) {
		if (bigDecimalIgual(a, new BigDecimal(0))) return 0;
		if (a.compareTo(new BigDecimal(0))>0) return 1;
		return -1;
	}


	/**
	 * Formatea un numero agregando ceros a izquierda
	 *
	 * @param value valor a formatear
	 * @param length longitud a la que debe llegar
	 */
	public static String zeroFill( long value, int length ) {
		StringBuilder result = new StringBuilder(String.valueOf(value));
		while ( result.length() < length )
			result.insert(0, "0");
		return result.toString();
	}

	public static String toString(double a)
	{
		return (toBigDecimal(a)).toString();
	}
	
	public static BigDecimal newBigDecimal() {
		return NumberUtil.newBigDecimal(BIGDECIMAL_SCALE);
	}
	public static BigDecimal newBigDecimal(int scale) {
		return NumberUtil.newBigDecimal("0",scale);
	}

	public static BigDecimal newBigDecimal(String value) {
		return newBigDecimal(value, BIGDECIMAL_SCALE);
	}

	public static BigDecimal newBigDecimal(BigDecimal value) {
		return newBigDecimal(value.toString(), BIGDECIMAL_SCALE);
	}

	public static BigDecimal newBigDecimal(String value, int scale) {
		if (value == null) {
			return null;
		}
		return new BigDecimal(value).setScale(scale, RoundingMode.HALF_UP);
	}
	
	public static BigDecimal toBigDecimal(double a)
	{
		return NumberUtil.toBigDecimal(a,BIGDECIMAL_SCALE);
	}

	public static BigDecimal toBigDecimalTC(double a) {
		return NumberUtil.toBigDecimal(a, BIGDECIMAL_SCALETC);
	}

    public static BigDecimal toBigDecimal(double a, int scale) {
        return (new BigDecimal(Double.toString(a))).setScale(scale, RoundingMode.HALF_UP); // se agrega redondeo para evitar ArithmeticException's
    }

    public static String formatNumber(double number, int scale) {
        return NumberUtil.toBigDecimal(number, scale).toString();
    }

    public static String formatNumber(double number, int integerSize, int scale) {
        return StringUtil.padNum(NumberUtil.formatNumber(number, scale), integerSize + scale);
    }

	public static String formatNumber(double number, int scale,String formatES_US) {
		return formatNumber( number,  scale,formatES_US, false);
	}
	public static String formatNumber(double number, int scale,String formatES_US, boolean useGrouping) {
		BigDecimal scaled = NumberUtil.toBigDecimal(number, scale);
		Locale locale;
		if (formatES_US.equals("es")) {
			locale = new Locale("es", "AR");
		} else if (formatES_US.equals("us")) {
			locale = Locale.US;
		} else {
			//un default
			locale = Locale.US;
		}
		NumberFormat numberFormat = NumberFormat.getInstance(locale);
		numberFormat.setMinimumFractionDigits(scale);
		numberFormat.setGroupingUsed(useGrouping);
		return numberFormat.format(scaled);
	}

	public static double round( double number, int numDec ) {
        BigDecimal bd = new BigDecimal( number );
        return bd.setScale( numDec, RoundingMode.HALF_EVEN ).doubleValue();
    }
    
    public static double trunc( double number, int numDec ) {
        BigDecimal bd = new BigDecimal( number );
        return bd.setScale( numDec, RoundingMode.DOWN ).doubleValue();
    }
    
    public static String safeLong( String longCandidate, boolean failOnNonNumeric ) {
		if (StringUtil.isEmpty(longCandidate)) {
			return  "0";
		}
        char digits[] = longCandidate.toCharArray();
        for ( int i = 0; i < digits.length; i++ )
            if ( !Character.isDigit( digits[ i ] ) )
                if ( failOnNonNumeric )
                    throw new IllegalArgumentException( "El caracter '"+digits[ i ]+"' no es un digito valido" );
                else
                    digits[ i ] = '0';
        return new String( digits );
    }
    
    /**
     * El metodo recibe un String y un valor por defecto. Si Integer.parseInt del String falla, entonces
     * retorno el valor por defecto.
     */
	public static int parseInt(String inputValue, int defaultValue) {
		if (StringUtil.isEmptyNull(inputValue)) return defaultValue;
		int retorno;
		try {
			retorno = Integer.parseInt(inputValue);
		} catch (Exception e) {
			retorno = defaultValue;
		}
		return retorno;
	}

	public static long parseLong(String inputValue, long defaultValue) {
		if (StringUtil.isEmptyNull(inputValue)) return defaultValue;
		long retorno;
		try {
			retorno = Long.parseLong(inputValue);
		} catch (Exception e) {
			retorno = defaultValue;
		}
		return retorno;
	}

    public static double parseDouble(String inputValue, double defaultValue) {
	    if (StringUtil.isEmptyNull(inputValue)) {
	        return defaultValue;
        }
		double retorno;
		try {
	        retorno = Double.parseDouble(inputValue);
        } catch (Exception e) {
	        retorno = defaultValue;
        }
	    return retorno;
    }


	private static DecimalFormat dfES = null; // 999.999,11
	private static DecimalFormat dfUS = null; // 999,999.11 o 999999.11

	public  static BigDecimal parseNumber(String text, String formatES_US) {
		if ("es".equalsIgnoreCase(formatES_US)) return parseNumberES(text);
		if ("us".equalsIgnoreCase(formatES_US)) return parseNumberUS(text);
		return ZERO_SCALED;
	}
	public  static BigDecimal parseNumberES(String text) {
		int sign = 1;
		if (text.contains("-")) {
			sign=-1;
			text = StringUtil.replace(text,"-","");
		}
		if (dfES == null) {
			dfES = new DecimalFormat();
			DecimalFormatSymbols symbols = new DecimalFormatSymbols();
			symbols.setDecimalSeparator(',');
			symbols.setGroupingSeparator('.');
			dfES.setDecimalFormatSymbols(symbols);
		}
		try {
			return new BigDecimal(dfES.parse(text).doubleValue()*sign);
		} catch (Exception e) {
			return BigDecimal.ZERO;
		}
	}

	public  static  BigDecimal parseNumberUS(String text) {
		int sign = 1;
		if (text.contains("-")) {
			sign=-1;
			text = StringUtil.replace(text,"-","");
		}
		if (dfUS == null) {
			dfUS = new DecimalFormat();
			DecimalFormatSymbols symbols = new DecimalFormatSymbols();
			symbols.setDecimalSeparator('.');
			symbols.setGroupingSeparator(',');
			dfUS.setDecimalFormatSymbols(symbols);
		}
		try {
			return new BigDecimal(dfUS.parse(text).doubleValue()*sign);
		} catch (Exception e) {
			return BigDecimal.ZERO;
		}
	}


	public static double safeDoubleValue(Double value, double defaultValue) {
		if (value!=null) return value.doubleValue();
		return defaultValue;
	}

	public static double safeDoubleValue(BigDecimal value, double defaultValue) {
		if (value!=null) return value.doubleValue();
		return defaultValue;
	}

	public static BigDecimal addToBigDecimal(double value1, double value2) {
		return toBigDecimal(value1).add(toBigDecimal(value2));
		
	}

	public static BigDecimal addToBigDecimal(double value1, BigDecimal value2) {
		return addToBigDecimal(value1, value2, BIGDECIMAL_SCALE);
	}
	public static BigDecimal addToBigDecimal(double value1, BigDecimal value2, int scale) {
		return toBigDecimal(value1,scale).add(value2);
	}

	public static BigDecimal subToBigDecimal(double value1, BigDecimal value2) {
		return subToBigDecimal(value1, value2, BIGDECIMAL_SCALE);
	}
	public static BigDecimal subToBigDecimal(double value1, BigDecimal value2, int scale) {
		return toBigDecimal(value1,scale).subtract(value2);
	}
	public static BigDecimal subToBigDecimal(double value1, double value2) {
		return toBigDecimal(value1).subtract(toBigDecimal(value2));
	}
	
	public static BigDecimal divideToBigDecimal(double value1,	BigDecimal value2) {
		return divideToBigDecimal(value1, value2, BIGDECIMAL_SCALE);
	}
	public static BigDecimal divideToBigDecimal(double value1,	BigDecimal value2, int scale) {
		return toBigDecimal(value1,scale).divide(value2, RoundingMode.HALF_UP);
	}
	public static BigDecimal divideToBigDecimal(BigDecimal value1, double value2) {
		return divideToBigDecimal(value1, value2, BIGDECIMAL_SCALE);
	}
	public static BigDecimal divideToBigDecimal(BigDecimal value1, double value2, int scale) {
		return value1.divide(toBigDecimal(value2, scale), RoundingMode.HALF_UP);
	}
	public static BigDecimal divideToBigDecimal(BigDecimal value1,	BigDecimal value2) {
		return value1.divide(value2, RoundingMode.HALF_UP);
	}

	public static BigDecimal divideToBigDecimal(double value1,	double value2) {
		return divideToBigDecimal(value1,	value2, BIGDECIMAL_SCALE); 
	}
	public static BigDecimal divideToBigDecimal(double value1,	double value2, int scale) {
		return toBigDecimal(value1,scale).divide(toBigDecimal(value2,scale), RoundingMode.HALF_UP);
	}
	
	public static BigDecimal multiplyToBigDecimal(double value1, BigDecimal value2) {
		return multiplyToBigDecimal(value1,value2,NumberUtil.BIGDECIMAL_SCALE);	
	}
	public static BigDecimal multiplyToBigDecimal(double value1, BigDecimal value2, int scale) {
		return toBigDecimal(value1, scale).multiply(value2).setScale(scale,RoundingMode.HALF_UP);
	}
	public static BigDecimal multiplyToBigDecimal(BigDecimal value1, BigDecimal value2, int scale) {
		return value1.multiply(value2).setScale(scale,RoundingMode.HALF_UP);
	}
    public static BigDecimal multiplyToBigDecimal(BigDecimal value1, BigDecimal value2) {
		return value1.multiply(value2).setScale(NumberUtil.BIGDECIMAL_SCALE,RoundingMode.HALF_UP);
	}
	public static BigDecimal multiplyToBigDecimal(double value1, double value2) {
		return toBigDecimal(value1).multiply(toBigDecimal(value2)).setScale(NumberUtil.BIGDECIMAL_SCALE,RoundingMode.HALF_UP);
	}

	public static double addToDouble(double value1, BigDecimal value2) {
		return addToDouble(value1, value2, BIGDECIMAL_SCALE);
	}
	public static double addToDouble(double value1, BigDecimal value2, int scale) {
		return addToBigDecimal(value1, value2,scale).doubleValue();
	}

	public static double subToDouble(double value1, BigDecimal value2) {
		return subToBigDecimal(value1, value2).doubleValue();
	}
	
	public static Double nvl(Double valor, Double valorDefault) {
		return (valor != null) ? valor : valorDefault;
	}

	public static BigDecimal nvl(Double valor, BigDecimal valorDefault) {
		return (valor != null) ? NumberUtil.toBigDecimal(valor) : valorDefault;
	}
	public static Long nvl(Long valor, Long valorDefault) {
		return (valor != null) ? valor : valorDefault;
	}

	/**
	 * Compara dos versiones del tipo maj.min, teniendo en cuenta que 2.1 != 2.10 
	 * (no se comparan como dos numeros decimales sino que se toma la parte entera y la parte decimal por separado)  
	 * 
	 * @param arg0
	 * @param arg1
	 * @param modo FULL = compara parte entera y decimal, INT = compara solo parte entera
	 * @return igual que el compareTo devuelve 
	 * 			0  si son iguales,
	 * 			-1 si arg0 es menor que arg1
	 * 			1  si arg0 es mayor que arg1
	 * 			
	 * 
	 */
	public static int compareMajorMinor(String arg0, String arg1, String modo) {
		//Separo parte entera de decimal
		Integer intArg0 = Integer.valueOf(arg0.substring(0, arg0.indexOf('.')));
		Integer decArg0 = Integer.valueOf(arg0.substring(arg0.indexOf('.') + 1));
		Integer intArg1 = Integer.valueOf(arg1.substring(0, arg1.indexOf('.')));
		Integer decArg1 = Integer.valueOf(arg1.substring(arg1.indexOf('.') + 1));
		
		if (modo.equals("FULL")) {
			//Comparo la parte entera
			int comparedIntPart = compareMajorMinor(arg0, arg1, "INT");
			//Si la parte entera es igual, comparo la decimal
			if (comparedIntPart == 0) {
				return decArg0.compareTo(decArg1);
			} else {
			//Si la parte entera no es igual, entonces devuelvo la comparacion de esta
				return comparedIntPart;
			}
			
		} else if (modo.equals("INT")) {
			//Comparo solo la parte entera
			return intArg0.compareTo(intArg1);
		} else {
			throw new IllegalArgumentException("modo [" +  modo + "] no valido");
		}
	}

    public static boolean safeEquals(Long l1, Long l2) {
	    if (l1 == null) {
	        return l2 == null;
        } else {
	        return l1.equals(l2);
        }
    }

    public static boolean safeIntegerEquals(Integer l1, Integer l2) {
        if (l1 == null) {
            return l2 == null;
        } else {
            return l1.equals(l2);
        }
    }

	/**
	 * Se asegura que se construya una collection de Longs (cuando a veces puede venir un String)
	 * @param collection
	 * @return
	 */
	public static Collection<Long> forceLongCollection(Collection<?> collection) {
		if (collection == null) {
			return null;
		}
		Collection<Long> result = new ArrayList<>();
		for (Object o : collection) {
			if (o instanceof Long) {
				result.add((Long) o);

			} else if (o instanceof String) {
				result.add(Long.valueOf((String) o));

			} else if (o instanceof Id) {
				result.add(((Id) o).longValue());

			} else {
				throw new ApplicationException("No vino un tipo valido en la collection [" + o + "]");
			}
		}
		return result;
	}

	public static boolean isValidInteger(String number) {
		if (StringUtil.isEmptyNull(number)) return false;
		try {
			Integer.parseInt(number);
		} catch (NumberFormatException e) {
			return false;
		}
		return true;
	}

    public static Integer getValidInteger(String number) {
        if (StringUtil.isEmptyNull(number)) return null;
        try {
            return Integer.parseInt(number);
        } catch (NumberFormatException e) {
            return null;
        }
    }

	public static boolean isValidDouble(String number) {
		if (StringUtil.isEmptyNull(number)) return false;
		try {
			Double.parseDouble(number);
		} catch (NumberFormatException e) {
			return false;
		}
		return true;
	}

	public static boolean isValidLong(String number) {
		if (StringUtil.isEmptyNull(number)) return false;
	    try {
            Long.parseLong(number);
        }catch (NumberFormatException e) {
            return false;
        }
        return true;
    }

	public static Long getValidLong(String number) {
		if (StringUtil.isEmptyNull(number)) return null;
		try {
			return Long.parseLong(number);
		} catch (NumberFormatException e) {
			return null;
		}
	}

	public static boolean emptySafeEquals(Integer i1, Integer i2) {
		if (i1 == null) {
			return i2 == null;
		} else {
			return i1.equals(i2);
		}
	}

	public static BigDecimal stringToBigDecimal(String number, Locale locale) throws ParseException {
        NumberFormat numberFormat = NumberFormat.getInstance(locale);
        Number numberFormatted = numberFormat.parse(number);
        return  NumberUtil.newBigDecimal(numberFormatted.toString(), 2);
    }

	public static double stringToDouble(String number) throws ParseException {
		return stringToDouble(number, Locale.ENGLISH);
	}

	public static double stringToDouble(String number, Double defaultValue) throws ParseException {
		return stringToDouble(number, defaultValue, Locale.ENGLISH);
	}

	public static double stringToDouble(String number, Locale locale) throws ParseException {
	    return stringToDouble(number, null, locale);
	}

	public static double stringToDouble(String number, Double defaultValue, Locale locale) throws ParseException {
		NumberFormat numberFormater = NumberFormat.getNumberInstance(locale);
		if (!StringUtil.isEmptyNull(number)) {
		    return numberFormater.parse(number).doubleValue();
        } else if (defaultValue != null) {
		    return defaultValue;
        } else {
		    return 0;
        }
	}

	public static boolean isInListOfIntegers(int item, Integer... items) {
        return items != null && Arrays.asList(items).contains(item);
    }

    public static boolean isInListOfLongs(long item, Long... items) {
        return items != null && Arrays.asList(items).contains(item);
    }

    public static boolean isInListOfDoubles(double item, Double... items) {
        return items != null && Arrays.asList(items).contains(item);
    }

    public static BigDecimal calculatePercetange(double total, double secondNumber) {
        return toBigDecimal(secondNumber * 100 / total);
    }

    public static BigDecimal calculateDiffPercetange(double total, double secondNumber) {
        return toBigDecimal((total - secondNumber) * 100 / total);
    }
}