package ar.com.sdd.commons.util.db;

import ar.com.sdd.commons.util.StringUtil;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

/*
 * Para perimitir busquedas en algunos campos de texto, generalmente los que soportan like
 *    La expresion de busqueda puede ser:
 *    TEXTO          =>  busca %TEXTO%
 *    /TEXTO         =>  busca  TEXTO%
 *    \TEXTO         =>  busca  %TEXTO
 *    /TEXT1-TEXT2   =>  busca TEXT1 <= x <= TEXT2
 *    /TEXT1-        =>  busca TEXT1 <= x
 *    /-TEXT2        =>  busca          x <= TEXT2
 *    TEXT1;TEXT2;.. =>  busca like %TEXT1% or like %TEXTO2% .....    Tambien se puede separar con |
 *    =TEXT1;TEXT2;.. =>  busca =TEXT1 or =TEXTO2 ..... , eliminando 0 de la izquierda   Tambien se puede separar con |
 *    ! EXPR         => niego la anterior (!TEXTO => busca not %TEXTO%), aplica a todos los anteriores
 *    !              => caso particular NOT IS NULL
 */
public class RangeExpresion {

    static private final Logger log = LogManager.getLogger(RangeExpresion.class);
    public static final int RANGE_NONE = 0;
    public static final int RANGE_LIKE = 1;
    public static final int RANGE_STARTSWITH = 2;
    public static final int RANGE_FROMTO = 3;
    public static final int RANGE_FROM = 4;
    public static final int RANGE_TO = 5;
    public static final int RANGE_ENDSWITH = 6;
    public static final int RANGE_LIST = 7;

    public int rangeFormat = RANGE_NONE;
    public boolean negate= false;
    public boolean useLikeInList=true; //Solo para el caso de listas
    public String rangeValue = null;
    public String rangeFrom = null;
    public String rangeTo = null;
    public String[] rangeList = null;

    private static final String DEBUG_PARAM_SEP = "|";

    public RangeExpresion(String rangeExp) {
        if (rangeExp!=null) {
            if(rangeExp.startsWith("!!")) {     // Caso raro: en el front-end ponen !. PEro si pasa por un getEscapedLke, se transformo en un !!, para evitar el char ! que
                negate=true;
                rangeExp=rangeExp.substring(2);
            }  else if(rangeExp.startsWith("!")) { //Por si no paso por el getEscapedLike
                negate=true;
                rangeExp=rangeExp.substring(1);
            }
            if(rangeExp.startsWith("=")) {     // Permido que se haga busquedas por igual, en listas
                useLikeInList = false;
                rangeExp = rangeExp.substring(1);
            }
        }
        rangeValue = rangeExp;
        if (StringUtil.isEmpty(rangeExp)) {
            rangeFormat = RANGE_NONE;
        } else if (rangeExp.contains(";")) {
            rangeFormat = RANGE_LIST;
            rangeList = StringUtil.splitNotEmpty(rangeExp, ';');
        } else if (rangeExp.contains("|")) {
            rangeFormat = RANGE_LIST;
            rangeList = StringUtil.splitNotEmpty(rangeExp, '|');
        } else {
            rangeFormat = RANGE_LIKE;
            if (rangeExp.startsWith("/")) {
                int dashIndex = rangeExp.indexOf('-');
                if (dashIndex == -1) {
                    rangeFormat = RANGE_STARTSWITH;
                    rangeValue = rangeExp.substring(1).trim().toUpperCase();
                    if (StringUtil.isEmpty(rangeValue)) {  // vino solo "/"
                        rangeFormat = RANGE_LIKE;
                        rangeValue = "/";
                    }
                } else {
                    rangeFormat = RANGE_FROMTO;
                    rangeFrom = rangeExp.substring(1, dashIndex).trim().toUpperCase();
                    rangeTo = rangeExp.substring(dashIndex + 1).trim().toUpperCase();
                    if (StringUtil.isEmpty(rangeFrom)) rangeFormat = RANGE_TO;
                    if (StringUtil.isEmpty(rangeTo)) rangeFormat = RANGE_FROM;
                }
            } else if (rangeExp.startsWith("\\")) {
                rangeFormat = RANGE_ENDSWITH;
                rangeValue = rangeExp.substring(1).trim().toUpperCase();
                if (StringUtil.isEmpty(rangeValue)) {  // vino solo "/"
                    rangeFormat = RANGE_LIKE;
                    rangeValue = "\\";
                }
            } else {
                rangeValue = rangeExp.trim();
            }
        }
    }

    // para manejar los casos de null
    //   a)  exrp="xx", "xx%", "xx-yy"  =>  si es null, no debe buscarlo
    //   b)  expr=""   =>  implica que va cualquier cosa, null incluido
    //  Al negarlo
    //   c)   exrp="!xx", "xx%", "xx-yy"  =>  si es null, debe incluirse
    //   d)   expre="!"                   =>  no incluye el caso null
    public StringBuilder appendToQuery(StringBuilder query, String fieldName) {
        String AND = " AND ";
        if (negate) AND = " AND NOT ";
        String nvlFieldName = negate? "nvl("+fieldName+",'$$NULL')"  : fieldName;
        switch (rangeFormat) {
            case RangeExpresion.RANGE_LIKE:
            case RangeExpresion.RANGE_STARTSWITH:
            case RangeExpresion.RANGE_ENDSWITH: {
                query.append(AND).append("(UPPER(").append(nvlFieldName).append(") like ? escape '!')");
                break;
            }
            case RangeExpresion.RANGE_FROMTO: {
                query.append(AND).append("(UPPER(TRIM(leading '0' from ").append(nvlFieldName).append(")) between ? and ? )");
                break;
            }
            case RangeExpresion.RANGE_FROM: {
                query.append(AND).append("(UPPER(TRIM(leading '0' from ").append(nvlFieldName).append(")) >= ? )");
                break;
            }
            case RangeExpresion.RANGE_TO: {
                query.append(AND).append("(UPPER(TRIM(leading '0' from ").append(nvlFieldName).append(")) <= ? )");
                break;
            }
            case RangeExpresion.RANGE_LIST: {
                //Por las dudas:
                if (rangeList.length>0) {
                    query.append(AND).append("(");
                    for (int j = 0; j < rangeList.length; j++) {
                        if (j != 0) query.append(" OR ");
                        if (useLikeInList) {
                            query.append(" (UPPER(").append(nvlFieldName).append(") like ? escape '!')");
                        } else {
                            query.append("(UPPER(TRIM(leading '0' from ").append(nvlFieldName).append(")) = ? )");
                        }

                    }
                    query.append(" )");
                }
                break;
            }
            case RangeExpresion.RANGE_NONE:
                //es un caso raro. Genera un AND NOT xxx IS NULL
                if (negate) {
                    query.append(AND).append("(").append(fieldName).append(" IS NULL)");
                }
                break;
            default: {
                log.error("Range format invalido para field[" + fieldName + "]:" + rangeFormat);
                break;
            }
        }
        return query;
    }

    @Deprecated
    public int appendParams(PreparedStatement stmt, int i, boolean debugAll, StringBuilder debugParameters) throws SQLException {
        return appendParams(stmt, i, debugAll, debugParameters, 1);
    }
    public int appendParams(PreparedStatement stmt, int i, boolean debugAll, List<Pair<String, Class<?>>> debugParameters) throws SQLException {
        return appendParams(stmt, i, debugAll, debugParameters, 1);
    }

    //Agrego los parametros. La opcion twice es poque en un caso busco con A OR B y tengo que hacer lo mismos dos veces
    // El metodo esta duplciado por dos esquemas de paramtros. NO ME GUSTA pero no quiero romper todo
    @Deprecated
    public int appendParams(PreparedStatement stmt, int i, boolean debugAll, StringBuilder debugParameters, int veces) throws SQLException {
        switch (rangeFormat) {
            case RangeExpresion.RANGE_LIKE: {
                String likeParam = "%" + QueryBuilder.getEscapedLike(rangeValue.toUpperCase()) + "%";
                for (int vez = 0; vez < veces; vez++) {
                    stmt.setString(i++, likeParam);
                    if (debugAll)
                        debugParameters.append(DEBUG_PARAM_SEP).append(i - 1).append(": ").append(likeParam);
                }
                break;
            }
            case RangeExpresion.RANGE_STARTSWITH: {
                String likeParam = QueryBuilder.getEscapedLike(rangeValue.toUpperCase()) + "%";
                for (int vez = 0; vez < veces; vez++) {
                    stmt.setString(i++, likeParam);
                    if (debugAll)
                        debugParameters.append(DEBUG_PARAM_SEP).append(i - 1).append(": ").append(likeParam);
                }
                break;
            }
            case RangeExpresion.RANGE_ENDSWITH: {
                String likeParam = "%" + QueryBuilder.getEscapedLike(rangeValue.toUpperCase());
                for (int vez = 0; vez < veces; vez++) {
                    stmt.setString(i++, likeParam);
                    if (debugAll)
                        debugParameters.append(DEBUG_PARAM_SEP).append(i - 1).append(": ").append(likeParam);
                }
                break;
            }
            case RangeExpresion.RANGE_FROMTO: {
                rangeFrom = StringUtil.trimLeading(rangeFrom, '0');
                rangeTo = StringUtil.trimLeading(rangeTo, '0');
                for (int vez = 0; vez < veces; vez++) {
                    stmt.setString(i++, rangeFrom);
                    if (debugAll)
                        debugParameters.append(DEBUG_PARAM_SEP).append(i - 1).append(": ").append(rangeFrom);
                    stmt.setString(i++, rangeTo);
                    if (debugAll)
                        debugParameters.append(DEBUG_PARAM_SEP).append(i - 1).append(": ").append(rangeTo);
                }
                break;
            }
            case RangeExpresion.RANGE_FROM: {
                rangeFrom = StringUtil.trimLeading(rangeFrom, '0');
                for (int vez = 0; vez < veces; vez++) {
                    stmt.setString(i++, rangeFrom);
                    if (debugAll)
                        debugParameters.append(DEBUG_PARAM_SEP).append(i - 1).append(": ").append(rangeFrom);
                }
                break;
            }
            case RangeExpresion.RANGE_TO: {
                rangeTo = StringUtil.trimLeading(rangeTo, '0');
                for (int vez = 0; vez < veces; vez++) {
                    stmt.setString(i++, rangeTo);
                    if (debugAll)
                        debugParameters.append(DEBUG_PARAM_SEP).append(i - 1).append(": ").append(rangeTo);
                }
                break;
            }
            case RangeExpresion.RANGE_LIST: {
                for (int j = 0; j < rangeList.length; j++) {
                    for (int vez = 0; vez < veces; vez++) {
                        String likeParam = rangeList[j].trim().toUpperCase();
                        if (useLikeInList) {
                            likeParam= "%" + QueryBuilder.getEscapedLike(likeParam) + "%";
                        }
                        stmt.setString(i++, likeParam);
                        if (debugAll)
                            debugParameters.append(DEBUG_PARAM_SEP).append(i - 1).append(": ").append(likeParam);
                    }
                }
                break;
            }
        }
        return i;
    }
    public int appendParams(PreparedStatement stmt, int i, boolean debugAll, List<Pair<String, Class<?>>> debugParameters, int veces) throws SQLException {
        switch (rangeFormat) {
            case RangeExpresion.RANGE_LIKE: {
                String likeParam = "%" + QueryBuilder.getEscapedLike(rangeValue.toUpperCase()) + "%";
                for (int vez = 0; vez < veces; vez++) {
                    stmt.setString(i++, likeParam);
                    if (debugAll) debugParameters.add(Pair.of(likeParam, String.class));
                }
                break;
            }
            case RangeExpresion.RANGE_STARTSWITH: {
                String likeParam = QueryBuilder.getEscapedLike(rangeValue.toUpperCase()) + "%";
                for (int vez = 0; vez < veces; vez++) {
                    stmt.setString(i++, likeParam);
                    if (debugAll) debugParameters.add(Pair.of(likeParam, String.class));
                }
                break;
            }
            case RangeExpresion.RANGE_ENDSWITH: {
                String likeParam = "%" + QueryBuilder.getEscapedLike(rangeValue.toUpperCase());
                for (int vez = 0; vez < veces; vez++) {
                    stmt.setString(i++, likeParam);
                    if (debugAll) debugParameters.add(Pair.of(likeParam, String.class));
                }
                break;
            }
            case RangeExpresion.RANGE_FROMTO: {
                rangeFrom = StringUtil.trimLeading(rangeFrom, '0');
                rangeTo = StringUtil.trimLeading(rangeTo, '0');
                for (int vez = 0; vez < veces; vez++) {
                    stmt.setString(i++, rangeFrom);
                    if (debugAll) debugParameters.add(Pair.of(rangeFrom, String.class));
                    stmt.setString(i++, rangeTo);
                    if (debugAll) debugParameters.add(Pair.of(rangeTo, String.class));
                }
                break;
            }
            case RangeExpresion.RANGE_FROM: {
                rangeFrom = StringUtil.trimLeading(rangeFrom, '0');
                for (int vez = 0; vez < veces; vez++) {
                    stmt.setString(i++, rangeFrom);
                    if (debugAll) debugParameters.add(Pair.of(rangeFrom, String.class));
                }
                break;
            }
            case RangeExpresion.RANGE_TO: {
                rangeTo = StringUtil.trimLeading(rangeTo, '0');
                for (int vez = 0; vez < veces; vez++) {
                    stmt.setString(i++, rangeTo);
                    if (debugAll) debugParameters.add(Pair.of(rangeTo, String.class));
                }
                break;
            }
            case RangeExpresion.RANGE_LIST: {
                for (int j = 0; j < rangeList.length; j++) {
                    for (int vez = 0; vez < veces; vez++) {
                        String likeParam = rangeList[j].trim().toUpperCase();
                        if (useLikeInList) {
                             likeParam = "%" + QueryBuilder.getEscapedLike(likeParam) + "%";
                        }
                        stmt.setString(i++, likeParam);
                        if (debugAll) debugParameters.add(Pair.of(likeParam, String.class));
                    }
                }
                break;
            }
        }
        return i;
    }


    public String getSQLFilter(String tableField) {
        return getSQLFilter(tableField, true);
    }

    public RangeExpresion fillSQLDebugParameters(List<Pair<String, Class<?>>> debugParameters ) {
        for(String value:this.getSQLValues()) {
            debugParameters.add(Pair.of(value, String.class));
        }
        return this;
    }

    public ArrayList<String> getSQLValues() {
        ArrayList<String> returnRangeList = new ArrayList<>();
        switch (rangeFormat) {
            case RangeExpresion.RANGE_LIKE: {
                returnRangeList.add("%" + QueryBuilder.getEscapedString(QueryBuilder.getEscapedLike(rangeValue)).toUpperCase() + "%");
                break;
            }
            case RangeExpresion.RANGE_STARTSWITH: {
                returnRangeList.add(QueryBuilder.getEscapedString(QueryBuilder.getEscapedLike(rangeValue)).toUpperCase() + "%");
                break;
            }
            case RangeExpresion.RANGE_ENDSWITH:{
                returnRangeList.add("%" + QueryBuilder.getEscapedString(QueryBuilder.getEscapedLike(rangeValue)).toUpperCase());
                break;
            }
            case RangeExpresion.RANGE_FROMTO: {
                returnRangeList.add(rangeFrom);
                returnRangeList.add(rangeTo);
                break;
            }
            case RangeExpresion.RANGE_FROM: {
                returnRangeList.add(rangeFrom);
                break;
            }
            case RangeExpresion.RANGE_TO: {
                returnRangeList.add(rangeTo);
                break;
            }
            case RangeExpresion.RANGE_LIST: {
                for (int n = 0; n < rangeList.length; n++) {
                    String likeParam = rangeList[n].toUpperCase();
                    if (useLikeInList) {
                        likeParam = "%" + QueryBuilder.getEscapedString( likeParam)+ "%";
                    }
                    returnRangeList.add(likeParam);
                }
                break;
            }
        }

        return returnRangeList;
    }

    public String getSQLFilter(String tableField, boolean withValues) {
        String condition = "";
        switch (rangeFormat) {
            case RangeExpresion.RANGE_LIKE: {
                String likeParam = withValues ? "'%" + QueryBuilder.getEscapedString(QueryBuilder.getEscapedLike(rangeValue)).toUpperCase() + "%'" : "?";
                condition = "(UPPER(" + tableField + ") like " + likeParam + " escape '!')";
                break;
            }
            case RangeExpresion.RANGE_STARTSWITH: {
                String likeParam = withValues ? "'" + QueryBuilder.getEscapedString(QueryBuilder.getEscapedLike(rangeValue)).toUpperCase() + "%'" : "?";
                condition = "(UPPER(" + tableField + ") like " + likeParam + " escape '!')";
                break;
            }
            case RangeExpresion.RANGE_ENDSWITH:{
                String likeParam = withValues ? "'%" + QueryBuilder.getEscapedString(QueryBuilder.getEscapedLike(rangeValue)).toUpperCase() + "'" : "?";
                condition = "(UPPER(" + tableField + ") like " + likeParam + " escape '!')";
                break;
            }
            case RangeExpresion.RANGE_FROMTO: {
                condition = "(UPPER(" + tableField + ") between " + (withValues ? "'" + rangeFrom + "'" : "?") + " AND " + (withValues ? "'" + rangeTo + "'" : "?") + ")";
                break;
            }
            case RangeExpresion.RANGE_FROM: {
                condition = "(UPPER(" + tableField + ") >= " + (withValues ? "'" + rangeFrom + "'" : "?") + ")";
                break;
            }
            case RangeExpresion.RANGE_TO: {
                condition = "(UPPER(" + tableField + ") <= " + (withValues ? "'" + rangeTo + "'" : "?") + ")";
                break;
            }
            case RangeExpresion.RANGE_LIST: {
                condition += "(";
                for (int j= 0; j < rangeList.length; j++) {
                    String likeParam = withValues ? rangeList[j].toUpperCase() : "?";
                    if (withValues && useLikeInList) {
                        likeParam = "%" + QueryBuilder.getEscapedString( likeParam)+ "%";
                    }

                    if (j != 0) condition += " OR ";
                    if (useLikeInList) {
                        condition += "(UPPER(" + tableField + ") like '" + likeParam + "' escape '!')";
                    } else {
                        condition += "(UPPER(" + tableField + ") =   '" + likeParam + "')";
                    }
                }
                condition += ")";
                break;
            }
        }
        //Contempl el caso NOT:
        if (negate) {
            if (rangeFormat == RANGE_NONE) {
                condition = "( NOT " + tableField + " IS NULL )";
            } else {
                condition = "(NOT " + condition + ")";
            }
        }
        return condition;
    }
}
