package ar.com.sdd.commons.util.xml;

import ar.com.sdd.commons.util.ObjectUtil;
//import ar.com.sdd.util.io.LineInputStream;
import ar.com.sdd.commons.util.InvalidArgumentException;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.xml.sax.Attributes;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.DefaultHandler;

import javax.swing.tree.TreeNode;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import java.io.*;
import java.util.*;

/**
 * Esta clase representa los datos a intercambiarse con AFIP. Brinda
 * funcionalidad para manejar el XML.
 */
public class XMLData {

	private static final Logger log = LogManager.getLogger(XMLData.class);
	
	private static final String PREFIX_XMLI = "xmli";

	private static final String NS_XMLI = "http://sdd.com.ar/xmlinclude/1.0";

	/**
	 * Datos del XML
	 */
	private XMLTag data = null;
	/**
	 * Si es true => usamos namespaces
	 */
	private boolean useNamespaces = false;

	private File resourcesUrl = null;

	// ----------------------------------------------------------------------------//

	/**
	 * Constructor default
	 */
	public XMLData() {
	}

	/**
	 * Recibe un String con el XML, lo parsea. Obtiene el primer nodo y lo
	 * almacena en data.
	 */
	public void parseFromString(String data) throws XMLDataException {
		ByteArrayInputStream dataStr = new ByteArrayInputStream(data.getBytes());
		parse(dataStr);
	}

	/**
	 * Recibe el InputStream con el XML, lo parsea. Obtiene el primer nodo y lo
	 * almacena en data. Esto impide el mecanismo de includes. USAR CON CUIDADO
	 */
	private void parse(InputStream inStream) throws XMLDataException {

		data = null; // Por si acaso

		try {

			SAXParserFactory spf = SAXParserFactory.newInstance();			
			spf.setNamespaceAware(true);
			SAXParser saxParser;

			saxParser = spf.newSAXParser();
			Reader reader = new Reader();

			try {

				saxParser.parse(inStream, reader);
				data = reader.getTop();

			} catch (SAXException e) {

				throw new XMLDataException("Error al parsear el stream", e,
						false);

			} catch (IOException e) {

				throw new XMLDataException("Error al leer el stream", e, false);

			}

		} catch (ParserConfigurationException e) {

			throw new XMLDataException("Error de configuracion del parser", e,
					true);

		} catch (SAXException e) {

			throw new XMLDataException("Error al inicializar el parser", e,
					true);

		}
	}

	/**
	 * Recibe el nombre del archivo con el XML, lo parsea, obtiene el primer
	 * nodo y lo almacena en data. Procesa los tags xmli:include
	 */
	public void parse(String xmlDocument) throws XMLDataException {
		parse(xmlDocument, true, 0);
	}

	public void parse(String xmlDocument, boolean doIncludes) throws XMLDataException {
		parse(xmlDocument, doIncludes, 0);
	}

	/**
	 * Recibe el nombre del archivo con el XML, lo parsea, obtiene el primer
	 * nodo y lo almacena en data.
	 */
	public void parse(String xmlDocument, boolean doIncludes, Integer skipLines) throws XMLDataException {

		data = null; // Por si acaso
		File parentFile = null;
		try {
			
			SAXParserFactory spf = SAXParserFactory.newInstance();
			spf.setNamespaceAware(true);			
			SAXParser saxParser;
			saxParser = spf.newSAXParser();			

			Reader reader = new Reader();

			try {
				File xmlDocumentFile = new File(xmlDocument);
				parentFile = xmlDocumentFile.getParentFile();
				//setResourcesUrl(xmlDocumentFile.getParentFile());
				if (skipLines != null && skipLines > 0) {
					//InputStream is = new LineInputStream(new FileInputStream(xmlDocumentFile), skipLines, false);
                    //@TODO: implementar el skipLines
                    InputStream is = new FileInputStream(xmlDocumentFile);
					saxParser.parse(is, reader);
				} else {
					saxParser.parse(xmlDocumentFile, reader);
				}

			} catch (SAXException e) {

				throw new XMLDataException("Error al parsear el documento ["
						+ xmlDocument + "]", e, false);

			} catch (IOException e) {

				throw new XMLDataException("Error al leer el documento ["
						+ xmlDocument + "]", e, false);

			}

			data = reader.getTop();

		} catch (SAXException ex) {

			throw new XMLDataException(
					"Error al inicializar el parser para el documento ["
							+ xmlDocument + "]", ex, true);

		} catch (ParserConfigurationException ex) {

			throw new XMLDataException(
					"Error de configuracion en el parser para el documento ["
							+ xmlDocument + "]", ex, true);

		}

		if (doIncludes) {
			try {
				resolveIncludes(parentFile,true);
			} catch (Exception ex) {
				throw new XMLDataException(
						"Error de resolucion de includes en el parser para el documento [" + xmlDocument + "]", ex, true);
			}
		}
						
	}

	/**
	 * Resuelve los tags <xmli:include><br/>
	 *
	 * Los tags que se resuelven son los pertenecientes al namespace
	 * "http://sdd.com.ar/xmlinclude/1.0", por lo que un tag que incluya a todas
	 * las directivas debe tener la declaracion de namespace correspondiente
	 * (xmlns:xmli="http://sdd.com.ar/xmlinclude/1.0")
	 *
	 * Los tags que se resuelve son:
	 * <ul>
	 * <li><xmli:key element="TAG" attributes="ATR1,ATR2,..."/> Declara que el
	 * matching sobre el tag "TAG" sera sobre los atributos "ATR1", "ATR2", etc
	 * <li><xmli:include href="..."/> Incluye (toma los defaults) de otro
	 * documento XML a buscar con el URL dado en "src"
	 * </ul>
	 * Cualquier tag en un documento puede tener el atributo "xmli:override",
	 * con los siguientes valores:
	 * <ul>
	 * <li>"add": Significa que el elemento se agregara independientemente de
	 * los elementos que pudieran existir en los documentos incluidos.
	 * <li>"all": El valor del elemento declarado y todos sus hijos reemplaza
	 * al que pudiera existir en los documentos incluidos.
	 * <li>"declared": Este es el valor implicito para los tags que no
	 * especifican el atributo. Solo se reemplazan los tags declarados
	 * <li>"delete": Se elimina el elemento que pudiera existir en los
	 * documentos incluidos.
	 * </ul>
	 * Este valor se propaga hacia abajo hasta que se redefina.<br/> Por
	 * ejemplo, con los siguientes XMLs: <br/><code>modelo.xml</code>:
	 *
	 * <pre>
	 * 	 &lt;ejemplo xmlns:xmli=&quot;http://sdd.com.ar/xmlinclude/1.0&quot;&gt;
	 * 	 &lt;xmli:key element=&quot;t1&quot; attributes=&quot;name&quot;/&gt;
	 * 	 &lt;xmli:key element=&quot;st1&quot; attributes=&quot;value&quot;/&gt;
	 * 	 &lt;xmli:key element=&quot;st2&quot; attributes=&quot;value&quot;/&gt;
	 *
	 * 	 &lt;t1 name=&quot;1&quot;&gt;
	 * 	 &lt;st1 value=&quot;1_1&quot;&gt;TEXTO 1_1&lt;/st1&gt;
	 * 	 &lt;st1 value=&quot;1_2&quot;&gt;TEXTO 1_2&lt;/st1&gt;
	 * 	 &lt;st1 value=&quot;1_3&quot;&gt;TEXTO 1_3&lt;/st1&gt;
	 * 	 &lt;st2 value=&quot;2&quot;/&gt;
	 * 	 &lt;/t1&gt;
	 *
	 * 	 &lt;t1 name=&quot;2&quot;&gt;
	 * 	 &lt;st1 value=&quot;2_1&quot;&gt;TEXTO 2_1&lt;/st1&gt;
	 * 	 &lt;st1 value=&quot;2_2&quot;&gt;TEXTO 2_2&lt;/st1&gt;
	 * 	 &lt;st1 value=&quot;2_3&quot;&gt;TEXTO 2_3&lt;/st1&gt;
	 * 	 &lt;st2 value=&quot;2&quot;/&gt;
	 * 	 &lt;/t1&gt;
	 *
	 * 	 &lt;t1 name=&quot;3&quot;&gt;
	 * 	 &lt;st1 value=&quot;3_1&quot;&gt;TEXTO 3_1&lt;/st1&gt;
	 * 	 &lt;st1 value=&quot;3_2&quot;&gt;TEXTO 3_2&lt;/st1&gt;
	 * 	 &lt;st1 value=&quot;3_3&quot;&gt;TEXTO 3_3&lt;/st1&gt;
	 * 	 &lt;st2 value=&quot;2&quot;/&gt;
	 * 	 &lt;/t1&gt;
	 *
	 * 	 &lt;t1 name=&quot;4&quot;&gt;
	 * 	 &lt;st1 value=&quot;4_1&quot;&gt;TEXTO 4_1&lt;/st1&gt;
	 * 	 &lt;st1 value=&quot;4_2&quot;&gt;TEXTO 4_2&lt;/st1&gt;
	 * 	 &lt;st1 value=&quot;4_3&quot;&gt;TEXTO 4_3&lt;/st1&gt;
	 * 	 &lt;st2 value=&quot;2&quot;/&gt;
	 * 	 &lt;/t1&gt;
	 *
	 * 	 &lt;/ejemplo&gt;
	 *
	 * </pre>
	 *
	 * <br/><code>edits.xml</code>:
	 *
	 * <pre>
	 * 	 &lt;ejemplo xmlns:xmli=&quot;http://sdd.com.ar/xmlinclude/1.0&quot;&gt;
	 * 	 &lt;xmli:include href=&quot;modelo.xml&quot;/&gt;
	 *
	 * 	 &lt;t1 name=&quot;3&quot; xmli:override=&quot;add&quot;&gt;
	 * 	 &lt;st1 value=&quot;3_2&quot;&gt;NUEVO 3_2&lt;/st1&gt;
	 * 	 &lt;/t1&gt;
	 *
	 * 	 &lt;t1 name=&quot;1&quot;&gt;
	 * 	 &lt;st1 value=&quot;1_2&quot;&gt;NUEVO 1_2&lt;/st1&gt;
	 * 	 &lt;/t1&gt;
	 *
	 * 	 &lt;t1 name=&quot;2&quot; xmli:override=&quot;all&quot;&gt;
	 * 	 &lt;st1 value=&quot;2_2&quot;&gt;NUEVO 2_2&lt;/st1&gt;
	 * 	 &lt;/t1&gt;
	 *
	 * 	 &lt;t1 name=&quot;4&quot; xmli:override=&quot;delete&quot;/&gt;
	 *
	 * 	 &lt;/ejemplo&gt;
	 *
	 * </pre>
	 *
	 * <br/>Resultado:
	 *
	 * <pre>
	 * 	 &lt;ejemplo&gt;
	 * 	 &lt;t1 name=&quot;1&quot;&gt;
	 * 	 &lt;st1 value=&quot;1_1&quot;&gt;TEXTO 1_1&lt;/st1&gt;
	 * 	 &lt;st1 value=&quot;1_2&quot;&gt;NUEVO 1_2&lt;/st1&gt;
	 * 	 &lt;st1 value=&quot;1_3&quot;&gt;TEXTO 1_3&lt;/st1&gt;
	 * 	 &lt;st2 value=&quot;2&quot;/&gt;
	 * 	 &lt;/t1&gt;
	 * 	 &lt;t1 name=&quot;2&quot;&gt;
	 * 	 &lt;st1 value=&quot;2_2&quot;&gt;NUEVO 2_2&lt;/st1&gt;
	 * 	 &lt;/t1&gt;
	 * 	 &lt;t1 name=&quot;3&quot;&gt;
	 * 	 &lt;st1 value=&quot;3_1&quot;&gt;TEXTO 3_1&lt;/st1&gt;
	 * 	 &lt;st1 value=&quot;3_2&quot;&gt;TEXTO 3_2&lt;/st1&gt;
	 * 	 &lt;st1 value=&quot;3_3&quot;&gt;TEXTO 3_3&lt;/st1&gt;
	 * 	 &lt;st2 value=&quot;2&quot;/&gt;
	 * 	 &lt;/t1&gt;
	 * 	 &lt;t1 name=&quot;3&quot;&gt;
	 * 	 &lt;st1 value=&quot;3_2&quot;&gt;NUEVO 3_2&lt;/st1&gt;
	 * 	 &lt;/t1&gt;
	 * 	 &lt;/ejemplo&gt;
	 *
	 * </pre>
	 *
	 * @throws XMLDataException
	 */
	public void resolveIncludes(File parentFile, boolean doCleanupImportTags)
			throws XMLDataException {
		// Obtener modelo (documento a incluir)
		XMLData model = getIncludedModel(parentFile);
		if (model != null) {
			Map includeMatchKeys = new HashMap();

			// Obtener keys para matching
			addIncludeKeys(includeMatchKeys, model.getRoot());

			// Override de keys si es necesario
			addIncludeKeys(includeMatchKeys, getRoot());
			
			// Editar el modelo con los cambios
			applyIncludeEdits(includeMatchKeys, model, null, getRoot(), null, new HashSet());

			data = model.getRoot();

			if (doCleanupImportTags)
				cleanupImportTags(data);
		}
	}

	private void cleanupImportTags(XMLTag tag) {
		List toCleanup = new ArrayList();

		// Hay que hacerlo en dos pasos porque las modificaciones
		// invalidan la Enumeration
		Enumeration childrenEnum = tag.breadthFirstEnumeration();
		while (childrenEnum.hasMoreElements()) {
			XMLTag childTag = (XMLTag) childrenEnum.nextElement();
			if (childTag.getNsUri() != null
					&& childTag.getNsUri().equals(NS_XMLI))
				toCleanup.add(childTag);
			else {
				// Se eliminan tambien los atributos XMLI
                // El vector se necesita para evitar concurrent modifications en el
                // hashtable
				Iterator attrs = (new ArrayList( childTag.getProperties().keySet() )).iterator();
				while (attrs.hasNext()) {
					String attr = (String) attrs.next();
					if (NS_XMLI.equals(childTag.getPropertyNsUri(attr)))
						childTag.removeProperty(attr);
				}
			}
		}

		Iterator i = toCleanup.iterator();
		while (i.hasNext()) {
			XMLTag element = (XMLTag) i.next();
			element.removeFromParent();
		}
	}

	private void addIncludeKeys(Map includeKeys, XMLTag tag) {
		String key = getXmliTagNameResolvingNamespaces("key");
		List keys = tag.getItems(NS_XMLI, key, true); //TODO : key
		Iterator keysIterator = keys.iterator();
		while (keysIterator.hasNext()) {
			XMLTag keyTag = (XMLTag) keysIterator.next();
			String keyElement = keyTag.getProperty("element");
			String keyAttributes = keyTag.getProperty("attributes");
			if (keyElement == null)
				throw new InvalidArgumentException(
						"Falta atributo 'element' para comparar tags");
			if (keyAttributes == null)
				throw new InvalidArgumentException(
						"Falta atributo 'attributes' para comparar tags");
			Set attributesSet = new HashSet();
			String attributes[] = keyAttributes.split(",");
			for (int i = 0; i < attributes.length; i++)
				attributesSet.add(attributes[i]);
			includeKeys.put(keyElement, attributesSet);
		}
	}

	private XMLData loadIncludedModel( File parentFile, XMLTag includeTag ) throws XMLDataException
	{
		String includeResourceUri = includeTag.getProperty("href");
		if (includeResourceUri == null)
			throw new InvalidArgumentException(
					"Falta atributo 'href' para incluir modelo");
		XMLData result = new XMLData();		
		result.setUseNamespaces(isUseNamespaces());
		File includedFile = new File(parentFile, includeResourceUri);
		result.parse(includedFile.getAbsolutePath(), false);
		
		result.resolveIncludes(includedFile.getParentFile(),false);
		return result;
	}

	private XMLData getIncludedModel(File parentFile) throws XMLDataException {
		XMLData result = null;

		// Si hay mas de un include, "anidar" manualmente en un unico modelo
		XMLTag root = getRoot();
		String include = getXmliTagNameResolvingNamespaces("include");
		List includeTags = root.getItems(NS_XMLI,include, true); // TODO : include
		while ( includeTags != null && !includeTags.isEmpty() )
		{
			XMLTag includeTag = (XMLTag) includeTags.get(0);
			if ( result == null )
				result = loadIncludedModel(parentFile, includeTag );
			else
			{
				XMLTag newInclude = new XMLTag( include, includeTag.getProperties() ); // TODO : include
				newInclude.setNsUri( NS_XMLI );
				result.getRoot().add( newInclude );
				result.resolveIncludes(parentFile, false );
			}
			includeTags.remove( includeTag );
			includeTag.removeFromParent();
		}
		return result;

//		if (includeTags.size() > 1)
//			throw new InvalidArgumentException(
//					"No puede haber mas de un include XML");
//		XMLTag includeTag = includeTags.isEmpty() ? null : (XMLTag) includeTags
//				.firstElement();
//		if (includeTag != null)
//			result = loadIncludedModel( includeTag );
//		return result;
	}

	private String getIncludeProperty( XMLTag tag, String propertyName ) {
		String property = tag.getProperty( propertyName );
		if ( property != null && !NS_XMLI.equals( tag.getPropertyNsUri( propertyName ) ) )
			property = null;
		return property;
	}

	private void applyIncludeEdits(Map includeKeys, XMLData model,
			XMLTag modelContext, XMLTag tag, String defaultOverride, Set alreadyEdited) throws XMLDataException {
		// Edicion a realizar
		String tagOverride = getIncludeProperty( tag, "override" );
		if (tagOverride == null)
			tagOverride = defaultOverride == null ? "declared"
					: defaultOverride;

		// Encontrar punto de edicion
		List editPoints = getIncludeEditPoints(includeKeys, model,
				modelContext, tag, tagOverride, alreadyEdited);

		// Si la edicion es "delete", borrarlos
		if (tagOverride.equals("delete")) {
			Iterator toDelete = editPoints.iterator();
			while (toDelete.hasNext()) {
				XMLTag deleteTag = (XMLTag) toDelete.next();
				deleteTag.removeFromParent();
			}
		
		} else {
			if (editPoints.size() > 1) {
				String msg = "Se selecciono mas de un tag para modificar '" + tag.getNamePath(); 
				log.error(msg);
				log.error("Verificar en cada archivo incluido que esta clave no esta mas de una vez declarada.");
				log.error(editPoints);
				throw new InvalidArgumentException( msg ); 
			}
			if (!editPoints.isEmpty()) {
				XMLTag newContext = (XMLTag) editPoints.get(0);
				if (tagOverride.equals("all")) {
					newContext.removeAllChildren();
					defaultOverride = "all";
				}
				// Fijar Atributos
				Iterator attrs = tag.getProperties().keySet().iterator();
				while (attrs.hasNext()) {
					String attr = (String) attrs.next();
					newContext.setProperty( tag.getPropertyNsUri(attr), attr, tag.getProperty( attr ) );
				}
				// Fijar Datos
				newContext.setData(tag.getData());
				// Fijar Hijos
				Set editedChildren = new HashSet();
				Enumeration e = tag.children();
				while (e.hasMoreElements()) {
					XMLTag child = (XMLTag) e.nextElement();
					applyIncludeEdits(includeKeys, model, newContext, child,
							defaultOverride, editedChildren);
				}
				
				// Marcar como editado, para no volver a editar si hay mas de
				// un nodo con el mismo xmli:key
				alreadyEdited.add( newContext );
			}
		}
	}

	private List<XMLTag> getIncludeEditPoints(Map includeKeys, XMLData model,
			XMLTag modelContext, XMLTag tag, String tagOverride, Set alreadyEdited) throws XMLDataException {
		List<XMLTag> result = new ArrayList<XMLTag>();

		if (modelContext != null) {
			// Obtener todos los edit points correspondientes al contexto del
			// modelo
			// Agregar como candidatos a los hijos que coincidan con el tag
			if (!tagOverride.equals("add")) {
				String forcedMatch = getIncludeProperty( tag, "match" );
				List<XMLTag> candidates = modelContext.getItems(tag.getName(), false);
				if ( forcedMatch != null ) {
					XMLTag tagToMatch = matchSiblingNode( tag, candidates, "match" );
					result.add( tagToMatch );
				} else {
					Iterator<XMLTag> candidatesIter = candidates.iterator();
					while (candidatesIter.hasNext()) {
						XMLTag candidate = candidatesIter.next();
						if (matchIncluded(includeKeys, candidate, tag, alreadyEdited))
							result.add(candidate);
					}
				}
			}

			// Si no se encontro ningun edit point, agregar al contexto
			// (a menos que la accion sea "delete") un clon del tag
			if (result.isEmpty() && !tagOverride.equals("delete")) {
				// Clonar TAG
				XMLTag clonedTag = null;

                // Si hay que duplicar, clonar de esa
                if ( tagOverride.equals( "duplicate" )) {
    				List<XMLTag> candidates = modelContext.getItems(tag.getName(), false);
                    XMLTag tagToClone = matchSiblingNode( tag, candidates, "duplicate" );
                    if ( tagToClone != null )
                        clonedTag = tagToClone.deepClone();
                }

                if ( clonedTag == null ) {
                    clonedTag = new XMLTag(tag.getName());
                    clonedTag.setNsUri(tag.getNsUri());

                    // Clonar ATRIBUTOS
                    Iterator attrs = tag.getProperties().keySet().iterator();
                    while (attrs.hasNext()) {
                        String attr = (String) attrs.next();
                        clonedTag.setProperty(tag.getPropertyNsUri(attr), attr, tag
                                .getProperty(attr));
                    }
                }

				// Agregar al hijo
               	addIncludeEditPoint(modelContext, clonedTag);
				result.add(clonedTag);
			}
		} else {
			// Devolver el nodo raiz, si es que coincide con el tag buscado
			if (!tag.getName().equals(model.getRoot().getName()))
				throw new InvalidArgumentException(
						"El tag raiz del XML incluido debe ser el mismo");
			result.add(model.getRoot());
		}
		return result;
	}

	private void addIncludeEditPoint(XMLTag modelContext, XMLTag tag) throws XMLDataException {
        // La version simple seria:
        //
		// modelContext.add( tag );
        //
		// Pero queremos agregar al final de los tags del mismo nombre o antes
        // del especificado en el tag "before"
		List mismoNombre = modelContext.getItems(tag.getName(), false);

        // OJO! El cast a TreeNode se hace sabiendo que la implementacion que
        // usamos implementa esa interface. Esto tranquilamente puede cambiar
        // en el futuro

        // Si se especifico "xmli:before", buscar el nodo antes del cual insertar
        XMLTag insertBefore = matchSiblingNode( tag, mismoNombre, "before" );

        // Si no se especifico "xmli:before" o no se encontro  el nodo antes
        // del cual insertar insertar despues del ultimo con el mismo nombre
        int insertBeforeIndex =
            insertBefore != null?
                    modelContext.getIndex( insertBefore ):
            !mismoNombre.isEmpty()?
                    modelContext.getIndex( (XMLTag) mismoNombre.get(mismoNombre.size()-1) ) + 1:
                    -1;

        // Agregar o insertar
		if ( insertBeforeIndex < 0 )
			modelContext.add( tag );
		else
			modelContext.insert( tag, insertBeforeIndex );
	}

    private XMLTag matchSiblingNode( XMLTag tag, List mismoNombre, String keyProperty ) throws XMLDataException {
        XMLTag result = null;
        String tagBefore = getIncludeProperty( tag, keyProperty );
        if ( tagBefore != null ) {
            int i;
            String terms[] = tagBefore.split(",");
            String attrValuePairs[][] = new String[ terms.length ][];
            for ( i = 0; i < terms.length; i++ ) {
                attrValuePairs[ i ] = terms[ i ].split( "=" );
                if ( attrValuePairs[ i ].length != 2 )
                    throw new XMLDataException( "xmli:"+keyProperty+" invalido '" + tagBefore + "', debe ser 'attr=value,attr=value...'", true );
            }

            Iterator tags = mismoNombre.iterator();
            while ( tags.hasNext() && result == null ) {
                XMLTag item = (XMLTag) tags.next();
                boolean found = true;
                for ( i = 0; found && i < terms.length; i++ )
                    found =
                        ObjectUtil.safeEquals(
                                item.getProperty( attrValuePairs[ i ][ 0 ] ),
                                attrValuePairs[ i ][ 1 ] );
                if ( found )
                    result = item;
            }
        }
        return result;
    }

	private boolean matchIncluded(Map includeKeys, XMLTag editCandidate,
			XMLTag tag, Set alreadyEdited) {
		boolean result = !alreadyEdited.contains( editCandidate );

		if ( result ) {
			Set attrs = (Set) includeKeys.get(tag.getName());
			if (attrs == null)
			{
				attrs = new HashSet( tag.getProperties().keySet() );
				Iterator j = attrs.iterator();
				while ( j.hasNext() )
				{
					String attr = (String) j.next();
					if ( NS_XMLI.equals( tag.getPropertyNsUri( attr ) ) )
						j.remove();
				}
			}
			Iterator i = attrs.iterator();
			while (result && i.hasNext()) {
				String attr = (String) i.next();
				String candidateValue = editCandidate.getProperty(attr);
				String tagValue = tag.getProperty(attr);
				result = (candidateValue == null && tagValue == null)
						|| ((candidateValue != null && tagValue != null) && candidateValue.equals(tagValue));
			}
		}
		return result;
	}

	/*Este metodo se movio a XMLDataRunner.java (en los tests) para que pueda ser ejecutado correctamente con todas las librerias
    (usaba Log4J que esta marcada como provided y solo dentro de los tests la toma como compiled

	public static void main(String[] args) {}
	*/

	/**
	 * Devuelve un Stream de donde leer el XML
	 */
	public InputStream toInputStream(boolean header) {

		ByteArrayOutputStream out = new ByteArrayOutputStream();
		this.saveToStream(out, header);
		ByteArrayInputStream retr = new ByteArrayInputStream(out.toByteArray());

		return retr;
	}

	/**
	 * Devuelve un Stream de donde leer el XML. Igual a toInputStream(true)
	 */
	public InputStream toInputStream() {

		ByteArrayOutputStream out = new ByteArrayOutputStream();
		this.saveToStream(out);
		ByteArrayInputStream retr = new ByteArrayInputStream(out.toByteArray());

		return retr;
	}

	/**
	 * Guarda el XML en el outStream, si header == true incluye una cabecera
	 */
	public void saveToStream(OutputStream outStream, boolean header) {

		if (!isEmpty()) {

			PrintWriter out = new PrintWriter(outStream);

			if (header)
				out.println("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
			if (data != null)
				data.saveToStream(out);

			out.flush();

			try {

				outStream.close();

			} catch (IOException e) {

				Logger log = LogManager.getLogger(this.getClass());
				log.warn("No se pudo cerrar correctamente el stream", e);

			}
		}
	}

	/**
	 * Guarda el XML en el outStream. Igual a saveToStream(outSteam, true)
	 */
	public void saveToStream(OutputStream outStream) {
		this.saveToStream(outStream, true);
	}

	/**
	 * true si no hay datos cargados
	 */
	public boolean isEmpty() {
		return (data == null) || (data.getChildCount() == 0);
	}

	/**
	 * Item padre
	 *
	 * @return XMLTag root del arbol de XML (null si vacio)
	 */
	public XMLTag getRoot() {
		return data;
	}

	/**
	 * Devuelve un vector de items que corresponden al nombre, con busqueda
	 * recursiva por defecto.
	 *
	 * @return Un ArrayList, vacio si no hay items
	 */
	public List<XMLTag>getItems(String name) {
		return getItems(name, true);
	}

	/**
	 * Devuelve un vector de items que corresponden al nombre,
	 *
	 * @return Un ArrayList, vacio si no hay items
	 */
	public List<XMLTag>getItems(String name, boolean recursive) {

		List<XMLTag>ret;
		if (data != null) {
			ret = data.getItems(name, recursive);
		} else {
			ret = new ArrayList<XMLTag>();
		}
		return ret;
	}
	/**
	 * Devuelve el primer item que corresponde al nombre
	 * Version especializada que transforma los tags a properties
	 * Cada items del formato  
	 * 		<name>
	 * 			<tag_name>valor</tagName>
	 *      	<tag_name>valor</tagName>
	 *      </name>
	 * se devuelve como una HashMap (tagName,valor)
	 */
	public HashMap<String, String> getItemsAsMap(String name, boolean recursive) {

		XMLTag tag= this.getItem(name,recursive);
		HashMap<String,String> tagMap = new HashMap<String,String>();
		Enumeration<TreeNode> childrenEnum = tag.children();
		while (childrenEnum.hasMoreElements()) {
			XMLTag item = (XMLTag) childrenEnum.nextElement();
			tagMap.put(item.getName(), item.getData());
		}
		return tagMap;
	}



	/**
	 * Devuelve el valor contenido en el tag, o null si no existe el tag
	 * @param name
	 * @return
	 */
	public String getItemData(String name)
	{
		XMLTag tag = this.getItem(name);
		return (tag == null) ? null : tag.getData();
	}


	/**
	 * Devuelve el primer item que corresponde al nombre, con busqueda recursiva
	 * por defecto.
	 */
	public XMLTag getItem(String name) {
		return getItem(name, true);
	}

	/**
	 * Devuelve el primer item que corresponde al nombre
	 */
	public XMLTag getItem(String name, boolean recursive) {

		List<XMLTag> v = getItems(name, recursive);
		return ((v.size() > 0) ? (v.get(0)) : null);
	}

	/**
	 * Devuelve String desde un XML
	 */
	public String getXMLString() throws XMLDataException {
		return getXMLString(true);
	}

	/**
	 * Devuelve String desde un XML (Con o sin header)
	 */
	public String getXMLString(boolean header) throws XMLDataException {

		ByteArrayOutputStream outStream = new ByteArrayOutputStream();
		saveToStream(outStream, header);
		return outStream.toString();
	}

	// -----CLASES
	// AUXILIARES----------------------------------------------------------//

	/***************************************************************************
	 * Reader.
	 */
	public class Reader extends DefaultHandler {

		private XMLTag current = null;
		private List<String> nsDeclarations = new ArrayList<String>();

		public Reader() {
		}

		public XMLTag getTop() {
			return current;
		}
		
		public void addNsDeclaration(String prefix, String uri){
			nsDeclarations.add("\n xmlns:"+ prefix + "=\"" + uri + "\" ");
		}
		
		

		@Override
		public void startPrefixMapping(String prefix, String uri) throws SAXException {
			addNsDeclaration(prefix, uri);
		}

		public void startElement(String uri, String localName, String name,
				Attributes attrs) {

			boolean useNS = isUseNamespaces();

			String useName = useNS ? name : localName;
			XMLTag item = new XMLTag(useName);
			//if (useNS) // TODO : no estaba
			item.setNsUri(uri);

			for (int i = 0; i < attrs.getLength(); i++)
				item.setProperty(attrs.getURI(i), useNS ? attrs.getQName(i)
						: attrs.getLocalName(i) , attrs.getValue(i));

			if (current != null)
				current.add(item);

			current = item;
		}

		public void characters(char[] ch, int start, int length)
				throws SAXException {

			char data[] = new char[length];
			System.arraycopy(ch, start, data, 0, length);
			String itemData = new String(data);
			if (itemData.length() > 0) {
				String prevData = current.getData();
				if (prevData != null) {
					current.setData(prevData + itemData);
				} else {
					current.setData(itemData);
				}
			}
		}

		public void endElement(String uri, String localName, String name) {
			if (current.getParent() != null){
				current = (XMLTag) current.getParent();
				if(isUseNamespaces()) current.getNsDeclarations().addAll(nsDeclarations);
				nsDeclarations.clear();
			}	
		}

	}

	/***************************************************************************
	 * XMLErrorHandler.
	 */
	class XMLErrorHandler implements ErrorHandler {

		public void error(SAXParseException e) {

			Logger log = LogManager.getLogger(XMLData.class);
			log.error("Error en XML parser en " + e.getPublicId() + " ("
					+ e.getLineNumber() + ", " + e.getColumnNumber() + ")", e);
		}

		public void fatalError(SAXParseException e) {

			Logger log = LogManager.getLogger(XMLData.class);
			log.fatal("Error en XML parser en " + e.getPublicId() + " ("
					+ e.getLineNumber() + ", " + e.getColumnNumber() + ")", e);
		}

		public void warning(SAXParseException e) {

			Logger log = LogManager.getLogger(XMLData.class);
			log.warn("Error en XML parser en " + e.getPublicId() + " ("
					+ e.getLineNumber() + ", " + e.getColumnNumber() + ")");
		}

	}

	public boolean isUseNamespaces() {
		return useNamespaces;
	}
	
	/**
	 * Si es true => usamos namespaces
	 * @param namespaceAware
	 */
	public void setUseNamespaces(boolean namespaceAware) {
		this.useNamespaces = namespaceAware;
	}

	public File getResourcesUrl() {
		return resourcesUrl;
	}

	public void setResourcesUrl(File resourcesUrl) {
		this.resourcesUrl = resourcesUrl;
	}
	
	/**
	 * Me devuelve el nombre del tag que le indico con el namespace incluido si es que corresponde
	 * @param tagName
	 * @return
	 */
	private String getXmliTagNameResolvingNamespaces(String tagName){
		String tagResolved = isUseNamespaces() ? PREFIX_XMLI + ":" + tagName : tagName;
		return tagResolved;
	}
}
