package ar.com.sdd.commons.util.db;

import ar.com.sdd.commons.util.ApplicationException;
import ar.com.sdd.commons.util.DateUtil;
import ar.com.sdd.commons.util.StringUtil;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.hibernate.criterion.MatchMode;

import java.io.Serializable;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.text.ParseException;
import java.util.*;
import java.util.stream.Collectors;

public class QueryBuilder implements Serializable {

    private static final long serialVersionUID = 5873238615031399220L;

    static Logger log = LogManager.getLogger(QueryBuilder.class);

    private static final int MAX_PARAMETERS = 50;
    private static final int MAX_IN_ARGUMENTS = 900; //creemos que llega hasta 1000

    private ArrayList parameters = null;
    //private StringBuilder selectPart = null;
    private ArrayList<String> optimizerHints = null;
    private ArrayList<String> selectPartExpr  = null;
    private ArrayList<String> selectPartAlias = null;
    private StringBuilder fromPart = null;
    private StringBuilder wherePart = null;
    private StringBuilder groupByPart = null;
    private StringBuilder havingPart = null;
    private StringBuilder orderByPart = null;
    private ArrayList<QueryBuilder> unions = null;

    public final static String STRCONST_NULL                            = "$$NULL";
    //Para construir un QB hay dos secuencias
    //  1) Construyendolo por partes, para un SELECT normal
    //      qb = new QueryBuilder()
    //	    qb.select(...)
    //      qb.from(...)
    //    	qb.andContition(..)
    //    	qb.groupBy(..)
    //    	qb.orderBy(..)
    // 2) Construyendolo con subQueryBuilder, para un UNION
    //      qb = new QueryBuilder()
    //      qa.addUnion(qb1)
    //      qa.addUnion(qb2)
    // Luego, se usa con:
    //      qb.toString()
    //      qb.createQuery(connection)
    public QueryBuilder() {
        reset();
    }


    public void reset() {
        parameters = new ArrayList();
        optimizerHints = new ArrayList<String>();
        selectPartExpr = new ArrayList<String>();
        selectPartAlias = new ArrayList<String>();
        fromPart = new StringBuilder();
        wherePart = new StringBuilder();
        groupByPart = new StringBuilder();
        orderByPart = new StringBuilder();
        havingPart = new StringBuilder();
        unions = new ArrayList<QueryBuilder>();
    }

    public void resetOrderBy() {
        orderByPart = new StringBuilder();
    }

    public QueryBuilder addUnion( QueryBuilder union) {
        unions.add( union );
        parameters.addAll(union.parameters);
        return this;
    }


    public QueryBuilder addParameter( Object parameter ) {
        if ( parameter instanceof Collection )
            parameters.addAll( (Collection)parameter );
        else
            parameters.add( parameter );
        return this;
    }

    public QueryBuilder andCondition(String condition, Object... parameters) {
        andCondition(condition);
        if (parameters != null) {
            for (Object parameter : parameters) {
                addParameter(parameter);
            }
        }
        return this;
    }

    public QueryBuilder andCondition( String condition  ) {
        if ( wherePart.length() > 0 )
            wherePart.append( " AND " );
        wherePart.append( "(" +  condition + ")" );
        return this;
    }

    /**
     * Agrega la condicion y los parametros por separado para que se vean en color azul en pantalla en la caja de la query en pantallas de reportes sql
     */
    public QueryBuilder andCondition(String field, RangeExpresion rangeExpresion) {
        this.andCondition(rangeExpresion.getSQLFilter(field, false));
        this.addParameters(rangeExpresion.getSQLValues());
        return this;
    }

    public QueryBuilder andEquals( String expression, Object value ) {
        return andCondition( expression + " = ?", value );
    }

    public QueryBuilder andLike( String expression, Object value) {
        return andLike( expression, value, true );
    }

    public QueryBuilder andLike( String expression, Object value, boolean escapeSpecialChars ) {
        if (escapeSpecialChars)
            return andCondition( expression + " LIKE ? ESCAPE '!'", getEscapedLike( value.toString() ) );
        else
            return andCondition( expression + " LIKE ? ",  value.toString() );
    }


    public QueryBuilder andNullOrIn( String expression, Collection values) {
        return andIn(expression, false, values, true);
    }

    public QueryBuilder andIn( String expression, Collection values) {
        boolean orNull=false;
        // Valido si contiene $$NULL, y si esta presente lo quito como parametro
        if (values!=null) orNull = values.remove(STRCONST_NULL);
        return andIn(expression, false, values, orNull );
    }

    public QueryBuilder andNotIn( String expression, Collection values) {
        boolean orNull=false;
        // Valido si contiene $$NULL, y si esta presente lo quito como parametro
        if (values!=null) orNull = values.remove(STRCONST_NULL);
        return andIn(expression, true, values, orNull );
    }


    //Arma una lista exlida de   in ('x','y','z')
    static public String conditionAndIn( String expression, boolean not, Collection values, boolean orNull ) {
        ArrayList valuesArrayList = new ArrayList( values );
        StringBuilder condition = new StringBuilder();

        // Si la lista esta vacia, solo puede valer null en la base
        if ( valuesArrayList.size() == 0 ) {
            condition.append(expression);
            condition.append(" is null ");
            // Si ve todo, ni me precupo
        } else if ( valuesArrayList.contains(StringUtil.LIST_ANY) ) {
            // sino, la condicion de null depende del parametro orNull
        } else if ( valuesArrayList.size() == 1 ) {
            condition.append("(");
            if (orNull) {
                condition.append(expression);
                condition.append(" is null or ");
            }
            condition.append(expression);
            condition.append(not?"!=" : "=");
            condition.append("'"+valuesArrayList.get(0)+"'");
            condition.append(")");
        } else {
            condition.append("(");
            if (orNull) {
                condition.append(expression);
                condition.append(" is null or ");
            }
            condition.append(expression);
            condition.append( not?" NOT ": "");
            condition.append( " IN (");
            Iterator valuesIterator = valuesArrayList.iterator();
            String separator = "";
            while ( valuesIterator.hasNext() ) {
                Object value = (Object) valuesIterator.next();
                condition.append( separator ).append( "'"+value+"'" );
                separator = ",";
            }
            condition.append( ")" ); //lista de valores
            condition.append( ")" ); // or null
        }
        return condition.toString();
    }


    private QueryBuilder andIn( String expression, boolean not, Collection values, boolean orNull ) {
        ArrayList valuesArrayList = new ArrayList( values );

        // Si la lista esta vacia, solo puede valer null en la base
        if ( valuesArrayList.size() == 0 ) {
            StringBuilder condition = new StringBuilder();
            condition.append(expression);
            condition.append(" is null ");
            // Si ve todo, ni me precupo
        } else if ( valuesArrayList.contains(StringUtil.LIST_ANY) ) {
            // sino, la condicion de null depende del parametro orNull
        } else if ( valuesArrayList.size() == 1 ) {
            StringBuilder condition = new StringBuilder();
            condition.append("(");
            if (orNull) {
                condition.append(expression);
                condition.append(" is null or ");
            }
            condition.append(expression);
            condition.append(not?"!=" : "=");
            condition.append("?");
            condition.append(")");
            andCondition( condition.toString(), valuesArrayList.get(0) );
        } else {
            StringBuilder condition = new StringBuilder();
            condition.append("(");
            if (orNull) {
                condition.append(expression);
                condition.append(" is null or ");
            }
            condition.append(expression);
            condition.append( not?" NOT ": "");
            condition.append( " IN (");
            Iterator valuesIterator = valuesArrayList.iterator();
            String separator = "";
            while ( valuesIterator.hasNext() ) {
                Object value = (Object) valuesIterator.next();
                condition.append( separator ).append( "?" );
                addParameter( value );
                separator = ",";
            }
            condition.append( ")" ); //lista de valores
            condition.append( ")" ); // or null
            andCondition( condition.toString() );
        }
        return this;
    }

    /**
     * Agrega una condicion del tipo:
     *      dateField between TO_DATE(optionalDateFrom,'././.') and TO_DATE(optionalDateTo...)
     *
     * Los parametros optional puede ser null, y entonces se agrega la condicion con los rangos abiertos
     * Si ambos parametros son null, no se agrega nada.
     * Las condiciones son inclusive (es decir >= y <= )
     * @param dateField
     * @param optionalDateFrom
     * @param optionalDateTo
     * @return
     */
    private QueryBuilder andFilterDateRange( String dateField, String optionalDateFrom, String optionalDateTo, boolean withNull) throws ParseException {
        //No uso Between para soportar un outerJoin en el primer parametro
        if (!StringUtil.isEmpty(optionalDateFrom)){
            String cond = dateField + " >= " + getDateExpression(optionalDateFrom);
            if (withNull) {
                cond = "(" +cond + " OR "+dateField+" is null)";
            }
            this.andCondition( cond );
        }
        if (!StringUtil.isEmpty(optionalDateTo)){  // ..TO
            String cond = dateField + " <= " + getDateExpression(optionalDateTo);
            if (withNull) {
                cond = "(" +cond + " OR "+dateField+" is null)";
            }
            this.andCondition( cond );
        }
        return this;
    }
    public QueryBuilder andFilterDateRange( String dateField, String optionalDateFrom, String optionalDateTo) throws ParseException {
        return andFilterDateRange( dateField, optionalDateFrom, optionalDateTo,false);
    }
    public QueryBuilder andFilterDateRangeWithNull( String dateField, String optionalDateFrom, String optionalDateTo) throws ParseException {
        return andFilterDateRange( dateField, optionalDateFrom, optionalDateTo,true);
    }
    public QueryBuilder andFilterDateRange( String dateField, Date optionalDateFrom, Date optionalDateTo) throws ParseException {
        String dateFrom = optionalDateFrom!=null?DateUtil.dateToString(optionalDateFrom, "dd/MM/yyyy"):null;
        String dateTo   = optionalDateTo!=null?  DateUtil.dateToString(optionalDateTo  , "dd/MM/yyyy"):null;
        return andFilterDateRange(dateField, dateFrom, dateTo, false);
    }
    public QueryBuilder andFilterDateRangeWithNull( String dateField, Date optionalDateFrom, Date optionalDateTo) throws ParseException {
        String dateFrom = optionalDateFrom!=null?DateUtil.dateToString(optionalDateFrom, "dd/MM/yyyy"):null;
        String dateTo   = optionalDateTo!=null?  DateUtil.dateToString(optionalDateTo  , "dd/MM/yyyy"):null;
        return andFilterDateRange(dateField, dateFrom, dateTo, true);
    }

    public QueryBuilder andFilterDateRange( String dateField, String optionalDateFrom, int deltaFrom, String optionalDateTo, int deltaTo) throws ParseException {
        //No uso Between para soportar un outerJoin en el primer parametro
        if (!StringUtil.isEmpty(optionalDateFrom)){
            this.andCondition( dateField + " >= " + getLiteralDateExpression(optionalDateFrom) + " - " + deltaFrom );
        }
        if (!StringUtil.isEmpty(optionalDateTo)){  // ..TO
            this.andCondition( dateField + " <= " + getLiteralDateExpression(optionalDateTo) + " + " + deltaTo );
        }
        return this;
    }

    /**
     * Agrega una condicion del tipo:
     *      dateField between TO_DATE(optionalDateFrom,'././.') and TO_DATE(optionalDateTo...)
     *
     * Los parametros optional puede ser null, y entonces se agrega la condicion con los rangos abiertos
     * Si ambos parametros son null, no se agrega nada.
     * Las condiciones son inclusive (es decir >= y <= )
     * @param timestampField
     * @param optionalTimestampFrom
     * @param optionalTimestampTo
     * @return
     */
    private QueryBuilder andFilterTimestampRange(String timestampField, String optionalTimestampFrom, String optionalTimestampTo, boolean withNull) throws ParseException {
        //No uso Between para soportar un outerJoin en el primer parametro
        if (!StringUtil.isEmpty(optionalTimestampFrom)) {
            String cond = timestampField + " >= " + getTimestampExpression(optionalTimestampFrom, false);
            if (withNull) {
                cond = "(" + cond + " OR " + timestampField + " is null)";
            }
            this.andCondition(cond);
        }
        if (!StringUtil.isEmpty(optionalTimestampTo)) {  // ..TO
            String cond = timestampField + " <= " + getTimestampExpression(optionalTimestampTo, false);
            if (withNull) {
                cond = "(" + cond + " OR " + timestampField + " is null)";
            }
            this.andCondition(cond);
        }
        return this;
    }

    public QueryBuilder andFilterTimestampRange(String timestampField, String optionalTimestampFrom, String optionalTimestampTo) throws ParseException {
        return andFilterTimestampRange(timestampField, optionalTimestampFrom, optionalTimestampTo, false);
    }

    public QueryBuilder andFilterTimestampRangeWithNull(String timestampField, String optionalTimestampFrom, String optionalTimestampTo) throws ParseException {
        return andFilterTimestampRange(timestampField, optionalTimestampFrom, optionalTimestampTo, true);
    }

    public QueryBuilder andFilterIn( String expression, Collection values) {
        if (values!=null && values.size() > 0) andIn(expression, values);
        return this;
    }

    public QueryBuilder andFilterNotIn( String expression, Collection values) {
        if (values!=null && values.size() > 0) andNotIn(expression, values);
        return this;
    }



    /**
     * Agrega una condicion del tipo:
     *      timestampField between TO_DATE(optionalDateFrom,'././.') and TO_DATE(optionalDateTo...)+1
     *
     * El +1 del dateTo sirve para contemplar que timestamp tiene HH:MM:SS pero el dateTo no
     *
     * Los parametros optional puede ser null, y entonces se agrega la condicion con los rangos abiertos
     * Si ambos parametros son null, no se agrega nada.
     * Las condiciones son inclusive (es decir >= y <= )
     * @param dateField
     * @param optionalDateFrom
     * @param optionalDateTo
     * @return
     */

    public QueryBuilder andFilterTimestampRangeAsDate(String dateField, String optionalDateFrom, String optionalDateTo) throws ParseException {
        if (!StringUtil.isEmpty(optionalDateFrom)){
            if (!StringUtil.isEmpty(optionalDateTo)){     // FROM .. TO
                this.andCondition( dateField + " BETWEEN " + getLiteralDateExpression(optionalDateFrom)
                        + " AND "     + getLiteralDateExpression(optionalDateTo) + "+1"
                );
            } else {                                        // FROM ..
                this.andCondition( dateField + " >= " + getLiteralDateExpression(optionalDateFrom) );
            }
        } else if (!StringUtil.isEmpty(optionalDateTo)){  // ..TO
            this.andCondition( dateField + " <= " + getLiteralDateExpression(optionalDateTo)+ "+1" );
        } else {
            //No agreego nada;
        }
        return this;
    }

    /**
     * Agrega una condicion del tipo:
     *     upper(textField) like '%'||upper(optionalText)||'%'
     *
     * Los parametros optional puede ser null, y entonces no se agrega nada
     *
     * @param textField
     * @param optionalText
     * @return
     */
    public QueryBuilder andFilterLike( String textField, String optionalText) {
        if (!StringUtil.isEmpty(optionalText)) {
            RangeExpresion reField = new RangeExpresion(optionalText.trim());
            this.andCondition(textField, reField);
        }
        return this;
    }

    public QueryBuilder andFilterNotLike( String textField, String optionalText) {
        if (!StringUtil.isEmpty(optionalText)) {
            optionalText = "%"+ getEscapedLike(optionalText.toUpperCase().trim())+"%";
            this.andCondition("UPPER("+textField+") NOT like ? ESCAPE '!'",optionalText);
        }
        return this;
    }

    /**
     * Agrega una condicion del tipo:
     *     field = optionalText
     *
     * Los parametros optional puede ser null, y entonces no se agrega nada
     *
     * @param textField
     * @param optionalText
     * @return
     */
    public QueryBuilder andFilterEquals( String textField, String optionalText) {
        if (!StringUtil.isEmptyNull(optionalText)) {
            this.andEquals(textField,optionalText);
        }
        return this;
    }

    /**
     * Agrega una condicion del tipo:
     *     field_expresion_with_?_argument   pasandole el parametro optionalText
     *
     * Los parametros optional puede ser null, y entonces no se agrega nada
     *
     * @param fieldExpresion
     * @param optionalText
     * @return
     */
    public QueryBuilder andFilterCondition( String fieldExpresion, String optionalText) {
        if (!StringUtil.isEmpty(optionalText)) {
            this.andCondition(fieldExpresion,optionalText);
        }
        return this;
    }


    public QueryBuilder optimizerHint( String hint ) {
        optimizerHints.add(hint);
        return this;
    }

    public QueryBuilder select( String exprList, String alias ) {
        //if ( selectPart.length() > 0 ) selectPart.append( ", " );
        //selectPart.append( exprList );
        //if ( alias != null ) {
        //   selectPart.append( " AS " ).append( alias );
        //}
        selectPartExpr.add( exprList );
        selectPartAlias.add( StringUtil.nonNull(alias).replace(' ', '_'));
        return this;
    }

    public QueryBuilder select( String exprList ) {
        return select( exprList, null );
    }

    public QueryBuilder from( String tableName, String alias ) {
        if ( fromPart.length() > 0 )
            fromPart.append( ", " );
        fromPart.append( tableName );
        if ( alias != null ) {
            fromPart.append( " " ).append( alias );
        }
        return this;
    }

    public QueryBuilder from(QueryBuilder subQuery) {
        this.addParameter(subQuery.getParameters());
        return from("( " + subQuery.toString() + ")");
    }

    public QueryBuilder from( QueryBuilder subQuery, String alias ) {
        this.addParameter(subQuery.getParameters());
        return from("( "+subQuery.toString()+" )", alias);
    }

    public QueryBuilder from( String tableName ) {
        return from(tableName,null);
    }

    public QueryBuilder groupBy(String exprList) {
        if (groupByPart.length() > 0 )
            groupByPart.append(", ");
        groupByPart.append( exprList );
        return this;
    }

    public QueryBuilder orderBy(String exprList) {
        if (orderByPart.length() > 0)
            orderByPart.append(", ");
        orderByPart.append( exprList );
        return this;
    }
    public QueryBuilder having(String havingExpr) {
        if (havingPart.length() > 0)
            havingPart.append(" AND ");
        havingPart.append( havingExpr );
        return this;
    }


    public void addParameters(Collection params){
        parameters.addAll(params);
    }

    public Collection getParameters(){
        return parameters;
    }


    private int createQuerySetParameters(int startIndex, PreparedStatement stmt) throws SQLException {
        int i, n = parameters.size();
        for ( i = 0; i < n; i++ ) {
            stmt.setObject( startIndex+i , parameters.get( i ) );
        }
        return startIndex+i;
    }

    public PreparedStatement createQuery( Connection conn ) throws SQLException {
        PreparedStatement stmt = conn.prepareStatement( toString() );
        //El tratemiento es distinto si tengo o no tengo union
        int startIndex = 1;
        if (unions.size() == 0 ){
            createQuerySetParameters(startIndex,stmt);
        } else {
            for (QueryBuilder union:unions) {
                startIndex = union.createQuerySetParameters(startIndex,stmt);
            }
        }
        return stmt;
    }

    public String toString() {
        StringBuilder result = new StringBuilder();
        //El formato es distinto si tengo o no tengo union
        if (unions.size() == 0 ){
            // select ... from .. where.. group .. sort
            if ( selectPartExpr != null ) {
                result.append( "SELECT " );
                if ( optimizerHints.size() > 0 )  {
                    result.append( "/*+" );
                    for (int i=0; i<optimizerHints.size();i++) {
                        result.append( optimizerHints.get(i)).append(" ");
                    }
                    result.append( "*/" );
                }
                if ( selectPartExpr.size() == 0 )  result.append( "*" );
                else {
                    //result.append( selectPart );
                    for (int i=0; i<selectPartExpr.size();i++) {
                        if (i!=0) result.append( ", ");
                        result.append( selectPartExpr.get(i));
                        if (!StringUtil.isEmpty(selectPartAlias.get(i))) {
                            result.append( " AS ");
                            result.append( selectPartAlias.get(i));
                        }
                    }
                }
                if ( fromPart.length() == 0 )  result.append( " FROM DUAL" );
                else  result.append( " FROM " ).append( fromPart );

                if ( wherePart.length() > 0 )  result.append( " WHERE " ).append( wherePart );
                if (groupByPart.length() > 0)  result.append( " GROUP BY " ).append( groupByPart);
                if (havingPart.length() > 0)   result.append( " HAVING " ).append( havingPart);
                if (orderByPart.length() > 0)  result.append( " ORDER BY " ).append( orderByPart );
            }
        } else {
            //  qb1 union qb2 ....
            String connector = "";
            for (QueryBuilder union:unions) {
                result.append(connector);
                result.append("("+ union.toString()+")");
                connector = " union all " ; //Sino pongo el ALL, puedo eliminar 2 lineas iguales sin querer
            }
            if (orderByPart.length() > 0)  result.append( " ORDER BY " ).append( orderByPart );
        }
        String query = result.toString();
        log.trace(query);
        return query;
    }

    /**
     * Ademas de mostrar la query me muestra el valor de los parametros usados en el preparedStatement
     * @return
     */
    public String toStringWithParameters() {
        StringBuilder sb = new StringBuilder(this.toString());
        for (int i = 0; i < parameters.size(); i++) sb.append(" (").append(i).append(")= ").append(parameters.get(i)).append(" |");
        return sb.toString();
    }

    /**
     * El metodo recibe un String y escapea los caracteres: '!','%','_'.
     * @param likeValue
     * @return
     */
    public static String getEscapedLike(String likeValue) {

        likeValue = likeValue.replaceAll("[!]","!!");
        likeValue = likeValue.replaceAll("[%]","!%");
        likeValue = likeValue.replaceAll("[_]","!_");

        return likeValue;
    }


    /**
     * Escapa los caracteres de un string de forma que sea seguro agregarlo a un query evitando SQL injection.
     *
     * Por ejemplo se puede usar como
     *
     * query = "SELECT * FROM x WHERE name = '" + QueryBuilder.getEscapedString( value ) + "'";
     *
     * @param value
     * @return
     */
    public static String getEscapedString( String value ) {
        value = value.replaceAll("[']","''");

        return value;
    }

    /**
     * Este metodo es una copia de getLiteralDateExpression(dateValue) pero que en vez de poner literalmente la fecha
     * la pasa como parametro y de esta manera aparece en pantalla (en la seccion de la query) el parametro de fecha de azul
     * El otro metodo no hace un 'addParameter' y por ende no se muestra como parametro
     * @param dateValue
     * @return
     * @throws ParseException
     */
    public String getDateExpression(String dateValue) throws ParseException {
        if (StringUtil.isEmptyNull(dateValue)) {
            return null;
        }

        Date date = DateUtil.parseDate(dateValue);
        if (date == null) {
            throw new ParseException("Fecha '" + dateValue + "' invalida", 0);
        }

        dateValue = DateUtil.dateToString( date );
        addParameter(dateValue);

        return "TO_DATE(?, '" + DateUtil.getFormat() + "')";
    }

    /**
     * Devuelve la expresion SQL que expresa una fecha dada en un string
     * @param dateValue representa la fecha, debe estar en el formato standard de la plataforma
     * @return Una expresion del tipo "TO_DATE(.....)" o null si la fecha es vacia
     * @throws ParseException si la fecha no esta en el formato standard de la plataforma
     */
    public static String getLiteralDateExpression( String dateValue ) throws ParseException {
        if ( StringUtil.isEmpty( dateValue ) || dateValue.equalsIgnoreCase( "null" ) )
            return null;

        Date date = DateUtil.parseDate(dateValue);
        if ( date == null )
            throw new ParseException("Fecha '" + dateValue + "' invalida", 0);

        dateValue = DateUtil.dateToString( date );
        return "TO_DATE('" + dateValue + "', '" + DateUtil.getFormat() + "')";

    }

    public String getTimestampExpression(String timestampValue, boolean useLiteral) throws ParseException {
        if (StringUtil.isEmptyNull(timestampValue)) {
            return null;
        }

        Date date = DateUtil.parseDate(timestampValue, DateUtil.FORMAT_TIMESTAMP);
        if (date == null) {
            throw new ParseException("Fecha y hora '" + timestampValue + "' invalida", 0);
        }

        timestampValue = DateUtil.dateToString(date, DateUtil.FORMAT_TIMESTAMP);

        if (useLiteral) {
            return "TO_TIMESTAMP('" + timestampValue + "', '" + DateUtil.ORACLE_DATEHOUR_FORMAT_STRING + "')";
        } else {
            addParameter(timestampValue);
            return "TO_TIMESTAMP(?, '" + DateUtil.ORACLE_DATEHOUR_FORMAT_STRING + "')";
        }
    }

    /**
     * Crea un string de forma "(field = ? OR field = ? OR ... field = ?)",
     * con el tama#o de la coleccion pasada.
     * Si uno de los items es el string "NULL" agrega ademas un  ..OR field IS NULL
     *
     * Luego hay que pasar los items, uno por uno, como parametros con addParameter()
     */
    public static String queryOrCollection(Collection<?> col, String field) {
        return queryOrCollection(col, field, false);
    }

    /**
     * Crea un string de forma "(field = ? OR field = ? OR ... field = ?)",
     * con el tama#o de la coleccion pasada.
     * Si uno de los items es el string "NULL" agrega ademas un  ..OR field IS NULL
     *
     * Luego hay que pasar los items, uno por uno, como parametros con addParameter()
     */
    public static String queryOrCollection(Collection<?> col, String field, boolean addParenthesis) {
        StringBuilder query = new StringBuilder("(");
        String and   = "";
        boolean hasNull = false;

        if (isNullParamCollection(col)) {
            hasNull = true;
        } else {
            for (Object colElem : col) {
                query.append(and);
                if (addParenthesis) query.append(" ( ");
                query.append(field).append(" = ?");
                if (addParenthesis) query.append(" ) ");
                and = " OR ";
                hasNull = hasNull || (colElem == null || "NULL".equalsIgnoreCase(colElem.toString()));
            }
        }

        if (hasNull) {
            query.append(and);
            if (addParenthesis) query.append(" ( ");
            query.append(field).append(" IS NULL");
            if (addParenthesis) query.append(" ) ");
        }

        query.append(")");

        return query.toString();
    }


    /**
     * Arma una query con OR, o con IN dependiendo del tamaño de la collection pasada <br>
     * Es importante que la collection pasada sea de Strings. Usar DocumentHelper#getIdStringCollection para asegurarse que se pasan los ids en String
     * @param col Collection de Strings representando los ids
     * @param field el campo por el cual se construye el OR o el IN
     * @return El string armado con la query por OR o IN
     */
    public static String queryOrCollectionWithMaxParameters(Collection<String> col, String field) {
        if (col.size() < MAX_PARAMETERS) {
            return queryOrCollection(col, field);
        } else {
            return queryINCollection(col, field);
        }
    }

    public static int addParametersToCollectionWithMaxParameters(Collection<Long> colId, PreparedStatement stmt, int i, boolean debugAll, StringBuilder debugParameters, String debugParameterSeparator) throws SQLException {
        if (colId.size() < MAX_PARAMETERS) {
            for (Long id : colId) {
                stmt.setLong(i++, id);
                if (debugAll) debugParameters.append(debugParameterSeparator).append(i-1).append(": ").append(id);
            }
        } else {
            //No hago nada porque los parametros se agregaron con IN
        }
        return i;
    }
    public static int addParametersToCollectionWithMaxParameters(Collection<Long> colId, PreparedStatement stmt, int i, boolean debugAll, List<Pair<String, Class<?>>> debugParameters, String debugParameterSeparator) throws SQLException {
        if (colId.size() < MAX_PARAMETERS) {
            for (Long id : colId) {
                stmt.setLong(i++, id);
                if (debugAll) debugParameters.add(Pair.of(id.toString(), Long.class));
            }
        } else {
            //No hago nada porque los parametros se agregaron con IN
        }
        return i;
    }

    /**
     * Crea un string de forma "(field != ? AND field != ? AND ... field != ?)",
     * con el tama#o de la coleccion pasada.
     *
     * Luego hay que pasar los items, uno por uno, como parametros con addParameter()
     */
    public static String queryNorCollection(Collection col, String field) {

        String query = "(";
        String and   = "";

        for (int i = 0; i < col.size(); i++) {

            query += and + field + " != ?";
            and = " AND ";

        }

        query += ")";

        return query;
    }

    public static String queryINCollection(String[] col, String field) {
        List<String> asList = Arrays.stream(col).collect(Collectors.toList());
        return queryINCollection(asList, field);
    }

    public static String queryNotINCollection(String[] col, String field) {
        StringBuilder query = new StringBuilder();
        String sep  = " ";
        query.append("(");
        query.append(field).append(" not in (");
        int count = 0;
        for (String aCol : col) {
            count++;
            if (count == MAX_IN_ARGUMENTS) {
                count = 0;
                query.append(" ) or ").append(field).append(" not in (");
                sep = "";
            }
            query.append(sep).append("'").append(aCol).append("'");
            sep = ",";
        }
        query.append(")");
        query.append(")");
        return query.toString();
    }

    public static String queryINCollection(Collection<String> col, String field) {
        StringBuilder query = new StringBuilder();
        String sep  = " ";
        query.append("(");
        query.append(field).append(" in (");
        int count = 0;
        for (String aCol : col) {
            count++;
            if (count == MAX_IN_ARGUMENTS) {
                count = 0;
                query.append(" ) or ").append(field).append(" in (");
                sep = "";
            }
            query.append(sep).append("'").append(aCol).append("'");
            sep = ",";
        }
        query.append(")");
        query.append(")");
        return query.toString();
    }
    public static String queryINCollectionLong(Collection<Long> col, String field) {

        StringBuilder query = new StringBuilder();
        String sep = " ";
        query.append("(");
        query.append(field).append(" in (");
        int count = 0;
        for (Long aCol : col) {
            count++;
            if (count == MAX_IN_ARGUMENTS) {
                count = 0;
                query.append(" ) or ").append(field).append(" in (");
                sep = "";
            }
            query.append(sep).append(aCol);
            sep = ",";
        }
        query.append(")");
        query.append(")");
        return query.toString();
    }


    /*
     * Consturye un par de opciones comodas:
     *     x  =>    like '%x%'
     *   ! x  =>  ! like '%x%'
     *   a,b  =>     in (a,b)
     *   !a,b =>  !  in (a,b)
     */
    public static String queryExpresionFilter(String expresion, String field) {
        String query = "";
        if (!StringUtil.isEmpty(expresion)) {
            expresion = expresion.trim();
            if (expresion.startsWith("!")) {
                query += " NOT ";
                expresion = expresion.substring(1);
            }
            String expr[] = StringUtil.split(expresion,',');
            if (expr.length==1) {
                query += field + " LIKE '%" + QueryBuilder.getEscapedLike( expr[0] )+ "%' ESCAPE '!'" ;
            } else 	{
                query += QueryBuilder.queryINCollection(expr, field);
            }
        }
        return query;
    }




    /** Operaciones mas complejas
     *
     *  Quiero tansformar un qb en uno que incluya ciertas lineas sumarizadas y otras en detalles
     *  Voy a llevar algo de la forma
     *       SELECT a,b,c
     *       FROM  F
     *       WHERE w
     *       GROUP BY g
     *       SORT BY s
     *  a la forma
     *       SELECT *
     *       FROM (
     *               SELECT max(a),max(b),max(c)
     *               FROM   f
     *               WHERE  W
     *                 AND pivot <= value
     *               GROUP BY g (sacando la-columna-pivot)
     *              UNION
     *               SELECT a,b,c
     *               FROM   f
     *               WHERE  W
     *                 AND pivot > value
     *               GROUP BY g (deberia tener la columan pivot)
     *            )
     *       SORT BY s (que-deberia-tener-la-columna-pivot)
     *
     * Version 1:
     *    solo funciona si la columna pivot fue agregada no-primera en la lista de groupBy, porque remueve la ', ' que la precede
     * @throws ParseException
     * **/

    public QueryBuilder transformSumarizeOnPivotDate(String sumarizeGroup, String pivot, String pivotDateValue, Map<String, Object> agregateMap ) throws ParseException {

        if (StringUtil.isEmpty(pivotDateValue)) {
            return this;
        };
        QueryBuilder qba = new QueryBuilder();
        qba.optimizerHints  = new ArrayList<String>(this.optimizerHints);
        qba.selectPartExpr  = new ArrayList<String>(this.selectPartExpr);
        qba.selectPartAlias = new ArrayList<String>(this.selectPartAlias);

        for (Map.Entry<String, Object> entry: agregateMap.entrySet()) {
            int i = qba.selectPartAlias.indexOf(entry.getKey());
            if (i==-1) {
                log.error("Alias inexistente:" + entry.getKey());
            } else {
                //Veo si es un valor fijo o no
                if (((String)entry.getValue()).startsWith("FIX")) {
                    String[] fix = ((String)entry.getValue()).split("[:]");
                    if (fix[0].equalsIgnoreCase("FIX_STRING")) {
                        qba.selectPartExpr.set(i, "'" + fix[1] + "'");
                    } else if (fix[0].equalsIgnoreCase("FIX_INTEGER")) {
                        qba.selectPartExpr.set(i, fix[1]);
                    }
                } else {
                    qba.selectPartExpr.set(i,"max('"+entry.getValue()+"')");
                }


                //Me fijo si viene un numero, u otra cosa para determinar si le ponemos comillas
    			/*
    			if (entry.getValue() instanceof Number) {
    				qba.selectPartExpr.set(i,"max("+entry.getValue()+")");
    			} else {
    				qba.selectPartExpr.set(i,"max('"+entry.getValue()+"')");
    			}
    			*/
            }
        }
        qba.fromPart   = new StringBuilder(this.fromPart);
        qba.wherePart  = new StringBuilder(this.wherePart);
        qba.parameters.addAll(this.parameters);
        qba.andCondition(pivot +" <= " +  getLiteralDateExpression(pivotDateValue) );
        qba.groupByPart = new StringBuilder(sumarizeGroup);
        qba.havingPart = new StringBuilder(this.havingPart);


        QueryBuilder qbb = new QueryBuilder();
        qbb.optimizerHints  = new ArrayList<String>(this.optimizerHints);
        qbb.selectPartExpr  = new ArrayList<String>(this.selectPartExpr);
        qbb.selectPartAlias = new ArrayList<String>(this.selectPartAlias);
        qbb.fromPart   = new StringBuilder(this.fromPart);
        qbb.wherePart  = new StringBuilder(this.wherePart);
        qbb.parameters.addAll(this.parameters);
        qbb.andCondition(pivot +" >  "+ getLiteralDateExpression(pivotDateValue) );
        qbb.groupByPart = new StringBuilder(this.groupByPart);
        qbb.havingPart = new StringBuilder(this.havingPart);

        QueryBuilder qb = new QueryBuilder();
        qb.addUnion(qba);
        qb.addUnion(qbb);
        qb.orderByPart = this.orderByPart;
        return qb;
    }



    //Recibe
    //  query :  select * from ... where a=? and b=?
    //  dbugParams:  |1:xxx|2:xxx
    public static String getSQL(String query, List<Pair<String, Class<?>>> debugParams) {
        while (query.contains("  ")) {
            query = query.replaceAll(" +", " ");
        }
        for (Pair<String, Class<?>> paramPair : debugParams) {
            String param = paramPair.getKey();
            Class<?> paramClass = paramPair.getValue();

            //Si el parametro es empty, lo tengo que poner igual. Aca no me importa. Solo saco el caso null, por las dudas
            param=StringUtil.nonNull(param);

            if (paramClass.equals(Timestamp.class)) {
                // && param.matches("^\\d{4}-\\d{2}-\\d{2} \\d{2}:\\d{2}:\\d{2}\\.\\d+")
                param = param.replaceAll("\\.\\d+", ""); // Eliminar milisegundos
                query = query.replaceFirst("[?]", "TO_TIMESTAMP('" + param.trim() + "', 'YYYY-MM-DD HH24:MI:SS')");
            } else if (paramClass.equals(Date.class)) {
                query = query.replaceFirst("[?]", "TO_DATE('" + param.trim() + "', 'YYYY-MM-DD')");
            } else if (paramClass.equals(Integer.class) || paramClass.equals(Long.class) || paramClass.equals(Double.class)) {
                query = query.replaceFirst("[?]", param.trim());
            } else if (paramClass.equals(String.class)){
                query = query.replaceFirst("[?]", "'" + param.trim() + "'");
            } else {
                throw new ApplicationException("No se reconoce el tipo de parametro [" + paramClass + "]");
            }

            /* Si tuviera parametos tipo objeto (y no strings) podria hacer esto para cada uno
            if (parameter == null) {
                return "NULL";
            } else {
            if (parameter instanceof String) {
                return "'" + ((String) parameter).replace("'", "''") + "'";
            } else if (parameter instanceof Timestamp) {
                return "to_timestamp('" + new SimpleDateFormat("MM/dd/yyyy HH:mm:ss.SSS").
                        format(parameter) + "', 'mm/dd/yyyy hh24:mi:ss.ff3')";
            } else if (parameter instanceof Date) {
                return "to_date('" + new SimpleDateFormat("MM/dd/yyyy HH:mm:ss").
                        format(parameter) + "', 'mm/dd/yyyy hh24:mi:ss')";
            } else if (parameter instanceof Boolean) {
                return ((Boolean) parameter).booleanValue() ? "1" : "0";
            } else {
                return parameter.toString();
            }
        }
            */
        }
        return query;
    }

    public static Pair<String, MatchMode> resolveLikeParam(String value) {
        MatchMode matchMode = MatchMode.ANYWHERE;
        if (value.startsWith("%")) {
            value = value.substring(1);
            matchMode = MatchMode.END;
        }

        if (value.endsWith("%")) {
            value = value.substring(0, value.length() - 1);
            matchMode = matchMode.equals(MatchMode.END) ? MatchMode.ANYWHERE : MatchMode.START;
        }

        return Pair.of(value, matchMode);
    }

    public static boolean isNullParamCollection(Collection<?> params) {
        if (params != null && params.size() == 1) {
            Object paramValue = params.iterator().next();
            return paramValue == null || "NULL".equalsIgnoreCase(paramValue.toString());
        }

        return false;
    }
}