package ar.com.sdd.commons.util;

import org.apache.commons.compress.archivers.ArchiveEntry;
import org.apache.commons.compress.archivers.ArchiveException;
import org.apache.commons.compress.archivers.ArchiveInputStream;
import org.apache.commons.compress.archivers.ArchiveStreamFactory;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.compressors.gzip.GzipCompressorInputStream;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.vfs.FileObject;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.*;
import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.RoundingMode;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.*;
import java.util.regex.Pattern;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

@SuppressWarnings("unused")
public class FileToolkit {

	private static final int BUFFER_SIZE = 1024;
	private static final Logger log = LogManager.getLogger(FileToolkit.class);
	public static final String IMAGE_TYPE_JPG = "jpg";
	public static final String IMAGE_TYPE_PNG = "png";
    public static final String SCRIPT_BARCODE_GENERATOR_FILENAME = "barcodegenerator.sh";

    public static final String FILE_TYPE_XLS 	= "xls";	public static final String FILE_EXTENSION_XLS	 = FilenameUtils.EXTENSION_SEPARATOR + FILE_TYPE_XLS;
    public static final String FILE_TYPE_XLSX 	= "xlsx";	public static final String FILE_EXTENSION_XLSX	 = FilenameUtils.EXTENSION_SEPARATOR + FILE_TYPE_XLSX;
    public static final String FILE_TYPE_PDF 	= "pdf";	public static final String FILE_EXTENSION_PDF	 = FilenameUtils.EXTENSION_SEPARATOR + FILE_TYPE_PDF;
    public static final String FILE_TYPE_CSV 	= "csv";   	public static final String FILE_EXTENSION_CSV	 = FilenameUtils.EXTENSION_SEPARATOR + FILE_TYPE_CSV;
    public static final String FILE_TYPE_JSON 	= "json"; 	public static final String FILE_EXTENSION_JSON	 = FilenameUtils.EXTENSION_SEPARATOR + FILE_TYPE_JSON;
    public static final String FILE_TYPE_ZIP 	= "zip";   	public static final String FILE_EXTENSION_ZIP	 = FilenameUtils.EXTENSION_SEPARATOR + FILE_TYPE_ZIP;
    public static final String FILE_TYPE_TAR_GZ = "tar.gz"; public static final String FILE_EXTENSION_TAR_GZ = FilenameUtils.EXTENSION_SEPARATOR + FILE_TYPE_TAR_GZ;
    public static final String FILE_TYPE_TGZ 	= "tgz";   	public static final String FILE_EXTENSION_TGZ	 = FilenameUtils.EXTENSION_SEPARATOR + FILE_TYPE_TGZ;
    public static final String FILE_TYPE_GZIP 	= "gz";   	public static final String FILE_EXTENSION_GZIP	 = FilenameUtils.EXTENSION_SEPARATOR + FILE_TYPE_GZIP;
    public static final String FILE_TYPE_TXT 	= "txt";   	public static final String FILE_EXTENSION_TXT	 = FilenameUtils.EXTENSION_SEPARATOR + FILE_TYPE_TXT;
    public static final String FILE_TYPE_XML 	= "xml";   	public static final String FILE_EXTENSION_XML	 = FilenameUtils.EXTENSION_SEPARATOR + FILE_TYPE_XML;

	public enum CompressionFormat {
		ZIP,
		GZIP
	}

	/**
	 * Crea un archivo vacio llamado <b>filepath</b> (incluye directorio).
	 *
	 * @param filepath Path completo al archivo.
	 */
	public static void createFile(String filepath) throws IOException {
		createFile(filepath, "");
	}

	/**
	 * Crea un archivo llamado <b>filepath</b> (incluye directorio).
	 * Le agrega contenido.
	 *
	 * @param filepath Path completo al archivo.
	 * @param content Contenido del archivo.
	 */
	public static void createFile(String filepath, String content) throws IOException {
        File theFile = new File(filepath);
        if (theFile.exists()) {
            theFile.delete();
        } else {
			checkDirectory(filepath);
		}
        theFile.createNewFile();
        FileWriter writer = new FileWriter(theFile);
        writer.write(content);
        writer.flush();
        writer.close();
    }

    /**
     * Crea un archivo llamado <b>filepath</b> (incluye directorio)
     * y lo llena con el contenido de <b>content</b>.
     */
    public static void appendToFile(String filepath, String content) throws IOException {
		File theFile 	  = new File(filepath);
		FileWriter writer = new FileWriter(theFile, true);
		writer.write(content);
		writer.flush();
		writer.close();
    }

    /**
     * Lee completamente un archivo llamado <b>fileName</b> en el directorio <b>directory</b>.
     * No debe ser usado para archivos grandes porque el String ocuparia mucho.
     */
    public static String readFile(String directory, String fileName) throws IOException {
        File theFile 	  = new File(directory, fileName);
        FileReader reader = new FileReader(theFile);

        char[] buff = new char[0xFFFF];
        int leidos 	= reader.read(buff);
        String datosLeidos = "";

        while (leidos > 0) {
            datosLeidos += new String(buff, 0, leidos);
            leidos = reader.read(buff);
        }

        return datosLeidos;
    }

    /**
     * Variante que lee un file en UTF-8. Deberia ser lo normal
     */
    public static String readFileUTF8(String fileName)  throws IOException {
    	 FileInputStream fis = new FileInputStream(fileName);
         BufferedReader reader = new BufferedReader(new InputStreamReader(fis, StandardCharsets.UTF_8));

         char[] buff = new char[0xFFFF];
         int leidos 	= reader.read(buff);
         String datosLeidos = "";
         while (leidos > 0) {
             datosLeidos += new String(buff, 0, leidos);
             leidos = reader.read(buff);
         }
         reader.close();
         return datosLeidos;
    }

    /**
      * Devuelve todos los archivos de un directorio <b>dir</b>.
      * <p>
      * Nota: este metodo deberia en un futuro ser reimplementado para que pueda
      * manejar wildcharacters.
      */
    public static File[] getDirectory(String dir, final boolean includeDirectories) throws IOException {
        return getDirectory(dir, "", "", includeDirectories);
    }

	public static void checkDirectory (String  fullFileName) {
		File pathFileName = new File(getPathOnly(fullFileName));
		if (!pathFileName.exists()) {
			boolean success = pathFileName.mkdirs();
			if (!success) {
				throw new ApplicationException("No se pudo crear la estructura de directorios para [" + fullFileName + "]");
			}
		}
	}
    public static void sortFileObjetList(List<FileObject> sibList, String readOrder){
    	//Ordeno la lista
		if (readOrder.equalsIgnoreCase("LIFO")) {
			Collections.sort(sibList,
				new Comparator<FileObject>() {
			    	public int compare(FileObject o1, FileObject o2) {
		    			return o2.getName().getBaseName().compareTo(o1.getName().getBaseName());
			    	}});
		} else {
			Collections.sort(sibList,
					new Comparator<FileObject>() {
				    	public int compare(FileObject o1, FileObject o2) {
			    			return o1.getName().getBaseName().compareTo(o2.getName().getBaseName());
				    	}});
		}
    }
    public static void sortFileList(List<File> sibList, String readOrder){
    	//Ordeno la lista
		if (readOrder.equalsIgnoreCase("LIFO")) {
			Collections.sort(sibList,
				new Comparator<File>() {
			    	public int compare(File o1, File o2) {
		    			return o2.getName().compareTo(o1.getName());
			    	}});
		} else {
			Collections.sort(sibList,
					new Comparator<File>() {
				    	public int compare(File o1, File o2) {
			    			return o1.getName().compareTo(o2.getName());
				    	}});
		}
    }
    public static boolean matchWildcard( String wildcard, String  fileBaseName ) {
		String wildRegExp =
			"^" +
			wildcard
				.replaceAll( "\\.", "\\." )
				.replaceAll( "\\*", ".*" )
				.replaceAll( "\\?", "." ) +
			"$";
		return fileBaseName.matches( wildRegExp );
	}

    public static File getFileByPattern(String pathWithPatern, String readOrder ) {
    	File file = new File( pathWithPatern );
		String fileDir = file.getParent();
		final String searchPattern = file.getName(); 
		
		FileFilter ff = new FileFilter() {
			public boolean accept(File pathname) {
				return FileToolkit.matchWildcard(searchPattern,pathname.getName());
			}
		};
		File f = new File(fileDir);
		File result=null;
		File[] listFiles = f.listFiles(ff);
		if (listFiles != null) {
			List<File> files = Arrays.asList(listFiles);
			sortFileList(files, readOrder);
	        for (Iterator<File> filesIter = files.iterator() ; filesIter.hasNext()&& result==null;) {
				result = filesIter.next();
	        	String baseName = result.getName();
                log.info("initializeStream: probando el matching con: {}", baseName);
			}
		}
        return result;
	}

	/**
	 * Devuelve todos los archivos de un directorio <b>dir</b>.
	 * Por el momento se filtra solo por comienzo y fin del nombre del archivo.
	 * Sirve para extensiones, por ejemplo.
	 * Si no se quiere filtrar, usar "" en <b>startsWith</b> and <b>endsWith</b>
	 * <p>
	 * Nota: este metodo deberia en un futuro ser reimplementado para que pueda
	 * manejar wildcharacters.
	 */
	public static File[] getDirectory(String dir, final String startsWith, final String endsWith, final boolean includeDirectories) throws IOException {
		FileFilter ff = new FileFilter() {

			public boolean accept(File pathname) {

				String filename = pathname.getName();

				if (includeDirectories || !includeDirectories && !pathname.isDirectory()) {

					if (filename.startsWith(startsWith)	&& filename.endsWith(endsWith)) {
						return true;
					} else {
						return false;
					}

				} else {
					return false;
				}
			}
		};

		File f = new File(dir);
		return f.listFiles(ff);
	}

    public static long copyStream( InputStream is, OutputStream os ) throws IOException {
        long written = 0;
        byte[] buff = new byte[ 1024 * BUFFER_SIZE ];
        int leidos  = is.read( buff );

        while (leidos > 0) {
            os.write( buff, 0, leidos );
            written += leidos;
            leidos = is.read( buff );
        }

        return written;
    }

	/**
	 * Mueve o copia un file de un path a otro. Agrega un prefijo y postfijo en el contenido.
	 * Borra si todo anduvo bien.
	 *
	 * @param file El archivo a copiar/mover.
	 * @param toFullPath Path destino de la copia.
	 * @param contentPrefix Contenido a "prefijar" en la copia.
	 * @param contentSufix Contenido a "sufijar" en la copia.
	 * @param deleteOnSuccess Indica si se debe borrar el archivo original, i.e. "moverlo".
	 *
	 * @return true en caso de exito
	 */
	public static boolean copyFile(File file, String toFullPath, String contentPrefix, String contentSufix, boolean deleteOnSuccess, boolean compressOnSuccess) throws IOException {
		checkDirectory(toFullPath);

		File toFile = new File(toFullPath);
		toFile.createNewFile();

		FileOutputStream writer = new FileOutputStream(toFile);
		long contentLength = 0;
		if (contentPrefix != null) {
			byte[] fixBuff = contentPrefix.getBytes();
			writer.write(fixBuff);
			contentLength += fixBuff.length;

		}

		FileInputStream reader = new FileInputStream(file);
        copyStream( reader, writer );

		if (contentSufix != null) {

			byte[] fixBuff = contentSufix.getBytes();
			writer.write(fixBuff);
			contentLength += fixBuff.length;

		}

		writer.flush();
		writer.close();
		reader.close();

		//Chequeo que este OK, por exists
		if (toFile.exists() && (file.length() + contentLength) == toFile.length()) {

			if (deleteOnSuccess) file.delete();  // Borra el archivo original

			if (compressOnSuccess){
				ArrayList filesToAdd = new ArrayList();
				filesToAdd.add(toFile);
				compressFiles(toFullPath,filesToAdd,true);
			}

			return true;

		}

		return false;
	}

	public static boolean copyFile(File file, String toFullPath, String contentPrefix, String contentSufix, boolean deleteOnSuccess) throws IOException {
		return copyFile(file,toFullPath,contentPrefix,contentSufix,deleteOnSuccess,false);
	}

	/**
	 * Mueve o copia un file de un path a otro. Agrega un prefijo y postfijo en el contenido.
	 * Borra si todo anduvo bien.
	 *
	 * @param fromFullPath Path origen de la copia.
	 * @param toFullPath Path destino de la copia.
	 * @param contentPrefix Contenido a "prefijar" en la copia.
	 * @param contentSufix Contenido a "sufijar" en la copia.
	 * @param deleteOnSuccess Indica si se debe borrar el archivo original, i.e. "moverlo".
	 *
	 * @return true en caso de exito
	 */
	public static boolean copyFile(String fromFullPath, String toFullPath, String contentPrefix, String contentSufix, boolean deleteOnSuccess, boolean compressOnSuccess) throws IOException {
		return copyFile(new File(fromFullPath),toFullPath,contentPrefix,contentSufix,deleteOnSuccess,compressOnSuccess);
	}

	public static boolean copyFile(String fromFullPath, String toFullPath, String contentPrefix, String contentSufix, boolean deleteOnSuccess) throws IOException {
		return copyFile(new File(fromFullPath),toFullPath,contentPrefix,contentSufix,deleteOnSuccess,false);
	}

	/**
	 * Variantes de copyFile: mover archivo sin agregar prefijo ni sufijo.
	 * Borra el archivo original.
	 */
	public static boolean moveFile(String fromFullPath, String toFullPath,boolean compressTarget) throws IOException {
		return moveFile(new File(fromFullPath), toFullPath,compressTarget);
	}

	public static boolean moveFile(String fromFullPath, String toFullPath) throws IOException {
		return moveFile(new File(fromFullPath), toFullPath,false);
	}

	public static boolean moveFile(File file, String toFullPath, boolean compressTarget) throws IOException {
		return copyFile(file, toFullPath, null, null, true,compressTarget);
	}

	public static boolean moveFile(File file, String toFullPath) throws IOException {
		return copyFile(file, toFullPath, null, null, true,false);
	}

	/**
	 * Mueve o copia un file de un path a otro. Agrega un prefijo y postfijo en
	 * el contenido. Borra si todo anduvo bien.
	 *
	 * Realiza sustituciones de bytes por cadenas de bytes, segun se indique
	 * en los parametros (ejemplo de sustitucion: '&' --> '&amp;')
	 *
	 * @param fromFullPath Path origen de la copia.
	 * @param toFullPath Path destino de la copia.
	 * @param contentPrefix Contenido a "prefijar" en la copia.
	 * @param contentSufix Contenido a "sufijar" en la copia.
	 * @param origChars chars a reemplazar en la copia.
	 * @param replChars chars con los que se reemplazan.
	 * @param encoding El encoding en el que se debe leer el archivo (recom. UTF-8)
	 * @param deleteOnSuccess Indica si se debe borrar el archivo original, i.e. "moverlo".
	 *
	 * @return true en caso de exito
	 */
	public static boolean copyFileReplaceBytes(	String fromFullPath,
												String toFullPath,
												String contentPrefix,
												String contentSufix,
												char[] origChars,
												char[] replChars,
												String encoding,
												boolean deleteOnSuccess)
								throws IOException {

		boolean copySuccess = true;

		File fromFile 	   = new File(fromFullPath);
		File toFile   	   = new File(toFullPath);

		checkDirectory(toFullPath);
		toFile.createNewFile();
		FileOutputStream writer = new FileOutputStream(toFile);

		if (contentPrefix != null) {

			byte[] fixBuff = contentPrefix.getBytes();
			writer.write(fixBuff);

		}

		FileInputStream reader = new FileInputStream(fromFile);
		//byte[] buff = new byte[1024*200];
		byte[] buff = new byte[1024 * BUFFER_SIZE];
		int leidos 	= reader.read(buff);

		while (leidos > 0) {

			// Realiza sustituciones y graba en el output file
			replaceBytesAndWrite(	writer,buff, leidos, origChars,replChars,encoding);

			//writer.write(buff, 0, leidos);

			// Lee el siguiente bloque de bytes del input
			leidos = reader.read(buff);

		}

		if (contentSufix != null) {

			byte[] fixBuff = contentSufix.getBytes();
			writer.write(fixBuff);
		}

		writer.flush();
		writer.close();
		reader.close();

		// Verifica que se copio la totalidad del archivo
		if (toFile.exists()) {

			if (deleteOnSuccess) fromFile.delete();  // Borra el archivo original
			copySuccess = true;

		} else {
			copySuccess = false;
		}

		return copySuccess;
	}

	/**
	 * Devuelve la extension de un filename. El nombre recibido puede contener
	 * el path completo o ser solo el filename.
	 * Si se pasa null, devuelve null
	 * Si se pasa un file sin extension, devuelve vacio ("")
	 */
	public static String getFileExtension(String fileName) {
		if (fileName == null) {
			return null;
		}
		int lastSep = fileName.lastIndexOf(".");
		if (lastSep != -1) {
			return fileName.substring(lastSep + 1);			
		} else {
			return "";
		}
	}
	
	/**
	 * Devuelve a partir de un path el nombre del filename.
	 * Si se pasa null, devuelve null 
	 */
	public static String getFileName(String fileName) {
		if (fileName == null) {
			return null;
		}
		int lastSep = fileName.replace('\\', '/').lastIndexOf("/");
		if (lastSep != -1) {
			return fileName.substring(lastSep + 1);			
		} else {
			return fileName;
		}
	}
	
	/**
	 * Elimina la extension del archivo, eliminando todo lo que haya a la derecha del ultimo punto
	 * @param fileName
	 * @return
	 */
	public static String getFileNameWithoutExtension(String fileName) {
		return getFileNameWithoutExtension(fileName, true);
	}

	public static String getFileNameWithoutExtension(String fileName, boolean removePath) {
		if (removePath) {
			fileName = getFileName(fileName);
		}

		if (fileName.lastIndexOf('.') > -1) {
			fileName =  fileName.substring(0, fileName.lastIndexOf('.'));
		}
		return fileName;
	}

	public static String getPathOnly(String fileName) {
		return getPathOnly(fileName, false);
	}
	/**
	 * Extrae del archivo la ruta, sin el nombre 
	 *   Ej: C:\\Temp\\universe.jpg -> C:\\Temp
	 *       /home/pepe/asd.txt -> /home/pepe
	 * Si se especifica el parametro addLastBar en true, agrega la ultima barra
	 *   Ej: C:\\Temp\\universe.jpg -> C:\\Temp\\
	 *       /home/pepe/asd.txt -> /home/pepe/
	 * @param fullPath
	 * @return
	*/
	public static String getPathOnly(String fullPath, boolean addLastBar) {
		//Prueba distintas combinacines de separador de archivos
		if (fullPath == null) {
			return null;
		}
		int lastSep = fullPath.replace('\\', '/').lastIndexOf("/");
		if (lastSep != -1) {
			if (addLastBar) {
				lastSep++;
			}
			return fullPath.substring(0, lastSep);
		} else {
			return fullPath;
		}
		
	}
	
	/**
	 * Le cambia la extension a un fileName.
	 * @param fileName
	 * @param targetExtension
	 * @return
	 */
	public static String changeExtension(String fileName, String targetExtension) {
		if (fileName.lastIndexOf('.') > -1) {
			fileName = fileName.substring(0, fileName.lastIndexOf('.')); 
		}
		return fileName + '.' + targetExtension;
	}
	 
	//----------------------------------------------------------------------------------//

	/**
	 * Realiza la sustitucion de bytes y escribe el resultado en el output file.
	 *
	 * @param writer Writer del file.
	 * @param bytesRead Array de bytes leidos del input, y que se estan procesando.
	 * @param leidos 
	 * @param origChars Array de chars a reemplazar.
	 * @param replChars Array chars que reemplazan los origChars.
	 * @param encoding El encoding en el que se debe leer el archivo (recom. UTF-8)
	 */
	private static long replaceBytesAndWrite(FileOutputStream writer, byte[] bytesRead, int leidos, char[] origChars, char[] replChars, String encoding) throws IOException {
		String st = StringUtil.replaceChars(new String(bytesRead, 0, leidos, encoding), origChars, replChars);
		byte[] bytesToWrite = st.getBytes(encoding);
		writer.write(bytesToWrite);
		return bytesToWrite.length;
	}

	/**
	 * Comprime un conjunto de archivos
	 * @param fileToCreate
	 * @param filesToAdd
	 */
	public static File compressFiles(String fileToCreate,Collection filesToAdd,boolean deleteAddedFiles){
		return compressFiles(fileToCreate, filesToAdd, deleteAddedFiles, true);
	}

	public static File compressFiles(String fileToCreate,Collection filesToAdd,boolean deleteAddedFiles, boolean addZipExtension){
			String extension = "";
			if (addZipExtension) {
				if (fileToCreate.endsWith(".")) extension = "zip";
				else extension = ".zip";

			}
			File zipFileToCreate = new File(fileToCreate + extension);
			try{

			  	FileOutputStream dest = new FileOutputStream(zipFileToCreate);
		        ZipOutputStream out = new ZipOutputStream(new BufferedOutputStream(dest));
		        // out.setMethod(ZipOutputStream.DEFLATED);

		        BufferedInputStream origin = null;
		        byte data[] = new byte[1024 * BUFFER_SIZE];
		        Iterator itFiles = filesToAdd.iterator();
		        while(itFiles.hasNext()) {
		           File actualFile = (File)itFiles.next();
                    log.debug("Agregando al zip: {}", actualFile);
		           FileInputStream fi = new FileInputStream(actualFile);
		           origin = new BufferedInputStream(fi, 1024 * BUFFER_SIZE);
		           // aca realmente se mete el file en el zip
		           ZipEntry entry = new ZipEntry(actualFile.getName());
		           out.putNextEntry(entry);
		           int count;
		           while((count = origin.read(data, 0, 1024 * BUFFER_SIZE)) != -1) {
		              out.write(data, 0, count);
		           }
		           origin.close();
		           if (deleteAddedFiles)
		        	   actualFile.delete();
		        }
		        out.close();

		  }
		  catch(FileNotFoundException e){
		   log.error("Ha ocurrido un error", e);
		  }
		  catch(IOException e){
			  log.error("Ha ocurrido un error", e);
		  }
		  return zipFileToCreate;
	}

	/**
	 * Descomprime un zip de un solo archivo manteniendo el nombre original del archivo dentro del zip
	 * @param compressedFile
	 * @throws IOException
	 * @throws ArchiveException
	 */
	public static String expandOneFile(String compressedFile) throws IOException, ArchiveException {
		return expandOneFile(compressedFile, null, CompressionFormat.ZIP);
	}

	/**
	 * Descomprime un zip de un solo archivo renombrando el archivo de salida
	 * @param compressedFile
	 * @param expandedFile
	 * @throws IOException
	 */
	public static String expandOneFile(String compressedFile, String expandedFile) throws IOException, ArchiveException {
		return expandOneFile(compressedFile, expandedFile, CompressionFormat.ZIP);
	}

	/**
	 * Descomprime un zip|gz|tag.gz de un solo archivo con el mismo nombre original o con el que me vino especificado en el 2do parametro
	 * Y devuelve el nombre del archivo descomprimido
	 * @param compressedFile
	 * @param expandedFile
	 * @throws IOException
	 */
	public static String expandOneFile(String compressedFile, String expandedFile, CompressionFormat compressionFormat) throws IOException, ArchiveException {
		String extractedFileName = null;

		if (compressedFile.endsWith(FILE_EXTENSION_GZIP)) {
			final int buffer = 2048*10;
			if (StringUtil.isEmpty(expandedFile)) {
				expandedFile = getFileNameWithoutExtension(compressedFile, false);
			}
			BufferedOutputStream out = new BufferedOutputStream(new FileOutputStream(expandedFile), buffer);
			GZIPInputStream in = new GZIPInputStream(new BufferedInputStream(new FileInputStream(compressedFile)));
			IOUtils.copy(in, out, buffer);
			out.close();
			in.close();
			extractedFileName = expandedFile;
		} else {
			try (BufferedInputStream bufferedInputStream = new BufferedInputStream(new FileInputStream(compressedFile));
				 ArchiveInputStream archiveInputStream = createArchiveInputStream(bufferedInputStream, getFileExtension(compressedFile))) {

				ArchiveEntry entry;
				// Itero hasta que extraigo 1 archivo
				while ((entry = archiveInputStream.getNextEntry()) != null && extractedFileName == null) {
					if (!archiveInputStream.canReadEntryData(entry) || entry.isDirectory()) {
						continue;
					}

					File entryFile;
					if (StringUtil.isNotEmpty(expandedFile)) {
						// Si vino definido el nombre con el cual quiero extraer el archivo, lo uso
						entryFile = new File(expandedFile);
					} else {
						// Sino, uso el nombre original
						entryFile = new File(Path.of(compressedFile).toAbsolutePath().getParent().toString(), entry.getName());
					}

					try (OutputStream out = new FileOutputStream(entryFile)) {
						IOUtils.copy(archiveInputStream, out);
					}

					// Guardar el nombre del archivo extraído
					extractedFileName = entryFile.getAbsolutePath();
				}
			}
		}
		return extractedFileName;
	}

	private static ArchiveInputStream createArchiveInputStream(InputStream inputStream, String fileExtension) throws IOException, ArchiveException {
		if (fileExtension.equals(FILE_EXTENSION_TAR_GZ) || fileExtension.equals(FILE_EXTENSION_TGZ)) {
			return new TarArchiveInputStream(new GzipCompressorInputStream(inputStream));
		} else {
			return new ArchiveStreamFactory().createArchiveInputStream(fileExtension, inputStream);
		}
	}

	public static byte[] gzipBytes(byte[] bytes) throws IOException {
	    ByteArrayOutputStream bos = new ByteArrayOutputStream();
        GZIPOutputStream zipStream = new GZIPOutputStream(bos);
        zipStream.write(bytes);
        zipStream.close();
        return bos.toByteArray();
    }

	public static List<File> expandFiles(String compressedFile, boolean deleteOnSuccess) throws IOException, ArchiveException {
		List<File> expandedFiles = new ArrayList<>();
		if (StringUtil.isEmpty(compressedFile) || !compressedFile.endsWith(FILE_EXTENSION_ZIP)) {
            log.warn("No es posible extraer los archivos de [{}]", compressedFile);
			return expandedFiles;
		}

        log.debug("[expandFiles] Por extraer los archivos de [{}]", compressedFile);
		try (BufferedInputStream bufferedInputStream = new BufferedInputStream(new FileInputStream(compressedFile));
			 ArchiveInputStream archiveInputStream = createArchiveInputStream(bufferedInputStream, getFileExtension(compressedFile))) {

			ArchiveEntry entry;
			while ((entry = archiveInputStream.getNextEntry()) != null) {
				if (!archiveInputStream.canReadEntryData(entry) || entry.isDirectory()) {
					continue;
				}

				File entryFile = new File(Path.of(compressedFile).toAbsolutePath().getParent().toString(), entry.getName());
				try (OutputStream out = new FileOutputStream(entryFile)) {
					IOUtils.copy(archiveInputStream, out);
				}

                log.debug("[expandFiles] Archivo [{}] extraido correctamente", entryFile.getName());

				// Guardar el nombre del archivo extraído
				expandedFiles.add(entryFile);
			}
		}
        log.debug("[expandFiles] Se extrajeron {} archivos. Fin de la extracion. {}", expandedFiles.size(), deleteOnSuccess ? "Procedo a eliminar el archivo comprimido" : "No se borra el archivo comprimido");
		if (deleteOnSuccess) {
			if (!new File(compressedFile).delete()) {
                log.warn("[expandFiles] No se pudo eliminar el archivo [{}]", compressedFile);
			}
		}

		return expandedFiles;
	}
	
	/**
	 * Verifica si el archivo es o no de 0 bytes
	 * 
	 * @param directory directorio o folder donde reside el archivo
	 * @param fileName nombre del archivo
	 * @return true si el archivo tiene 0 bytes
	 */
    public static boolean isEmptyFile(String directory, String fileName) {
        File theFile = new File(directory, fileName);
        return theFile.length() == 0;
    }

    public static StringBuilder cleanNamespaceProps(String filePath){
    	File file = new File(filePath);
    	StringBuilder sb = new StringBuilder();
    	FileReader fr = null;
    	BufferedReader br = null;
    	String line = null;
    	try {
			fr = new FileReader(file);
	    	br = new BufferedReader(fr); 
		} catch (FileNotFoundException e) {log.error("Ha ocurrido un error", e);}
		
		try {			
			while((line = br.readLine()) != null) { 
				if(line.contains("xmlns=")){
					String out = ""; 
					int init = line.indexOf("xmlns=");
					out += line.substring(0, init);
					// en esta linea el 7 es el lenght de 'xmlns="' 
					// y el 1 que se le suma es para que escape a la comilla que cierra
					int end = line.indexOf("\"", init+7)+1;
					out += line.substring(end);
					sb.append(out);
				}else{
					sb.append(line);
				}				
			}
			fr.close(); 
		} catch (IOException e) {log.error("Ha ocurrido un error", e);}
    	return sb;
    }
    
    public static String concatPathKey(String path, String key){
    	String pathAux = path.substring(path.length()-1, path.length());
    	String keyAux =  key.substring(0, 1);
    	// si son iguales
    	if(pathAux.equalsIgnoreCase(keyAux)){
    		// son iguales pero no son '/' entonces lo agrego
    		if(!pathAux.equalsIgnoreCase("/")){
    			return path + "/" + key;
    		} else {
    			// los 2 son '/' tengo que dejar solo una
    			return path.substring(0, path.length()-1) + key;
    		}
    	} else {
    		// si ninguno es el separador '/' lo agrego
    		if(!pathAux.equalsIgnoreCase("/") && !keyAux.equalsIgnoreCase("/")){
    			return path + "/" + key;
    		}
    	}
    	// formato correcto
    	return path + key;
    }

    /**
     * Concatena el path y el file para crear un path completo
     * 
     * @param path Directorio, con o sin la ultima barra. Ej: c:\temp, c:\downloads\, /usr/local, /tmp/
     * @param file El nombre del archivo a concatenar
     * @param dirSeparator El separador de directorios. Ej: /, \
     * @return Devuelve la union del path con el file, poniendo o no entre ellos el separador de directorios de ser necesario
     */
    public static String getFullPath(String path, String file, String dirSeparator) {
    	return (!path.endsWith(dirSeparator)) ? path + dirSeparator + file : path + file;
    }
    
    /**
     * Agrega el separador de directorios al final del path si es que no lo tiene. Ej si viene c:/temp devuelve c:/temp/, si viene /usr/local/ devuelve lo mismo
     * 
     * @param path
     * @param dirSeparator
     * @return
     */
	public static String appendDirSeparator(String path, String dirSeparator) {
		return (!path.endsWith(dirSeparator)) ? path + dirSeparator : path;
	}
	
	/**
	 * Devuelve un conjunto de archivos dentro de un directorio
	 * Soporta que se le pase la ruta completa al directorio y archivos con wildcards
	 * 
	 * @param filePathWithWildcards ruta completa al directorio y nombre de archivo con wildcards. Ej: c:\\Temp\\pepe*.txt
	 * @param includeDirectories incluye los subdirectorios en la busqueda
	 * @return un conjunto de archivos que matchean con el filename pasado
	 */
	public static File[] getDirectoryWithWildcard(String filePathWithWildcards, final boolean includeDirectories) {
		String pathOnly = getPathOnly(filePathWithWildcards);
		String fileName = getFileName(filePathWithWildcards);
		
		File dir = new File(pathOnly);
		String pattern = fileName.replace("*", ".*").replace("?", ".");
		final Pattern compiledPattern = Pattern.compile(pattern);
		FileFilter ff = new FileFilter() {
			public boolean accept(File pathname) {
				return compiledPattern.matcher(pathname.getName()).find();
			}
		};
		
		return dir.listFiles(ff);
	}

	public static String getOSName() {
		return System.getProperty("os.name").toLowerCase();
	}
	
	public static boolean isWindowsOS() {
		return getOSName().startsWith("win");
	}

	public static boolean isLinuxOS() {
		return getOSName().startsWith("lin");
	}

	/**
	 * Extrae caracteres 'raros' del nombre de un archivo
	 * OJO! No usar con paths porque les va a eliminar las barras
	 * 
	 * @param filename el nombre de archivo sin la ruta
	 * @return
	 */
	public static String sanitizeFilename(String filename) {
		if (filename == null) return null;
		// Primero intento cambiar los acentos a sin acentos, y luego los caracteres "raros"
		return StringUtil.sanitizeASCII(filename, false)
				.replaceAll("[^\\p{ASCII}]", "_")   //Todo lo no ASCII a _ y
				.replaceAll("[&]", "Y")             //ampersand a Y
				.replaceAll("[\"'`,+!@#$%^*()]", "")//los bichos raros a nada
				.replaceAll("[\\\\/\\*\\? ]", "_"); //Las barras, asterisco y pregunta (escapadas muchas veces!) y espacio a _
	}

	/**
	 * Devuelve el path completo con el nombre del archivo sanitizado
	 * Ej. 
	 *   c:\test\pepe*.txt  --> c:\test\pepe_.txt
	 *   /usr/home/cumpleanos.doc --> /usr/home/cumplea_os.doc
	 * 
	 * @param fullPath
	 * @return
	 */
	public static String sanitizeFilenameFromFullPath(String fullPath) {
		if (hasPathBars(fullPath)) {
			return getPathOnly(fullPath, true) + sanitizeFilename(getFileName(fullPath));
		} else {
			return sanitizeFilename(getFileName(fullPath));
		}
	}


	/**
	 * Devuelve true si en el string pasado hay separadores de path como "/" o "\" 
	 * 
	 * @param fullPath
	 * @return
	 */
	public static boolean hasPathBars(String fullPath) {
		return fullPath != null && (fullPath.indexOf("/") != -1 || fullPath.indexOf("\\") != -1); 
	}

	/**
	 * Guardar una imagen que esta en un string Base64 en un archivo
	 * Los tipos de imagen pueden ser FileToolkit.IMAGE_TYPE_JPG o FileToolkit.IMAGE_TYPE_PNG
	 * 
	 * @param image64
	 * @param imageType
	 * @param file
	 * @throws IOException
	 */
	public static void saveImage64ToFile(String image64, String imageType, File file) throws IOException {
		BufferedImage image = null;
        byte[] imageByte;
	    imageByte = Base64.getDecoder().decode(image64);
	    ByteArrayInputStream bis = new ByteArrayInputStream(imageByte);
	    image = ImageIO.read(bis);
	    bis.close();
	    ImageIO.write(image, imageType, file);
	}

	public static boolean rename(String oldName, String newName) {
		File oldFile = new File(oldName);
	    File newFile = new File(newName);
	    return oldFile.renameTo(newFile);
	}

//	/**
//	 * Convierte paths del estilo ${inbox} a rutas reales en el filesystem
//	 *
//	 * @param instance
//	 * @param owner
//	 * @param pathToResolve
//	 * @return
//	 */
//	public static String resolvePath(Instance instance, Company owner, String pathToResolve) {
//		long ackNumber = HomeManager.getInstance().getCompanyFinder().getNextEbfDownloadNumber();
//		final Map<String, Object> contextMap = DataProcessor.getStreamContext(instance, ackNumber, null, null, owner.getIdAsLong(), null, null);
//		ObjectBuilderContext context = contextMap::get;
//		return ObjectBuilder.contextReplace(context, pathToResolve);
//	}

	public static int executeScript(String[] args) {
	    int exitValue = -1;
        log.debug("Por ejecutar script con parametros {}", Arrays.toString(args).replace(',', ' ')); //formateo para poder copiar/pegar del log a la linea de comando
	    try {
            ProcessBuilder pb = new ProcessBuilder(args);
            pb.redirectErrorStream(true);
            OutputExecToBuffer outputExecToBuffer = new OutputExecToBuffer();
            outputExecToBuffer.setProcessRunning(true);
            Process p = pb.start();
            outputExecToBuffer.setInputStream(p.getInputStream());
            outputExecToBuffer.start();
            p.waitFor();
            outputExecToBuffer.setProcessRunning(false);
            exitValue = p.exitValue();

        } catch (Exception e) {
	        log.error("Error en ejecucion de script", e);
        }
	    return exitValue;
    }

    /**
     * Redirecciona la salida de la ejecucion al log
     * @author eviera
     *
     */
    public static class OutputExecToBuffer extends Thread {

        private boolean processRunning;
        private InputStream inputStream;

        public OutputExecToBuffer() {
        }

        public void setProcessRunning(boolean processRunning) {
            this.processRunning = processRunning;
        }

        public void setInputStream(InputStream inputStream) {
            this.inputStream = inputStream;
        }

        @Override
        public void run() {
            while (processRunning) {
                try {
                    BufferedReader br = new BufferedReader(new InputStreamReader(inputStream));
                    String line;
                    while ((line = br.readLine()) != null) {
                        log.info(line);
                    }
                } catch (IOException e) {
                    log.error("Ha ocurrido un error", e);
                }
            }

        }
    }

    public enum FileSizeUnits {
        KB, MB, GB, TB
    }

    public static BigInteger toBytes(Long size, FileSizeUnits fileSizeUnits) {
        if (fileSizeUnits == null) {
            fileSizeUnits = FileSizeUnits.KB;
        }

        switch (fileSizeUnits) {
            case KB:
                return BigInteger.valueOf(FileUtils.ONE_KB).multiply(BigInteger.valueOf(size));
            case MB:
                return BigInteger.valueOf(FileUtils.ONE_MB).multiply(BigInteger.valueOf(size));
            case GB:
                return BigInteger.valueOf(FileUtils.ONE_GB).multiply(BigInteger.valueOf(size));
            case TB:
                return BigInteger.valueOf(FileUtils.ONE_TB).multiply(BigInteger.valueOf(size));
        }

        log.warn("[toBytes] No se pudo convertir el valor {}{}a bytes. Devuelvo {}", size, fileSizeUnits.name(), size);
        return BigInteger.valueOf(size);
    }

    public static BigDecimal bytesTo(long bytes, FileSizeUnits fileSizeUnitsTo) {
        if (fileSizeUnitsTo == null) {
            return NumberUtil.ZERO_SCALED;
        }

        switch (fileSizeUnitsTo) {
            case KB:
                return BigDecimal.valueOf(bytes).divide(BigDecimal.valueOf(FileUtils.ONE_KB), NumberUtil.BIGDECIMAL_SCALE, RoundingMode.HALF_UP);
            case MB:
                return BigDecimal.valueOf(bytes).divide(BigDecimal.valueOf(FileUtils.ONE_MB), NumberUtil.BIGDECIMAL_SCALE, RoundingMode.HALF_UP);
            case GB:
                return BigDecimal.valueOf(bytes).divide(BigDecimal.valueOf(FileUtils.ONE_GB), NumberUtil.BIGDECIMAL_SCALE, RoundingMode.HALF_UP);
            case TB:
                return BigDecimal.valueOf(bytes).divide(BigDecimal.valueOf(FileUtils.ONE_TB), NumberUtil.BIGDECIMAL_SCALE, RoundingMode.HALF_UP);
        }

        log.warn("[toBytes] No se pudo convertir el valor {} bytes a {}. Devuelvo {}", bytes, fileSizeUnitsTo.name(), bytes);
        return BigDecimal.valueOf(bytes);
    }

//    public static boolean hasValidExtensionForUpload(@NotNull File file, Long billerId) {
//        return hasValidExtensionForUpload(file.getName(),billerId);
//    }
//    public static boolean hasValidExtensionForUpload(String filename, Long billerId) {
//        String filenameExtension = FilenameUtils.getExtension(filename);
//		//Si alguna esta vacioa, no se controla; sino, debe estar (o no esar)
//		String suffix = billerId!=null?"."+billerId:"";
//
//		String invalidFileExtensionsForUpload=EBFConfiguration.getInstance().getProperty("invalidFileExtensionsForUpload"+suffix);
//		if (StringUtil.isEmpty(invalidFileExtensionsForUpload)) invalidFileExtensionsForUpload=EBFConfiguration.getInstance().getProperty("invalidFileExtensionsForUpload");
//
//		String validFileExtensionsForUpload=EBFConfiguration.getInstance().getProperty("validFileExtensionsForUpload"+suffix);
//		if (StringUtil.isEmpty(validFileExtensionsForUpload)) validFileExtensionsForUpload=EBFConfiguration.getInstance().getProperty("validFileExtensionsForUpload");
//
//		return StringUtil.isNotEmpty(filenameExtension)
//				&& (StringUtil.isEmpty(invalidFileExtensionsForUpload) ||StringUtil.isNotInList(invalidFileExtensionsForUpload, filenameExtension))
//		        && (StringUtil.isEmpty(validFileExtensionsForUpload)   ||StringUtil.isInList(validFileExtensionsForUpload, filenameExtension));
//    }

	public static String encodeFileToBase64(String fileName) throws IOException {
		File file = new File(fileName);
		return Base64.getEncoder().encodeToString(FileUtils.readFileToByteArray(file));
	}

    public static File zipFile(File fileToZip) throws IOException {
        log.debug("[zipFile] Por zippear el archivo [{}] a [{}.zip]", fileToZip.getAbsolutePath(), fileToZip.getName());
        byte[] buffer = new byte[BUFFER_SIZE];

        File zipFile = new File(fileToZip.getParentFile(), fileToZip.getName() + ".zip");

        try (FileOutputStream fos = new FileOutputStream(zipFile);
             ZipOutputStream zos = new ZipOutputStream(fos);
             FileInputStream fis = new FileInputStream(fileToZip)) {

            // Definir el nombre del archivo dentro del zip
            ZipEntry ze = new ZipEntry(fileToZip.getName());
            zos.putNextEntry(ze);

            // Copiar el contenido del archivo al zip
            int len;
            while ((len = fis.read(buffer)) > 0) {
                zos.write(buffer, 0, len);
            }

            // Cerrar el zip entry
            zos.closeEntry();
        }

        log.debug("[zipFile] Archivo zippeado");

        return zipFile;
    }

	public static void deleteEmptyFile(String file) {
		File theFile = new File(file);
		if (!theFile.exists()) {
			return;
		}

		if (theFile.length() == 0) {
            log.info("Se elimina file [{}] por estar vacio", file);
			theFile.delete();
		}
	}

	public static void addLibraryPath(String libPath) {
        try {
            final Field usrPathsField;
            usrPathsField = ClassLoader.class.getDeclaredField("usr_paths");
            usrPathsField.setAccessible(true);

            //get array of paths
            final String[] usrPaths = (String[])usrPathsField.get(null);

            //check if the path to add is already present
            boolean alreadyInPath=false;
            for(String path : usrPaths) {
                if(path.equals(libPath)) {
                    alreadyInPath=true;
                }
            }
            if (!alreadyInPath) {
                //add the new path
                final String[] newPaths = Arrays.copyOf(usrPaths, usrPaths.length + 1);
                newPaths[newPaths.length-1] = libPath;
                usrPathsField.set(null, newPaths);
            }
        } catch (NoSuchFieldException | IllegalAccessException e) {
            throw new RuntimeException(e);
        }
    }
}