package ar.com.sdd.commons.util;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.util.*;

/**
 * Implementacion de una cache con posiblidad de setear un timeout a los objetos guardados
 */
public class SimpleCache {

    private static final Logger log = LogManager.getLogger(SimpleCache.class);

    public static final int ONE_DAY = 86400;        //24hs en segundos
    public static final int ONE_HOUR = 3600;        // 1h  en segundos
    public static final int TEN_MINUTES = 600;      //10m  en segundos
    public static final int FIVE_MINUTES = 300;     // 5m  en segundos
    private static final long SAFE_GAP = 1000;      // 1s  en milis: un minimo delay para pode para consultar y asegurarme que va a estar al buscarla

    private final Map<String, SimpleCacheEntry> cache;
    private final SimpleCacheManager.SimpleCacheLevel cacheLevel;
    private final String cacheName;

    public SimpleCache(String cacheName, SimpleCacheManager.SimpleCacheLevel cacheLevel) {
        cache = new HashMap<>();
        //Por default es una WEAK
        if (cacheLevel == null) {
            cacheLevel = SimpleCacheManager.SimpleCacheLevel.WEAK;
        }
        this.cacheLevel = cacheLevel;
        this.cacheName = cacheName;
    }

    /**
     * Inserta un objeto en la cache sin tiempo de expiracion
     *
     * @param key
     * @param value
     */
    public synchronized void put(String key, Object value) {
        cache.put(key, new SimpleCacheEntry(value));
    }

    /**
     * Inserta un objeto en la cache con tiempo de expiracion en n segundos en el futuro
     *
     * @param key
     * @param value
     * @param timeToLiveInSeconds
     */
    public synchronized void put(String key, Object value, Integer timeToLiveInSeconds) {
        SimpleCacheEntry entry = new SimpleCacheEntry(value);
        if (timeToLiveInSeconds != null) {
            entry.setTimeout(timeToLiveInSeconds.longValue());
        }
        cache.put(key, entry);
    }

    /**
     * Recupera un objeto de la cache. Si el objeto tiene tiempo de expiracion no nulo y es menor a ahora (System.currentTimeMillis()),
     * se lo devuelve. Pero si esta expirado, se devuelve null y se remueve el objeto de la cache
     *
     * extendTimeout :
     * @param key
     * @return
     */
    public Object get(String key) {
        return get(key,0, null);
    }
    public Object get(String key, Long newTimeout) {
        return get(key,0, newTimeout);
    }

    private Object get(String key, long gap, Long newTimeout) {
        SimpleCacheEntry entry = cache.get(key);

        if (entry != null) {
            //Chequeo si no expiro (si tiene ttl)
            if (entry.isExpired(gap)) {
                //expiro, la remuevo
                cache.remove(key);
                log.debug("[" + cacheName + "] Se remueve la key [" + key + "] por expirada");
                return null;
            }
            if (newTimeout !=null ) {
                entry.setTimeout(newTimeout);
            }
            //Si no expiro, o no tenia expiracion, devuelvo el valor
            return entry.getValue();

        } else {
            //No la encontre
            return null;
        }
    }


    public void clear() {
        cache.clear();
    }

    public int size() {
        return cache.size();
    }

    public boolean containsKey(String key) {
        return get(key,SAFE_GAP, null)!=null;
    }
    // Timestamp
    public Long  getCreationTimestamp(String key) {
        SimpleCacheEntry entry  = (SimpleCacheEntry)cache.get(key);
        if (entry == null) return null;
        return  entry.getCreationTimestamp();
    }
    //Milisecundos
    public Long  getRemainigTime(String key) {
        SimpleCacheEntry entry  = (SimpleCacheEntry)cache.get(key);
        if (entry == null) return null;
        return  entry.getRemainigTime();
    }

    public synchronized Object remove(String key) {
        SimpleCacheEntry entry = cache.remove(key);
        if (entry != null && !entry.isExpired(0)) {
            return entry.getValue();
        } else {
            return null;
        }
    }

    /**
     * Hace un pasamanos de la implementacion interna del entrySet, pasando por el get que remueve el objeto de la cache si este expiro.
     *
     * @return
     */
    public Set<Map.Entry<String, Object>> entrySet() {
        Set<Map.Entry<String, Object>> result = new HashSet<>();
        List<String> entriesKeyToRemove = new ArrayList<>();

        for (Map.Entry<String, SimpleCacheEntry> entry : cache.entrySet()) {
            //No puedo usar el metodo get() porque remueve los objetos de cache y da ConcurrentModificationException. Entonces me los guardo en un list para borrarlos despues
            SimpleCacheEntry simpleCacheEntry = cache.get(entry.getKey());
            if (simpleCacheEntry != null && !simpleCacheEntry.isExpired(0)) {
                result.add(new AbstractMap.SimpleEntry<>(entry.getKey(), simpleCacheEntry.getValue()));
            } else {
                entriesKeyToRemove.add(entry.getKey());
            }
        }

        //Saco de la cache los elementos expirados
        if (!CollectionUtils.isEmpty(entriesKeyToRemove)) {
            for (String entryKey : entriesKeyToRemove) {
                cache.remove(entryKey);
            }
        }

        return result;
    }

    public SimpleCacheManager.SimpleCacheLevel getCacheLevel() {
        return cacheLevel;
    }

    @Override
    public String toString() {
        return "SimpleCache ["
                + ((cache != null) ? "cache=" + cache + ", " : "")
                + ((cacheLevel != null) ? "cacheLevel=" + cacheLevel + ", " : "")
                + ((cacheName != null) ? "cacheName=" + cacheName : "")
                + "]";
    }

    private static class SimpleCacheEntry {

        private Object value;
        private Long timeToLive;
        private Long creationTimestamp;

        public SimpleCacheEntry(Object value) {
            this(value, null);
        }

        public SimpleCacheEntry(Object value, Long timeToLive) {
            this.creationTimestamp = System.currentTimeMillis();
            this.value = value;
            this.timeToLive = timeToLive;
        }

        public Object getValue() {
            return value;
        }

        public void setValue(Object value) {
            this.value = value;
        }

        public Long getTimeToLive() {
            return timeToLive;
        }
        //milisegundos que quedan
        public Long getRemainigTime() {
            return timeToLive-System.currentTimeMillis();
        }
        public Long getCreationTimestamp() { return creationTimestamp;}

        /*
                  Setea la hora de expracion
                     setTimeToLive : una hora en el futuro
                     setTimeout    : milisegundos desde ahora
                 */
        public SimpleCacheEntry setTimeToLive(Long timeToLive) {
            this.timeToLive = timeToLive;
            return  this;
        }
        public SimpleCacheEntry setTimeout(long timeoutFromNow) {
            this.timeToLive = System.currentTimeMillis() + timeoutFromNow * 1000;;
            return this;
        }

        private boolean isExpired(long gap) {
            return getTimeToLive() != null && System.currentTimeMillis() > getTimeToLive()+gap;
        }

        @Override
        public String toString() {
            return "SimpleCacheEntry ["
                    + ((value != null) ? "value=" + value + ", " : "")
                    + ((timeToLive != null) ? "timeToLive=" + timeToLive : "")
                    + "]";
        }
    }

}
