package ar.com.sdd.creditosimple.core.galicia;

import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSecurityManager;
import ar.com.sdd.commons.util.SimpleCache;
import ar.com.sdd.commons.util.SimpleCacheManager;
import ar.com.sdd.commons.util.StringUtil;
import ar.com.sdd.creditosimple.entity.galicia.*;
import ar.com.sdd.creditosimple.io.galicia.*;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.ws.rs.client.Invocation;
import java.util.Map;

public class CreditoSimpleNeraV2Connector implements RestSecurityManager {

    private static final Logger log = LogManager.getLogger(CreditoSimpleNeraV2Connector.class);

    private final RestConnector restConnector;
    private final String email;
    private final String password;

    private final SimpleCache cache = SimpleCacheManager.getInstance().getCache(CreditoSimpleNeraV2Connector.class.getName());
    private final String accessTokenCacheKey = SimpleCacheManager.buildKey("accessToken");
    private final String refreshTokenCacheKey = SimpleCacheManager.buildKey("refreshToken");
    private final String idTokenCacheKey = SimpleCacheManager.buildKey("idToken");
    private boolean doingLogin;

    public CreditoSimpleNeraV2Connector(CreditoSimpleNeraV2ConnectorContext context) {
        String baseUrl = context.getBaseUrl();
        email = context.getEmail();
        password = context.getPassword();

        log.debug("Creando conector para CreditoSimple Galicia con baseUrl [" + baseUrl + "], email [" + email + "] y password [" + password + "]");

        RestConnectorEnvironment environment = new RestConnectorEnvironment(baseUrl, context.getKeyStorePath(), context.getKeyStorePassword(), null, null);
        restConnector = new RestConnector(environment, this);
    }

    // ------------------------- [PRODUCTS] -------------------------

    public ProductV2[] getProducts() throws RestConnectorException {
        final String path = "/products";
        log.debug("[getProducts] Por consultar productos con path [{}]", path);
        return restConnector.genericGet(null, ProductV2[].class, ErrorV2Response.class, path);
    }

    public ProductV2 getProductByKeyName(String productKeyName) throws RestConnectorException {
        final String path = "/products/" + productKeyName;
        log.debug("[getProducts] Por consultar producto con keyName [{}] y path [{}]", productKeyName, path);
        return restConnector.genericGet(null, ProductV2.class, ErrorV2Response.class, path);
    }

    // ------------------------- [PARTNERS] -------------------------

    public PartnerV2[] getPartners() throws RestConnectorException {
        final String path = "/partners";
        log.debug("[getPartners] Por consultar partners con path [{}]", path);
        return restConnector.genericGet(null, PartnerV2[].class, ErrorV2Response.class, path);
    }

    public PartnerV2 getPartnerByKeyName(String productKeyName) throws RestConnectorException {
        final String path = "/partners/" + productKeyName;
        log.debug("[getPartners] Por consultar partners con keyName [{}] y path [{}]", productKeyName, path);
        return restConnector.genericGet(null, PartnerV2.class, ErrorV2Response.class, path);
    }

    // ------------------------- [AGREEMENTS] -------------------------

    public String[] getAgreementsLoanTypes() throws RestConnectorException {
        final String path = "/agreements/loan-types";
        log.debug("[getAgreementsLoanTypes] Por consultar tipos de prestamos de convenios con path [{}]", path);
        return restConnector.genericGet(null, String[].class, ErrorV2Response.class, path);
    }

    /**
     * TODO: En la documentacion no hay ejemplos de que devuelve esta invocacion
     */
    public AgreementV2 getAgreementDetails(Long agreementId) throws RestConnectorException {
        final String path = "/agreements/details/" + agreementId;
        log.debug("[getAgreementsLoanTypes] Por consultar detalles convenio [{}] con path [{}]", agreementId, path);
        return restConnector.genericGet(null, AgreementV2.class, ErrorV2Response.class, path);
    }

    /**
     * @param merchantCuit   (Required) Merchant CUIT (Clave Única de Identificación Tributaria)
     * @param loanType       Type of loan associated with the agreement
     * @param farmerCuit     Farmer CUIT (if applicable)
     * @param productId      ID of the product associated with the agreement
     * @param partnerCuit    CUIT of the partner involved in the agreement
     * @param productKeyName Product key name associated with the agreement
     */
    public AgreementV2[] getAgreementsByMerchantCuit(String merchantCuit, String segmentName, String loanType, String farmerCuit, String productId, String partnerCuit, String productKeyName) throws RestConnectorException {
        return getAgreements(merchantCuit, null, loanType, farmerCuit, productId, partnerCuit, productKeyName);
    }

    /**
     * @param merchantCuit   (Required) Merchant CUIT (Clave Única de Identificación Tributaria)
     * @param segmentName    (Required) The name of the segment to filter agreements by.
     * @param loanType       Type of loan associated with the agreement
     * @param farmerCuit     Farmer CUIT (if applicable)
     * @param productId      ID of the product associated with the agreement
     * @param partnerCuit    CUIT of the partner involved in the agreement
     * @param productKeyName Product key name associated with the agreement
     */
    public AgreementV2[] getAgreementsByMerchantCuitAndSegmentName(String merchantCuit, String segmentName, String loanType, String farmerCuit, String productId, String partnerCuit, String productKeyName) throws RestConnectorException {
        return getAgreements(merchantCuit, segmentName, loanType, farmerCuit, productId, partnerCuit, productKeyName);
    }

    private AgreementV2[] getAgreements(String merchantCuit, String segmentName, String loanType, String farmerCuit, String productId, String partnerCuit, String productKeyName) throws RestConnectorException {
        String path = "/agreements/" + merchantCuit;
        if (StringUtil.isNotEmpty(segmentName)) {
            path += "/segments/" + segmentName;
        }

        char paramSep = '?';
        if (StringUtil.isNotEmpty(loanType)) {
            path += paramSep + "loan_type=" + loanType;
            paramSep = '&';
        }
        if (StringUtil.isNotEmpty(farmerCuit)) {
            path += paramSep + "farmer_cuit=" + farmerCuit;
            paramSep = '&';
        }
        if (StringUtil.isNotEmpty(productId)) {
            path += paramSep + "product_id=" + productId;
            paramSep = '&';
        }
        if (StringUtil.isNotEmpty(partnerCuit)) {
            path += paramSep + "partner_cuit=" + partnerCuit;
            paramSep = '&';
        }
        if (StringUtil.isNotEmpty(productKeyName)) {
            path += paramSep + "product_key_name=" + productKeyName;
        }

        log.debug("[getAgreements] Por consultar convenios con path [{}]", path);
        return restConnector.genericGet(null, AgreementV2[].class, ErrorV2Response.class, path);
    }

    // ------------------------- [ORDERS] -------------------------

    public OrderV2 getOrder(String orderId) throws RestConnectorException {
        final String path = "/order/" + orderId;
        log.debug("[getOrder] Por consultar order [{}] y path [{}]", orderId, path);
        return restConnector.genericGet(null, OrderV2.class, ErrorV2Response.class, path);
    }

    public OrderV2Response postOrder(OrderV2Request orderRequest) throws RestConnectorException {
        final String path = "/order";
        log.debug("[postOrder] Por crear order [{}]", orderRequest);
        return restConnector.genericGet(orderRequest, OrderV2Response.class, ErrorV2Response.class, path);
    }

    // ------------------------- [FARMERS] -------------------------

    public FarmerV2 getFarmer(String farmerCuit) throws RestConnectorException {
        final String path = "/farmers/" + farmerCuit;
        log.debug("[getFarmer] Por consultar farmer con cuit path [{}]", path);
        return restConnector.genericGet(null, FarmerV2.class, ErrorV2Response.class, path);
    }

    public FarmerV2 getFarmerMipyme(String partnerKeyName, String farmerCuit) throws RestConnectorException {
        final String path = "/farmers/mipyme/" + partnerKeyName + "/" + farmerCuit;
        log.debug("[getFarmerMipyme] Por consultar farmer con cuit path [{}]", path);
        return restConnector.genericGet(null, FarmerV2.class, ErrorV2Response.class, path);
    }

    // ------------------------- [ELIGIBILITY] -------------------------

    public EligibilityV2Response postEligibility(EligibilityV2Request eligibilityV2Request) throws RestConnectorException {
        final String path = "/eligibility";
        log.debug("[postEligibility] Por hacer postEligibility para [{}]", eligibilityV2Request);
        return restConnector.genericPost(eligibilityV2Request, EligibilityV2Response.class, ErrorV2Response.class, path);
    }

    // ------------------------- [CREDIT CALCULATION] -------------------------

    public CreditCalculationV2Response postCreditCalculation(CreditCalculationV2Request creditCalculationV2Request) throws RestConnectorException {
        final String path = "/credit-calculation";
        log.debug("[postCreditCalculation] Por hacer el calculo de creadito para [{}]", path);
        return restConnector.genericPost(creditCalculationV2Request, CreditCalculationV2Response.class, ErrorV2Response.class, path);
    }

    // ------------------------- [METRICS] -------------------------

    public String getMetrics() throws RestConnectorException {
        final String path = "/metrics";
        log.debug("[getMetrics] Por consultar metricas con path [{}]", path);
        return restConnector.genericGet(null, String.class, ErrorV2Response.class, path);
    }

    // ------------------------- [LOGIN STAFF] -------------------------

    private String getAccessToken() throws RestConnectorException {
        return getToken(accessTokenCacheKey);
    }

    private String getRefreshToken() throws RestConnectorException {
        return getToken(refreshTokenCacheKey);
    }

    private String getIdToken() throws RestConnectorException {
        return getToken(idTokenCacheKey);
    }

    private String getToken(String cacheKey) throws RestConnectorException {
        String token = (String) cache.get(cacheKey);
        if (token == null) {
            login();
            token = (String) cache.get(cacheKey);
        }
        return token;
    }

    private void login() throws RestConnectorException {
        log.debug("[login] Por hacer login");
        doingLogin = true;
        TokenOAuth2Nera tokenOAuth2 = restConnector.genericPost(Map.of("email", email, "password", password), TokenOAuth2Nera.class, ErrorV2Response.class, "/auth/login");
        doingLogin = false;

        final String accessToken = tokenOAuth2.getAccessToken();
        final String refreshToken = tokenOAuth2.getRefreshToken();
        final String idToken = tokenOAuth2.getIdToken();
        final Integer expiresIn = tokenOAuth2.getExpiresIn();
        log.debug("[login] Login realizado. Se guardan: accessToken [{}], refreshToken [{}] y idToken [{}] por [{}] segs", accessToken, refreshToken, idToken, expiresIn);
        cache.put(accessTokenCacheKey, accessToken, expiresIn);
        cache.put(refreshTokenCacheKey, refreshToken, expiresIn);
        cache.put(idTokenCacheKey, idToken, expiresIn);
    }

    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        if (builder != null) {
            if (!doingLogin) {
                builder.header("Authorization", "Bearer " + getAccessToken());
                builder.header("id_token", getIdToken());
            }
        }
        return builder;
    }

    @Override
    public boolean getDisableHTTPSErrors() {
        return false;
    }

    @Override
    public boolean retryOnUnauthorized() {
        if (doingLogin) {
            return false;
        } else {
            try {
                login();
                return true;
            } catch (RestConnectorException e) {
                return false;
            }
        }
    }
}