package ar.com.sdd.debin.core;

import ar.com.sdd.debin.io.*;
import ar.com.sdd.debin.token.SecurityManager;
import org.apache.log4j.Logger;

import javax.ws.rs.ProcessingException;
import javax.ws.rs.client.Entity;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.util.HashMap;
import java.util.Map;

public class DebinConnector {

    private static final Logger log = Logger.getLogger(DebinConnector.class);

    /**
     * URL de API debin
     */
    private String debinBaseUrl;

    /**
     * El administrador de tokens. Si no se usa autenticacion por token, evita hacer la conexion para recuperar el token
     */
    private SecurityManager securityManager;

    /**
     * Crea un conector sencillo sin autenticacion, ni keyStores
     *
     * @param environment
     */
    public DebinConnector(DebinEnvironment environment) {
        this(null, null, null, null, environment.getAuthBaseUrl(), environment.getDebinBaseUrl(), null, null, null, false);
    }

    public DebinConnector(String authBaseUrl, String debinBaseUrl) {
        this(null, null, null, null, authBaseUrl, debinBaseUrl, null, null, null, false);
    }

    public DebinConnector(String authBaseUrl, String debinBaseUrl, String keyStorePath, String keyStoreOpenPassword, String keyStorePrivateKeyPassword) {
        this(null, null, null, null, authBaseUrl, debinBaseUrl, keyStorePath, keyStoreOpenPassword, keyStorePrivateKeyPassword, false);
    }

    public DebinConnector(String clientId, String clientSecret, String username, String password, String authBaseUrl, String debinBaseUrl, String keyStorePath, String keyStoreOpenPassword, String keyStorePrivateKeyPassword, boolean useTokenAuthentication) {
        log.debug("Creando DebinConnector con clientId:[" + clientId + "], clientSecret:[" + clientSecret + "], username:[" + username + "], " +
                "password:[" + password + "], authBaseUrl:[" + authBaseUrl + "], debinBaseUrl:[" + debinBaseUrl + "]," +
                "keyStorePath:[" + keyStorePath + "], keyStoreOpenPassword:[" + keyStoreOpenPassword + "], keyStorePrivateKeyPassword:[" + keyStorePrivateKeyPassword + "], " +
                "useTokenAuthentication: [" + useTokenAuthentication + "]");
        this.debinBaseUrl = debinBaseUrl;

        //Se inicializa el securityManager
        securityManager = new SecurityManager(clientId, clientSecret, username, password, authBaseUrl, keyStorePath, keyStoreOpenPassword, keyStorePrivateKeyPassword, useTokenAuthentication);
    }


    /*
     *
     * Metodos Debin
     *
     */

    public CrearDebinResponse crearDebin(CrearDebinRequest request) throws DebinException {
        request.validate();
        return genericPost(request, CrearDebinResponse.class, "/Debin/Debin");
    }

    public ConsultarDebinResponse consultarDebin(ConsultarDebinRequest request) throws DebinException {
        return genericGet(request, ConsultarDebinResponse.class, "/Debin/Debin/{id}", "id", request.getDebinId());
    }

    public ConfirmarDebitoResponse confirmarDebito(ConfirmarDebitoRequest request) throws DebinException {
        return genericPost(request, ConfirmarDebitoResponse.class, "/Debin/ConfirmaDebito");
    }

    public EliminarDebinResponse eliminarDebin(EliminarDebinRequest request) throws DebinException {
        Map<String, Object> templateParameters = new HashMap<>();
        templateParameters.put("id", request.getDebinId());
        templateParameters.put("cuit", request.getCuit());
        return genericDelete(request, EliminarDebinResponse.class, "/Debin/Debin/{id}/{cuit}", templateParameters);
    }

    public ListarDebinResponse listarDebin(ListarDebinRequest request) throws DebinException {
        return genericPost(request, ListarDebinResponse.class, "/Debin/Lista");
    }

    public EstablecerModoBancoResponse establecerModoBanco(EstablecerModoBancoRequest request) throws DebinException {
        return genericPost(request, EstablecerModoBancoResponse.class, "/Bancos/ModoBanco");
    }

    public void respuestaDebin(RespuestaDebinRequest request) throws DebinException {
        genericPost(request, Void.class, "/RespuestaDebin");
    }

    public int echo() throws DebinException {
        Response getResponse = null;
        try {
            getResponse = securityManager.buildRequest(debinBaseUrl + "/Echo", null)
                .accept(MediaType.APPLICATION_JSON)
                .get();
            return getResponse.getStatus();
        } finally {
            if (getResponse != null) {
                getResponse.close();
            }
        }
    }


    /*
     *
     * Metodos de Vendedores y Compradores
     *
     */

    public AdherirVendedorResponse adherirVendedor(AdherirVendedorRequest request) throws DebinException {
        return genericPost(request, AdherirVendedorResponse.class, "/Vendedor/Adhesion");
    }

    public AdherirCompradorResponse adherirComprador(AdherirCompradorRequest request) throws DebinException {
        return genericPost(request, AdherirCompradorResponse.class, "/Comprador/Adhesion");
    }

    public QuitarVendedorResponse quitarVendedor(QuitarVendedorRequest request) throws DebinException {
        return genericPost(request, QuitarVendedorResponse.class, "/Vendedor/BajaCuenta");
    }

    public QuitarCompradorResponse quitarComprador(QuitarCompradorRequest request) throws DebinException {
        return genericPost(request, QuitarCompradorResponse.class, "/Comprador/BajaCuenta");
    }

    public ConsultarCompradorResponse consultarComprador(ConsultarCompradorRequest request) throws DebinException {
        return genericGet(request, ConsultarCompradorResponse.class, "/Comprador/Comprador/{cuit}", "cuit", request.getCuit());
    }

    public ConsultarVendedorResponse consultarVendedor(ConsultarVendedorRequest request) throws DebinException {
        return genericGet(request, ConsultarVendedorResponse.class, "/Vendedor/Vendedor/{cuit}", "cuit", request.getCuit());
    }

    public LimitesModificarVendedorResponse limitesModificarVendedor(LimitesModificarVendedorRequest request) throws DebinException {
        return genericPost(request, LimitesModificarVendedorResponse.class, "/Vendedor/LimitesModificar");
    }

    public LimitesConsultaVendedorResponse limitesConsultaVendedor(LimitesConsultaVendedorRequest request) throws DebinException {
        return genericGet(request, LimitesConsultaVendedorResponse.class, "/Vendedor/LimitesConsulta/{cuit}", "cuit", request.getCuit());
    }

    public ConsumosConsultaVendedorResponse consumosConsultaVendedor(ConsumosConsultaVendedorRequest request) throws DebinException {
        return genericGet(request, ConsumosConsultaVendedorResponse.class, "/Vendedor/ConsumosConsulta/{cuit}", "cuit", request.getCuit());
    }

    public PedidoRecurrenciaVendedorResponse pedidoRecurrenciaVendedor(PedidoRecurrenciaVendedorRequest request) throws DebinException {
        return genericPost(request, PedidoRecurrenciaVendedorResponse.class, "/Vendedor/PedidoRecurrencia");
    }

    public AdherirRecurrenciaCompradorResponse adherirRecurrenciaComprador(AdherirRecurrenciaCompradorRequest request) throws DebinException {
        return genericPost(request, AdherirRecurrenciaCompradorResponse.class, "/Comprador/AdherirRecurrencia");
    }

    public CompradorRecurrenciaListaCompradorResponse compradorRecurrenciaListaComprador(CompradorRecurrenciaListaCompradorRequest request) throws DebinException {
        return genericPost(request, CompradorRecurrenciaListaCompradorResponse.class, "/Comprador/CompradorRecurrenciaLista");
    }

    public CompradorRecurrenciaCompradorResponse compradorRecurrenciaComprador(CompradorRecurrenciaCompradorRequest request) throws DebinException {
        Map<String, Object> templateParameters = new HashMap<>();
        templateParameters.put("id", request.getId());
        templateParameters.put("cuit", request.getCuit());
        return genericGet(request, CompradorRecurrenciaCompradorResponse.class, "/Comprador/CompradorRecurrencia/{cuit}/{id}", templateParameters);
    }

    public VendedorRecurrenciaListaVendedorResponse vendedorRecurrenciaListaVendedor(VendedorRecurrenciaListaVendedorRequest request) throws DebinException {
        return genericPost(request, VendedorRecurrenciaListaVendedorResponse.class, "/Vendedor/VendedorRecurrenciaLista");
    }

    public VendedorRecurrenciaVendedorResponse vendedorRecurrenciaVendedor(VendedorRecurrenciaVendedorRequest request) throws DebinException {
        Map<String, Object> templateParameters = new HashMap<>();
        templateParameters.put("id", request.getId());
        templateParameters.put("cuit", request.getCuit());
        return genericGet(request, VendedorRecurrenciaVendedorResponse.class, "/Vendedor/VendedorRecurrencia/{cuit}/{id}", templateParameters);
    }

    public SolicitudContraCargoCompradorResponse solicitudContraCargoComprador(SolicitudContraCargoCompradorRequest request) throws DebinException {
        return genericPost(request, SolicitudContraCargoCompradorResponse.class, "/Comprador/SolicitudContraCargo");
    }

    public OrdenDeCreditoCompradorResponse ordenDeCreditoComprador(OrdenDeCreditoCompradorRequest request) throws DebinException {
        return genericPost(request, OrdenDeCreditoCompradorResponse.class, "/Comprador/OrdenDeCredito");
    }

    public OrdenDeDebitoCompradorResponse ordenDeDebitoComprador(OrdenDeDebitoCompradorRequest request) throws DebinException {
        return genericPost(request, OrdenDeDebitoCompradorResponse.class, "/Comprador/OrdenDeDebito");
    }

    /*
     *
     * Metodos de Alias
     *
     */

    public ConsultarAliasResponse consultarAlias(ConsultarAliasRequest request) throws DebinException {
        return genericPost(request, ConsultarAliasResponse.class, "/Alias/Consultar");
    }


    /*
     *
     * Metodos genericos
     *
     */

    private <Res, Req> Res genericGet(Req request, Class<Res> responseType, String path) throws DebinException {
        return genericGet(request, responseType, path, null);
    }

    private <Res, Req> Res genericGet(Req request, Class<Res> responseType, String path, String templateKey, String templateValue) throws DebinException {
        Map<String, Object> templateParameters = new HashMap<>();
        templateParameters.put(templateKey, templateValue);
        return genericGet(request, responseType, path, templateParameters);
    }

    private <Res, Req> Res genericGet(Req request, Class<Res> responseType, String path, Map<String, Object> templateParameters) throws DebinException {
        log.trace("Get cliente request [" + request + "]");

        Response getResponse = null;
        try {

            getResponse = securityManager.buildRequest(debinBaseUrl + path, templateParameters)
                    .accept(MediaType.APPLICATION_JSON)
                    .get();
            log.trace("Get cliente response [" + getResponse.getStatus() + "/" + getResponse.getStatusInfo() + "]");

            if (getResponse.getStatus() != Response.Status.OK.getStatusCode()) {
                throw new DebinException("La consulta no devolvio OK. Status: [" + getResponse.getStatus() + "/" + getResponse.getStatusInfo() + "]");
            }

            Res entityResponse;
            //Caso especial para cuando se invoca con Void y no se espera respuesta (solo que responda OK)
            if (getResponse.getEntity() == null && responseType == Void.class) {
                entityResponse = null;
                log.trace("Entity response [vacio]");
            } else {
                entityResponse = getResponse.readEntity(responseType);
                log.trace("Entity response [" + entityResponse + "]");
            }

            return entityResponse;

        } catch (ProcessingException e) {
            throw new DebinException("Error de comunicaciones", e);

        } finally {
            if (getResponse != null) {
                getResponse.close();
            }
        }

    }

    private <Res, Req> Res genericPost(Req request, Class<Res> responseType, String path) throws DebinException {
        log.trace("Post cliente request [" + request + "]");

        Response postResponse = null;
        try {
            postResponse = securityManager.buildRequest(debinBaseUrl + path, null)
                    .accept(MediaType.APPLICATION_JSON)
                    .post(Entity.entity(request, MediaType.APPLICATION_JSON));
            log.trace("Post cliente response [" + postResponse.getStatus() + "/" + postResponse.getStatusInfo() + "]");

            if (postResponse.getStatus() != Response.Status.OK.getStatusCode()) {
                throw new DebinException("La consulta no devolvio OK. Status: [" + postResponse.getStatus() + "/" + postResponse.getStatusInfo() + "]");
            }

            Res entityResponse;
            //Caso especial para cuando se invoca con Void y no se espera respuesta (solo que responda OK)
            if (postResponse.getEntity() == null && responseType == Void.class) {
                entityResponse = null;
                log.trace("Entity response [vacio]");
            } else {
                entityResponse = postResponse.readEntity(responseType);
                log.trace("Entity response [" + entityResponse + "]");
            }

            return entityResponse;

        } catch (ProcessingException e) {
            throw new DebinException("Error de comunicaciones", e);

        } finally {
            if (postResponse != null) {
                postResponse.close();
            }
        }
    }

    private <Res, Req> Res genericDelete(Req request, Class<Res> responseType, String path, Map<String, Object> templateParameters) throws DebinException {
        log.trace("Delete cliente request [" + request + "]");

        Response deleteResponse = null;
        try {
            deleteResponse = securityManager.buildRequest(debinBaseUrl + path, templateParameters)
                    .accept(MediaType.APPLICATION_JSON)
                    .delete();
            log.trace("Delete cliente response [" + deleteResponse.getStatus() + "/" + deleteResponse.getStatusInfo() + "]");

            if (deleteResponse.getStatus() != Response.Status.OK.getStatusCode()) {
                throw new DebinException("La consulta no devolvio OK. Status: [" + deleteResponse.getStatus() + "/" + deleteResponse.getStatusInfo() + "]");
            }

            Res entityResponse = deleteResponse.readEntity(responseType);
            log.trace("Entity response [" + entityResponse + "]");

            return entityResponse;

        } catch (ProcessingException e) {
            throw new DebinException("Error de comunicaciones", e);

        } finally {
            if (deleteResponse != null) {
                deleteResponse.close();
            }
        }

    }

}
