package ar.com.sdd.debin.core;

import ar.com.sdd.debin.io.*;
import ar.com.sdd.debin.token.SecurityManager;
import org.apache.log4j.Logger;

import javax.ws.rs.ProcessingException;
import javax.ws.rs.client.Entity;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.util.HashMap;
import java.util.Map;

public class DebinConnector {

    private static final Logger log = Logger.getLogger(DebinConnector.class);

    public static final int MAX_ENTITY_SIZE = 1024 * 8;

    /**
     * URL de API debin
     */
    private String debinBaseUrl;

    /**
     * El administrador de tokens. Si no se usa autenticacion por token, evita hacer la conexion para recuperar el token
     */
    private SecurityManager securityManager;

    /**
     * Crea un conector sencillo sin autenticacion, ni keyStores
     *
     * @param environment
     */
    public DebinConnector(DebinEnvironment environment) {
        this(null, null, null, null, environment.getAuthBaseUrl(), environment.getDebinBaseUrl(), null, null, null, false);
    }

    public DebinConnector(String authBaseUrl, String debinBaseUrl) {
        this(null, null, null, null, authBaseUrl, debinBaseUrl, null, null, null, false);
    }

    public DebinConnector(String authBaseUrl, String debinBaseUrl, String keyStorePath, String keyStoreOpenPassword, String keyStorePrivateKeyPassword) {
        this(null, null, null, null, authBaseUrl, debinBaseUrl, keyStorePath, keyStoreOpenPassword, keyStorePrivateKeyPassword, false);
    }

    public DebinConnector(String clientId, String clientSecret, String username, String password, String authBaseUrl, String debinBaseUrl, String keyStorePath, String keyStoreOpenPassword, String keyStorePrivateKeyPassword, boolean useTokenAuthentication) {
        log.debug("Creando DebinConnector con clientId:[" + clientId + "], clientSecret:[" + clientSecret + "], username:[" + username + "], " +
                "password:[" + password + "], authBaseUrl:[" + authBaseUrl + "], debinBaseUrl:[" + debinBaseUrl + "]," +
                "keyStorePath:[" + keyStorePath + "], keyStoreOpenPassword:[" + keyStoreOpenPassword + "], keyStorePrivateKeyPassword:[" + keyStorePrivateKeyPassword + "], " +
                "useTokenAuthentication: [" + useTokenAuthentication + "]");
        this.debinBaseUrl = debinBaseUrl;

        //Se inicializa el securityManager
        securityManager = new SecurityManager(clientId, clientSecret, username, password, authBaseUrl, keyStorePath, keyStoreOpenPassword, keyStorePrivateKeyPassword, useTokenAuthentication);
    }


    /*
     *
     * Metodos Debin
     *
     */

    public CrearDebinResponse crearDebin(CrearDebinRequest request) throws DebinException {
        request.validate();
        return genericPost(request, CrearDebinResponse.class, "/Debin/Debin");
    }

    public ConsultarDebinResponse consultarDebin(ConsultarDebinRequest request) throws DebinException {
        return genericGet(request, ConsultarDebinResponse.class, "/Debin/Debin/{id}", "id", request.getDebinId());
    }

    public ConfirmarDebitoResponse confirmarDebito(ConfirmarDebitoRequest request) throws DebinException {
        return genericPost(request, ConfirmarDebitoResponse.class, "/Debin/ConfirmaDebito");
    }

    public EliminarDebinResponse eliminarDebin(EliminarDebinRequest request) throws DebinException {
        Map<String, Object> templateParameters = new HashMap<>();
        templateParameters.put("id", request.getDebinId());
        templateParameters.put("cuit", request.getCuit());
        return genericDelete(request, EliminarDebinResponse.class, "/Debin/Debin/{id}/{cuit}", templateParameters);
    }

    public ListarDebinResponse listarDebin(ListarDebinRequest request) throws DebinException {
        return genericPost(request, ListarDebinResponse.class, "/Debin/Lista");
    }

    /*
     *
     *  Metodos Banco
     *
     */

    public EstablecerModoBancoResponse establecerModoBanco(EstablecerModoBancoRequest request) throws DebinException {
        return genericPost(request, EstablecerModoBancoResponse.class, "/Bancos/ModoBanco");
    }

    public AltaCuentaEspecialBancoResponse altaCuentaEspecialBanco(AltaCuentaEspecialBancoRequest request) throws DebinException {
        return genericPost(request, AltaCuentaEspecialBancoResponse.class, "/Bancos/CuentaEspecial");
    }

    public ConsultaCuentaEspecialBancoResponse consultaCuentaEspecialBanco(ConsultaCuentaEspecialBancoRequest request) throws DebinException {
        return genericGet(request, ConsultaCuentaEspecialBancoResponse.class, " /Bancos /CuentaEspecial");
    }



    public void respuestaDebin(RespuestaDebinRequest request) throws DebinException {
        genericPost(request, Void.class, "/RespuestaDebin");
    }

    public int echo() throws DebinException {
        Response getResponse = null;
        try {
            getResponse = securityManager.buildRequest(debinBaseUrl + "/Echo", null)
                .accept(MediaType.APPLICATION_JSON)
                .get();
            return getResponse.getStatus();
        } finally {
            if (getResponse != null) {
                getResponse.close();
            }
        }
    }


    /*
     *
     * Metodos de Vendedores y Compradores
     *
     */

    public AdherirVendedorResponse adherirVendedor(AdherirVendedorRequest request) throws DebinException {
        return genericPost(request, AdherirVendedorResponse.class, "/Vendedor/Adhesion");
    }

    public AdherirCompradorResponse adherirComprador(AdherirCompradorRequest request) throws DebinException {
        return genericPost(request, AdherirCompradorResponse.class, "/Comprador/Adhesion");
    }

    public QuitarVendedorResponse quitarVendedor(QuitarVendedorRequest request) throws DebinException {
        return genericPost(request, QuitarVendedorResponse.class, "/Vendedor/BajaCuenta");
    }

    public QuitarCompradorResponse quitarComprador(QuitarCompradorRequest request) throws DebinException {
        return genericPost(request, QuitarCompradorResponse.class, "/Comprador/BajaCuenta");
    }

    public ConsultarCompradorResponse consultarComprador(ConsultarCompradorRequest request) throws DebinException {
        return genericGet(request, ConsultarCompradorResponse.class, "/Comprador/Comprador/{cuit}", "cuit", request.getCuit());
    }

    public ConsultarVendedorResponse consultarVendedor(ConsultarVendedorRequest request) throws DebinException {
        return genericGet(request, ConsultarVendedorResponse.class, "/Vendedor/Vendedor/{cuit}", "cuit", request.getCuit());
    }

    public AdherirRecurrenciaCompradorResponse adherirRecurrenciaComprador(AdherirRecurrenciaCompradorRequest request) throws DebinException {
        return genericPost(request, AdherirRecurrenciaCompradorResponse.class, "/Comprador/AdherirRecurrencia");
    }

    public AdherirRecurrenciaVendedorResponse adherirRecurrenciaVendedor(AdherirRecurrenciaVendedorRequest request) throws DebinException {
        return genericPost(request, AdherirRecurrenciaVendedorResponse.class, "/Vendedor/AdherirRecurrencia");
    }

    public CompradorRecurrenciaListaCompradorResponse compradorRecurrenciaListaComprador(CompradorRecurrenciaListaCompradorRequest request) throws DebinException {
        return genericPost(request, CompradorRecurrenciaListaCompradorResponse.class, "/Comprador/CompradorRecurrenciaLista");
    }

    public VendedorRecurrenciaListaVendedorResponse vendedorRecurrenciaListaVendedor(VendedorRecurrenciaListaVendedorRequest request) throws DebinException {
        return genericPost(request, VendedorRecurrenciaListaVendedorResponse.class, "/Vendedor/VendedorRecurrenciaLista");
    }

    public VendedorListadoVendedorResponse vendedorListadoVendedor(VendedorListadoVendedorRequest request) throws DebinException {
        return genericPost(request, VendedorListadoVendedorResponse.class, "/Vendedor/ListadoVendedores");
    }

    public CrearPrestacionVendedorResponse crearPrestacionVendedor(CrearPrestacionVendedorRequest request) throws DebinException {
        return genericPost(request, CrearPrestacionVendedorResponse.class, "/Vendedor/Prestacion");
    }

    public ConsultarPrestacionVendedorResponse consultarPrestacionVendedor(ConsultarPrestacionVendedorRequest request) throws DebinException {
        return genericGet(request, ConsultarPrestacionVendedorResponse.class, "/Vendedor/Prestacion/{cuit}", "cuit", request.getCuit());
    }

    public EliminarPrestacionVendedorResponse eliminarPrestacionVendedor(EliminarPrestacionVendedorRequest request) throws DebinException {
        Map<String, Object> templateParameters = new HashMap<>();
        templateParameters.put("cuit", request.getCuit());
        templateParameters.put("prestacion", request.getPrestacion());
        return genericDelete(request, EliminarPrestacionVendedorResponse.class, " /Vendedor/DeletePrestacion/{cuit}/{prestacion}", templateParameters);
    }

    public SolicitudContraCargoCompradorResponse solicitudContraCargoComprador(SolicitudContraCargoCompradorRequest request) throws DebinException {
        return genericPost(request, SolicitudContraCargoCompradorResponse.class, "/Comprador/SolicitudContraCargo");
    }

    public OrdenDeCreditoCompradorResponse ordenDeCreditoComprador(OrdenDeCreditoCompradorRequest request) throws DebinException {
        return genericPost(request, OrdenDeCreditoCompradorResponse.class, "/Comprador/OrdenDeCredito");
    }

    public OrdenDeDebitoCompradorResponse ordenDeDebitoComprador(OrdenDeDebitoCompradorRequest request) throws DebinException {
        return genericPost(request, OrdenDeDebitoCompradorResponse.class, "/Comprador/OrdenDeDebito");
    }

    /*
     *
     * Metodos de Alias
     *
     */

    public ConsultarAliasResponse consultarAlias(ConsultarAliasRequest request) throws DebinException {
        return genericPost(request, ConsultarAliasResponse.class, "/Alias/Consultar");
    }

    public ModificarAliasResponse modificarAlias(ModificarAliasRequest request) throws DebinException {
        return genericPost(request, ModificarAliasResponse.class, "/Alias/Agregar");
    }



    /*
     *
     * Metodos de CVU
     *
     */

    public AltaCvuResponse altaCvu(AltaCvuRequest request) throws DebinException {
        return genericPost(request, AltaCvuResponse.class, "/Debin/CVU/AltaCVU");
    }

    public ModificacionCvuResponse modificacionCvu(ModificacionCvuRequest request) throws DebinException {
        Map<String, Object> templateParameters = new HashMap<>();
        templateParameters.put("cvu", request.getCvu().getCvu());
        templateParameters.put("cuit", request.getCvu().getCuit());
        return genericPut(request, ModificacionCvuResponse.class, null, "/Debin/CVU/ModifCVU/{cvu}/{cuit}", MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON, templateParameters);
    }

    public BajaCvuResponse bajaCvu(BajaCvuRequest request) throws DebinException {
        Map<String, Object> templateParameters = new HashMap<>();
        templateParameters.put("cvu", request.getCvu().getCvu());
        templateParameters.put("cuit", request.getCvu().getCuit());
        return genericDelete(request, BajaCvuResponse.class, null, "/Debin/CVU/BajaCVU/{cvu}/{cuit}", templateParameters);
    }


    /*
     *
     * Metodos genericos
     *
     */

    public <Res, Req> Res genericGet(Req request, Class<Res> responseType, String path) throws DebinException {
        return genericGet(request, responseType, path, null);
    }

    public <Res, Req, Err> Res genericGet(Req request, Class<Res> responseType, Class<Err> errorType, String path) throws DebinException {
        return genericGet(request, responseType, errorType, path, null);
    }

    public <Res, Req> Res genericGet(Req request, Class<Res> responseType, String path, String templateKey, String templateValue) throws DebinException {
        Map<String, Object> templateParameters = new HashMap<>();
        templateParameters.put(templateKey, templateValue);
        return genericGet(request, responseType, path, templateParameters);
    }

    private <Res, Req> Res genericGet(Req request, Class<Res> responseType, String path, Map<String, Object> templateParameters) throws DebinException {
        return genericGet(request, responseType, null, path, templateParameters);
    }

    private <Res, Req, Err> Res genericGet(Req request, Class<Res> responseType, Class<Err> errorType, String path, Map<String, Object> templateParameters) throws DebinException {
        log.trace("[DebinConnector] Get cliente request [" + request + "]");

        Response getResponse = null;
        try {
            getResponse = securityManager.buildRequest(debinBaseUrl + path, templateParameters)
                    .accept(MediaType.APPLICATION_JSON)
                    .get();
            log.trace("[DebinConnector] Get cliente response [" + getResponse.getStatus() + "/" + getResponse.getStatusInfo() + "]");

            checkSuccessfulOrException(getResponse, errorType);

            return processResponse(getResponse, responseType);

        } catch (ProcessingException e) {
            throw new DebinException("Error de comunicaciones", e);

        } finally {
            if (getResponse != null) {
                getResponse.close();
            }
        }

    }

    public <Res, Req> Res genericPost(Req request, Class<Res> responseType, String path) throws DebinException {
        return genericPost(request, responseType, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON);
    }

    public <Res, Req> Res genericPost(Req request, Class<Res> responseType, String path, String acceptMediaType, String postMediaType) throws DebinException {
        return genericPost(request, responseType,null, path, acceptMediaType, postMediaType);
    }

    public <Res, Req, Err> Res genericPost(Req request, Class<Res> responseType, Class<Err> errorType, String path) throws DebinException {
        return genericPost(request, responseType, errorType, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON);
    }

    public <Res, Req, Err> Res genericPost(Req request, Class<Res> responseType, Class<Err> errorType, String path, String acceptMediaType, String postMediaType) throws DebinException {
        log.trace("[DebinConnector] Post cliente request [" + request + "]");

        Response postResponse = null;
        try {
            postResponse = securityManager.buildRequest(debinBaseUrl + path, null)
                    .accept(acceptMediaType)
                    .post(Entity.entity(request, postMediaType));
            log.trace("[DebinConnector] Post cliente response [" + postResponse.getStatus() + "/" + postResponse.getStatusInfo() + "]");

            checkSuccessfulOrException(postResponse, errorType);

            return processResponse(postResponse, responseType);

        } catch (ProcessingException e) {
            throw new DebinException("Error de comunicaciones", e);

        } finally {
            if (postResponse != null) {
                postResponse.close();
            }
        }
    }

    public <Res, Req> Res genericPut(Req request, Class<Res> responseType, String path) throws DebinException {
        return genericPut(request, responseType, null, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON, null);
    }

    public <Res, Req, Err> Res genericPut(Req request, Class<Res> responseType, Class<Err> errorType, String path) throws DebinException {
        return genericPut(request, responseType, errorType, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON, null);
    }

    public <Res, Req, Err> Res genericPut(Req request, Class<Res> responseType, Class<Err> errorType, String path, String acceptMediaType, String postMediaType, Map<String, Object> templateParameters) throws DebinException {
        log.trace("[DebinConnector] Put cliente request [" + request + "]");

        Response putResponse = null;
        try {
            putResponse = securityManager.buildRequest(debinBaseUrl + path, templateParameters)
                    .accept(acceptMediaType)
                    .put(Entity.entity(request, postMediaType));
            log.trace("[DebinConnector] Put cliente response [" + putResponse.getStatus() + "/" + putResponse.getStatusInfo() + "]");

            checkSuccessfulOrException(putResponse, errorType);

            return processResponse(putResponse, responseType);

        } catch (ProcessingException e) {
            throw new DebinException("Error de comunicaciones", e);

        } finally {
            if (putResponse != null) {
                putResponse.close();
            }
        }
    }


    public <Res, Req> Res genericDelete(Req request, Class<Res> responseType, String path, Map<String, Object> templateParameters) throws DebinException {
        return genericDelete(request, responseType, null, path, templateParameters);
    }

    public <Res, Req, Err> Res genericDelete(Req request, Class<Res> responseType, Class<Err> errorType, String path, Map<String, Object> templateParameters) throws DebinException {
        log.trace("[DebinConnector] Delete cliente request [" + request + "]");

        Response deleteResponse = null;
        try {
            deleteResponse = securityManager.buildRequest(debinBaseUrl + path, templateParameters)
                    .accept(MediaType.APPLICATION_JSON)
                    .delete();
            log.trace("[DebinConnector] Delete cliente response [" + deleteResponse.getStatus() + "/" + deleteResponse.getStatusInfo() + "]");

            checkSuccessfulOrException(deleteResponse, errorType);

            Res entityResponse = deleteResponse.readEntity(responseType);
            log.trace("[DebinConnector] Entity response [" + entityResponse + "]");

            return entityResponse;

        } catch (ProcessingException e) {
            throw new DebinException("Error de comunicaciones", e);

        } finally {
            if (deleteResponse != null) {
                deleteResponse.close();
            }
        }

    }


    /**
     * Comprueba que venga una respuesta successful (200 o 2NN), y sino tira una excepcion. Si vino informado un errorType (el body de la respuesta erronea tiene un json),
     * este se sube a la excepcion.
     *
     * @param response
     * @param errorType
     * @param <Err>
     * @throws DebinException
     */
    private <Err> void checkSuccessfulOrException(Response response, Class<Err> errorType) throws DebinException {
        if (response.getStatusInfo().getFamily() != Response.Status.Family.SUCCESSFUL) {
            throw new DebinException("La consulta no devolvio OK. Status: [" + response.getStatus() + "/" + response.getStatusInfo() + "]");
        }
    }

    /**
     * Devuelve la response con la entity correcta considerando el caso en que pueda venir vacia
     *
     * @param response
     * @param responseType
     * @param <Res>
     * @return
     */
    private <Res> Res processResponse(Response response, Class<Res> responseType) {
        Res entityResponse;
        //Caso especial para cuando se invoca con Void y no se espera respuesta (solo que responda OK)
        if (response.getEntity() == null || responseType == Void.class) {
            entityResponse = null;
            log.trace("[DebinConnector] Entity response [vacio]");
        } else {
            entityResponse = response.readEntity(responseType);
            log.trace("[DebinConnector] Entity response [" + entityResponse + "]");
        }
        return entityResponse;
    }

    public static String getCacheSafeValue() {
        long ts = System.currentTimeMillis();
        return Long.toString(ts, Character.MAX_RADIX);
    }

}
