package ar.com.sdd.debin.token;

import ar.com.sdd.debin.core.Authenticator;
import ar.com.sdd.debin.core.DebinException;
import ar.com.sdd.debin.log.ClientLoggingFilter;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;

import javax.net.ssl.SSLSession;
import javax.ws.rs.ProcessingException;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.client.Client;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.Entity;
import javax.ws.rs.client.Invocation;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.Form;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.io.FileInputStream;
import java.security.KeyStore;
import java.util.Map;

public class SecurityManager {

    private static final Logger log = Logger.getLogger(SecurityManager.class);

    /**
     * Id del cliente que se utiliza para la autenticacion basica previa al servicio de autenticacion
     */
    private String clientId;

    /**
     * Password del cliente que se utiliza para la autenticacion basica previa al servicio de autenticacion
     */
    private String clientSecret;

    /**
     * Nombre de usuario del servicio de autenticacion para obtener el token
     */
    private String username;

    /**
     * Password del usuario del servicio de autenticacion para obtener el token
     */
    private String password;

    /**
     * URL de autenticacion
     */
    private String authBaseUrl;

    /**
     * Indica si se usa o no la autenticacion con token y su posterior envio en cada llamada
     */
    private boolean useTokenAuthentication;

    /**
     * El token de esta sesion. Puede estar expirado o no existir.
     */
    private DebinToken actualToken;

    /**
     * KeyStore de certificados privados (nuestros) y publicos (de ellos)
     */
    private KeyStore keyStore;

    /**
     * Password para la clave privada nuestra dentro del keystore
     */
    private String keyStorePrivateKeyPassword;

    /**
     * Filtros que logguean los request/response
     */
    private ClientLoggingFilter clientLoggingFilter;


    public SecurityManager(String clientId, String clientSecret, String username, String password, String authBaseUrl, String keyStorePath, String keyStoreOpenPassword, String keyStorePrivateKeyPassword, boolean useTokenAuthentication) {
        this.clientId = clientId;
        this.clientSecret = clientSecret;
        this.username = username;
        this.password = password;
        this.authBaseUrl = authBaseUrl;
        this.keyStorePrivateKeyPassword = keyStorePrivateKeyPassword;
        this.useTokenAuthentication = useTokenAuthentication;

        if (!StringUtils.isEmpty(keyStorePath)) {
            try {
                keyStore = loadStore(keyStorePath, keyStoreOpenPassword);
            } catch (Exception e) {
                log.error("No se pudo cargar el keystore de [" + keyStorePath + "] con password de apertura [" + keyStoreOpenPassword + "]", e);
            }
        }

        //Inicializo los filtros
        clientLoggingFilter = new ClientLoggingFilter();

    }

    private KeyStore loadStore(String storePath, String storePassword) throws Exception {
        KeyStore store = KeyStore.getInstance("JKS");
        store.load(new FileInputStream(storePath), storePassword.toCharArray());
        return store;
    }

    /**
     * Si hay un token no expirado, devuelve ese, sino <br>
     * pide un nuevo token al servicio de autenticacion
     *
     * @return un token que puede estar expirado o no
     */
    private DebinToken requestToken() throws DebinException {
        log.trace("Pidiendo el token");

        //Chequea si hay token y si esta o no expirado. Si no hay, o esta expirado, pide uno nuevo
        if (actualToken.isExpired()) {

            log.trace("El token esta expirado, se pide uno nuevo");
            //Pide un nuevo token
            Client client = ClientBuilder.newClient();
            try {

                WebTarget authResource = client.target(authBaseUrl + "/auth/apiAuthv2/token");
                authResource.register(new Authenticator(clientId, clientSecret));

                Form form = new Form();
                form.param("grant_type", "password");
                form.param("username", username);
                form.param("password", password);

                try {
                    actualToken = authResource.request(MediaType.APPLICATION_JSON)
                            .accept(MediaType.APPLICATION_JSON)
                            .post(Entity.entity(form, MediaType.APPLICATION_FORM_URLENCODED), DebinToken.class);


                } catch (WebApplicationException e) {
                    Response response = e.getResponse();
                    throw new DebinException("Error al pedir el token, con response [" + response + "]", e);

                } catch (ProcessingException e) {
                    throw new DebinException("Error al procesar el request del token", e);
                }
            } finally {
                if (client!=null) client.close();
            }
        }

        log.trace("Devolviendo el token [" + actualToken + "]");
        return actualToken;
    }

    /**
     * Genera un request agregandole el header con autenticacion si lo necesita
     *
     * @param url
     * @return
     * @throws DebinException
     */
    public Invocation.Builder buildRequest(String url, Map<String, Object> templateParameters) throws DebinException {

        ClientBuilder builder = ClientBuilder.newBuilder();
        if (keyStore != null) {
            log.debug("Usando keyStore [" + keyStore +  "], keyStorePrivateKeyPassword [" + keyStorePrivateKeyPassword + "]");
            //Uso el mismo keystore para los certificados privados (keyStore) y los publicos (tustStore)
            builder.keyStore(keyStore, keyStorePrivateKeyPassword);
            builder.trustStore(keyStore);
            builder.hostnameVerifier((String hostName, SSLSession session) -> true);
        }

        WebTarget resource = builder.build().target(url);
        resource.register(clientLoggingFilter);

        if (templateParameters != null) {
            resource = resource.resolveTemplates(templateParameters);
        }

        Invocation.Builder request = resource.request();

        if (useTokenAuthentication) {
            return request.header("Authorization", "Bearer " + requestToken().getAccessToken());
        } else {
            return request;
        }
    }
}
