package ar.com.sdd.debin.core;

import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSecurityManager;
import ar.com.sdd.debin.io.*;
import org.apache.http.HttpStatus;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.ws.rs.client.Invocation;
import javax.ws.rs.core.MediaType;
import java.util.HashMap;
import java.util.Map;

public class DebinConnector implements RestSecurityManager {

    private static final Logger log = LogManager.getLogger(DebinConnector.class);

    private final RestConnector restConnector;

    private SecurityManager securityManager;

    public DebinConnector(DebinConnectorContext context) {
        log.debug("Creando DebinConnector con context:[" + context + "]");
        RestConnectorEnvironment environment = new RestConnectorEnvironment(context.getDebinBaseUrl(), context.getKeyStorePath(), context.getKeyStorePassword(), null, null);
        //En el jks donde estan las credenciales de login de Gire, tambien esta el cerfificado para el trustStore, por use usamos el ultimo parametro en true (useKeyStoreWithEmbeddedTrustStore)
        restConnector = new RestConnector(environment, this, true, true);
    }

    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        return builder;
    }

    @Override
    public boolean getDisableHTTPSErrors() {
        return false;
    }

    /*
     *
     * Metodos Debin
     *
     */

    public CrearDebinResponse crearDebin(CrearDebinRequest request) throws RestConnectorException, DebinException {
        request.validate();
        return restConnector.genericPost(request, CrearDebinResponse.class, "/Debin/Debin");
    }

    public ConsultarDebinResponse consultarDebin(ConsultarDebinRequest request) throws RestConnectorException, DebinException {
        return restConnector.genericGet(request, ConsultarDebinResponse.class, "/Debin/Debin/{id}", "id", request.getDebinId());
    }

    public ConfirmarDebitoResponse confirmarDebito(ConfirmarDebitoRequest request) throws RestConnectorException, DebinException {
        return restConnector.genericPost(request, ConfirmarDebitoResponse.class, "/Debin/ConfirmaDebito");
    }

    public EliminarDebinResponse eliminarDebin(EliminarDebinRequest request) throws RestConnectorException, DebinException {
        Map<String, Object> templateParameters = new HashMap<>();
        templateParameters.put("id", request.getDebinId());
        templateParameters.put("cuit", request.getCuit());
        return restConnector.genericDelete(request, EliminarDebinResponse.class, "/Debin/Debin/{id}/{cuit}", templateParameters);
    }

    public ListarDebinResponse listarDebin(ListarDebinRequest request) throws RestConnectorException, DebinException {
        return restConnector.genericPost(request, ListarDebinResponse.class, "/Debin/Lista");
    }

    /*
     *
     *  Metodos Banco
     *
     */

    public EstablecerModoBancoResponse establecerModoBanco(EstablecerModoBancoRequest request) throws RestConnectorException, DebinException {
        return restConnector.genericPost(request, EstablecerModoBancoResponse.class, "/Bancos/ModoBanco");
    }

    public AltaCuentaEspecialBancoResponse altaCuentaEspecialBanco(AltaCuentaEspecialBancoRequest request) throws RestConnectorException, DebinException {
        return restConnector.genericPost(request, AltaCuentaEspecialBancoResponse.class, "/Bancos/CuentaEspecial");
    }

    public ConsultaCuentaEspecialBancoResponse consultaCuentaEspecialBanco(ConsultaCuentaEspecialBancoRequest request) throws RestConnectorException, DebinException {
        return restConnector.genericGet(request, ConsultaCuentaEspecialBancoResponse.class, " /Bancos /CuentaEspecial");
    }


    public void respuestaDebin(RespuestaDebinRequest request) throws RestConnectorException, DebinException {
        restConnector.genericPost(request, Void.class, "/RespuestaDebin");
    }

    public int echo() throws RestConnectorException, DebinException {
        restConnector.genericGet(null, null, "/Echo");
        return HttpStatus.SC_OK;
    }


    /*
     *
     * Metodos de Vendedores y Compradores
     *
     */

    public AdherirVendedorResponse adherirVendedor(AdherirVendedorRequest request) throws RestConnectorException, DebinException {
        return restConnector.genericPost(request, AdherirVendedorResponse.class, "/Vendedor/Adhesion");
    }

    public AdherirCompradorResponse adherirComprador(AdherirCompradorRequest request) throws RestConnectorException, DebinException {
        return restConnector.genericPost(request, AdherirCompradorResponse.class, "/Comprador/Adhesion");
    }

    public QuitarVendedorResponse quitarVendedor(QuitarVendedorRequest request) throws RestConnectorException, DebinException {
        return restConnector.genericPost(request, QuitarVendedorResponse.class, "/Vendedor/BajaCuenta");
    }

    public QuitarCompradorResponse quitarComprador(QuitarCompradorRequest request) throws RestConnectorException, DebinException {
        return restConnector.genericPost(request, QuitarCompradorResponse.class, "/Comprador/BajaCuenta");
    }

    public ConsultarCompradorResponse consultarComprador(ConsultarCompradorRequest request) throws RestConnectorException, DebinException {
        return restConnector.genericGet(request, ConsultarCompradorResponse.class, "/Comprador/Comprador/{cuit}", "cuit", request.getCuit());
    }

    public ConsultarVendedorResponse consultarVendedor(ConsultarVendedorRequest request) throws RestConnectorException, DebinException {
        return restConnector.genericGet(request, ConsultarVendedorResponse.class, "/Vendedor/Vendedor/{cuit}", "cuit", request.getCuit());
    }

    public AdherirRecurrenciaCompradorResponse adherirRecurrenciaComprador(AdherirRecurrenciaCompradorRequest request) throws RestConnectorException, DebinException {
        return restConnector.genericPost(request, AdherirRecurrenciaCompradorResponse.class, "/Comprador/AdherirRecurrencia");
    }

    public AdherirRecurrenciaVendedorResponse adherirRecurrenciaVendedor(AdherirRecurrenciaVendedorRequest request) throws RestConnectorException, DebinException {
        return restConnector.genericPost(request, AdherirRecurrenciaVendedorResponse.class, "/Vendedor/AdherirRecurrencia");
    }

    public CompradorRecurrenciaListaCompradorResponse compradorRecurrenciaListaComprador(CompradorRecurrenciaListaCompradorRequest request) throws RestConnectorException, DebinException {
        return restConnector.genericPost(request, CompradorRecurrenciaListaCompradorResponse.class, "/Comprador/CompradorRecurrenciaLista");
    }

    public VendedorRecurrenciaListaVendedorResponse vendedorRecurrenciaListaVendedor(VendedorRecurrenciaListaVendedorRequest request) throws RestConnectorException, DebinException {
        return restConnector.genericPost(request, VendedorRecurrenciaListaVendedorResponse.class, "/Vendedor/VendedorRecurrenciaLista");
    }

    public VendedorListadoVendedorResponse vendedorListadoVendedor(VendedorListadoVendedorRequest request) throws RestConnectorException, DebinException {
        return restConnector.genericPost(request, VendedorListadoVendedorResponse.class, "/Vendedor/ListadoVendedores");
    }

    public CrearPrestacionVendedorResponse crearPrestacionVendedor(CrearPrestacionVendedorRequest request) throws RestConnectorException, DebinException {
        return restConnector.genericPost(request, CrearPrestacionVendedorResponse.class, "/Vendedor/Prestacion");
    }

    public ConsultarPrestacionVendedorResponse consultarPrestacionVendedor(ConsultarPrestacionVendedorRequest request) throws RestConnectorException, DebinException {
        return restConnector.genericGet(request, ConsultarPrestacionVendedorResponse.class, "/Vendedor/Prestacion/{cuit}", "cuit", request.getCuit());
    }

    public EliminarPrestacionVendedorResponse eliminarPrestacionVendedor(EliminarPrestacionVendedorRequest request) throws RestConnectorException, DebinException {
        Map<String, Object> templateParameters = new HashMap<>();
        templateParameters.put("cuit", request.getCuit());
        templateParameters.put("prestacion", request.getPrestacion());
        return restConnector.genericDelete(request, EliminarPrestacionVendedorResponse.class, " /Vendedor/DeletePrestacion/{cuit}/{prestacion}", templateParameters);
    }

    public SolicitudContraCargoCompradorResponse solicitudContraCargoComprador(SolicitudContraCargoCompradorRequest request) throws RestConnectorException, DebinException {
        return restConnector.genericPost(request, SolicitudContraCargoCompradorResponse.class, "/Comprador/SolicitudContraCargo");
    }

    public OrdenDeCreditoCompradorResponse ordenDeCreditoComprador(OrdenDeCreditoCompradorRequest request) throws RestConnectorException, DebinException {
        return restConnector.genericPost(request, OrdenDeCreditoCompradorResponse.class, "/Comprador/OrdenDeCredito");
    }

    public OrdenDeDebitoCompradorResponse ordenDeDebitoComprador(OrdenDeDebitoCompradorRequest request) throws RestConnectorException, DebinException {
        return restConnector.genericPost(request, OrdenDeDebitoCompradorResponse.class, "/Comprador/OrdenDeDebito");
    }

    /*
     *
     * Metodos de Alias
     *
     */

    public ConsultarAliasResponse consultarAlias(ConsultarAliasRequest request) throws RestConnectorException, DebinException {
        return restConnector.genericPost(request, ConsultarAliasResponse.class, "/Alias/Consultar");
    }

    public ModificarAliasResponse modificarAlias(ModificarAliasRequest request) throws RestConnectorException, DebinException {
        return restConnector.genericPost(request, ModificarAliasResponse.class, "/Alias/Agregar");
    }



    /*
     *
     * Metodos de CVU
     *
     */

    public AltaCvuResponse altaCvu(AltaCvuRequest request) throws RestConnectorException, DebinException {
        return restConnector.genericPost(request, AltaCvuResponse.class, "/Debin/CVU/AltaCVU");
    }

    public ModificacionCvuResponse modificacionCvu(ModificacionCvuRequest request) throws RestConnectorException, DebinException {
        Map<String, Object> templateParameters = new HashMap<>();
        templateParameters.put("cvu", request.getCvu().getCvu());
        templateParameters.put("cuit", request.getCvu().getCuit());
        return restConnector.genericPut(request, ModificacionCvuResponse.class, null, "/Debin/CVU/ModifCVU/{cvu}/{cuit}", MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON, templateParameters);
    }

    public BajaCvuResponse bajaCvu(BajaCvuRequest request) throws RestConnectorException, DebinException {
        Map<String, Object> templateParameters = new HashMap<>();
        templateParameters.put("cvu", request.getCvu().getCvu());
        templateParameters.put("cuit", request.getCvu().getCuit());
        return restConnector.genericDelete(request, BajaCvuResponse.class, null, "/Debin/CVU/BajaCVU/{cvu}/{cuit}", templateParameters);
    }


}
