package ar.com.sdd.ebfapi.client.core;

import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSecurityManager;
import ar.com.sdd.commons.rest.model.OAuth2Credential;
import ar.com.sdd.commons.rest.model.TokenOAuth2;
import ar.com.sdd.commons.util.SimpleCache;
import ar.com.sdd.commons.util.SimpleCacheManager;
import ar.com.sdd.ebfapi.client.model.ActionLink;
import ar.com.sdd.ebfapi.client.model.RedirectionActionLinkData;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.ws.rs.client.Invocation;

@SuppressWarnings("unused")
public class CuentaSimpleEbfApiConnector implements RestSecurityManager {

    private final static Logger log = LogManager.getLogger(CuentaSimpleEbfApiConnector.class);

    private final RestConnector restConnector;
    private final String clientId;
    private final String clientSecret;
    private final String applicationId;
    private final String ownerName;
    private boolean doingLogin;
    private boolean forceRefresh;

    public CuentaSimpleEbfApiConnector(CuentaSimpleEbfApiConnectorContext context) {
        this.clientId = context.getClientId();
        this.clientSecret = context.getClientSecret();
        this.applicationId = context.getApplicationId();
        this.ownerName = context.getOwnerName();

        log.trace("Creando CuentaSimpleEbfApiConnector con {}", context);

        RestConnectorEnvironment environment = new RestConnectorEnvironment(context.getBaseUrl());
        restConnector = new RestConnector(environment, this);
    }

    public ActionLink postGenerateRedirectionActionLink(RedirectionActionLinkData redirectionActionLinkData) throws RestConnectorException {
        final String path = "/actionlink/aid/" + applicationId  + "/owner/" + ownerName + "/redirectLink";
        log.debug("[postGenerateRedirectionActionLink] Por solicitar un link de redireccionamiento con path [{}] y {}", path, redirectionActionLinkData);
        return restConnector.genericPost(redirectionActionLinkData, ActionLink.class, path);
    }

    private String getAccessToken() throws RestConnectorException {
        SimpleCache cache = SimpleCacheManager.getInstance().getCache(CuentaSimpleEbfApiConnector.class.getName());
        String cacheKey = SimpleCacheManager.buildKey("accessToken");
        String accessToken = (String) cache.get(cacheKey);
        if (accessToken == null || forceRefresh) {
            OAuth2Credential oAuth2Credential = new OAuth2Credential();
            oAuth2Credential.setClientId(clientId);
            oAuth2Credential.setClientSecret(clientSecret);
            oAuth2Credential.setGrantType("client_credentials");

            doingLogin = true;
            final String path = "/auth/aid/" + applicationId + "/owner/" + ownerName + "/token.oauth2";
            TokenOAuth2 tokenOAuth2Response = restConnector.genericPost(oAuth2Credential, TokenOAuth2.class, path);
            doingLogin = false;

            log.debug("[getAccessToken] Token recuperado [{}]. Expira en [{}] segs. Lo agrego al a cache", tokenOAuth2Response, tokenOAuth2Response.getExpiresIn());
            accessToken = tokenOAuth2Response.getAccessToken();
            cache.put(cacheKey, accessToken, tokenOAuth2Response.getExpiresIn());

            if (forceRefresh) forceRefresh = false;
        }

        return accessToken;
    }

    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        if (builder != null) {
            if (!doingLogin) builder.header("Authorization", "Bearer " + getAccessToken());
        }
        return builder;
    }

    @Override
    public boolean retryOnUnauthorized() {
        forceRefresh = true;
        return true;
    }

    @Override
    public boolean getDisableHTTPSErrors() {
        return false;
    }
}