package ar.com.sdd.fiservapi.core;

import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSignSecurityManager;
import ar.com.sdd.fiservapi.io.*;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.lang3.NotImplementedException;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import javax.ws.rs.client.Invocation;
import javax.ws.rs.core.MediaType;
import java.time.Instant;
import java.util.Base64;
import java.util.UUID;

@SuppressWarnings("unused")
public class FiservApiConnector implements RestSignSecurityManager {

    private final static Logger log = LogManager.getLogger(FiservApiConnector.class);

    private final RestConnector restConnector;
    private final String accessTokenBaseUrl;
    private final String xApiKey;
    private final String xApiSecret;
    private final String clientId;
    private final String username;
    private final String password;

    private final static int MAX_RETRIES = 3;
    private final static int RETRY_WAIT_MILIS = 200;

    public FiservApiConnector(FiservApiConnectorContext context) {
        this.accessTokenBaseUrl = context.getAccessTokenBaseUrl();
        this.clientId = context.getClientId();
        this.username = context.getUsername();
        this.password = context.getPassword();
        this.xApiKey = context.getXApiKey();
        this.xApiSecret = context.getXApiSecret();

        log.trace("Creando FiservApiConnector para url [{}]", context.getBaseUrl());

        RestConnectorEnvironment environment = new RestConnectorEnvironment(context.getBaseUrl());
        restConnector = new RestConnector(environment, this);
    }

    /*
        |==== METODOS ACTUALES ====|

        - Creacion de Transacciones:
            - createPaymentCardCreditTransaction
            - createPaymentCardPayerAuthTransaction
            - createPaymentCardPreAuthTransaction
            - createPaymentCardSaleTransaction
            - createPaymentTokenCreditTransaction
            - createPaymentTokenPreAuthTransaction
            - createPaymentTokenSaleTransaction (Segun documentacion queremos usar este por el momento)

         - Consulta Status de Transacciones:
            - getPaymentCardCreditTransactionStatus
            - getPaymentCardPayerAuthTransactionStatus
            - getPaymentCardPreAuthTransactionStatus
            - getPaymentCardSaleTransactionStatus
            - getPaymentTokenCreditTransactionStatus
            - getPaymentTokenPreAuthTransactionStatus
            - getPaymentTokenSaleTransactionStatus (Segun documentacion queremos usar este por el momento)
    */

    public CommonResponse createPaymentCardCreditTransaction(PaymentCardCreditTransactionRequest request) throws RestConnectorException {
        final String path = "/payments";
        log.debug("[createPaymentCardCreditTransaction] Request POST por creacion de una transaccion, request [{}] path [{}]", request, path);

       return restConnector.genericPost(request, CommonResponse.class, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
    }

    public CommonResponse getPaymentCardCreditTransactionStatus(PaymentCardCreditTransactionRequest request) throws RestConnectorException {
        String transactionId = String.valueOf(request.getIpgTransactionId());
        String path = "/payments/" + transactionId;
        log.debug("[getPaymentCardCreditTransactionStatus] Request GET por consultar el estado de una transaccion, request [{}] path [{}]", request, path);

        return restConnector.genericGet(null, CommonResponse.class, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
    }

    public CommonResponse createPaymentCardPayerAuthTransaction(PaymentCardPayerAuthTransactionRequest request) throws RestConnectorException {
        final String path = "/payments";
        log.debug("[createPaymentCardPayerAuthTransaction] Request POST por creacion de una transaccion, request [{}] path [{}]", request, path);

        return restConnector.genericPost(request, CommonResponse.class, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
    }

    public CommonResponse getPaymentCardPayerAuthTransactionStatus(PaymentCardPayerAuthTransactionRequest request) throws RestConnectorException {
        String transactionId = String.valueOf(request.getIpgTransactionId());
        String path = "/payments/" + transactionId;
        log.debug("[getPaymentCardPayerAuthTransactionStatus] Request GET por consultar el estado de una transaccion, request [{}] path [{}]", request, path);

        return restConnector.genericGet(null, CommonResponse.class, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
    }

    public CommonResponse createPaymentCardPreAuthTransaction(PaymentCardPreAuthTransactionRequest request) throws RestConnectorException {
        final String path = "/payments";
        log.debug("[createPaymentCardPreAuthTransaction] Request POST por creacion de una transaccion, request [{}] path [{}]", request, path);

        return restConnector.genericPost(request, CommonResponse.class, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
    }

    public CommonResponse getPaymentCardPreAuthTransactionStatus(PaymentCardPreAuthTransactionRequest request) throws RestConnectorException {
        String transactionId = String.valueOf(request.getIpgTransactionId());
        String path = "/payments/" + transactionId;
        log.debug("[getPaymentCardPreAuthTransactionStatus] Request GET por consultar el estado de una transaccion, request [{}] path [{}]", request, path);

        return restConnector.genericGet(null, CommonResponse.class, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
    }

    public CommonResponse createPaymentCardSaleTransaction(PaymentCardSaleTransactionRequest request) throws RestConnectorException {
        final String path = "/payments";
        log.debug("[createPaymentCardSaleTransaction] Request POST por creacion de una transaccion, request [{}] path [{}]", request, path);

        return restConnector.genericPost(request, CommonResponse.class, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
    }

    public CommonResponse getPaymentCardSaleTransactionStatus(PaymentCardSaleTransactionRequest request) throws RestConnectorException {
        String transactionId = String.valueOf(request.getIpgTransactionId());
        String path = "/payments/" + transactionId;
        log.debug("[getPaymentCardSaleTransactionStatus] Request GET por consultar el estado de una transaccion, request [{}] path [{}]", request, path);

        return restConnector.genericGet(null, CommonResponse.class, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
    }

    public CommonResponse createPaymentTokenCreditTransaction(PaymentTokenCreditTransactionRequest request) throws RestConnectorException {
        final String path = "/payments";
        log.debug("[createPaymentTokenCreditTransaction] Request POST por creacion de una transaccion, request [{}] path [{}]", request, path);

        return restConnector.genericPost(request, CommonResponse.class, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
    }

    public CommonResponse getPaymentTokenCreditTransactionStatus(PaymentTokenCreditTransactionRequest request) throws RestConnectorException {
        String transactionId = String.valueOf(request.getIpgTransactionId());
        String path = "/payments/" + transactionId;
        log.debug("[getPaymentTokenCreditTransactionStatus] Request GET por consultar el estado de una transaccion, request [{}] path [{}]", request, path);

        return restConnector.genericGet(null, CommonResponse.class, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
    }

    public CommonResponse createPaymentTokenPreAuthTransaction(PaymentTokenPreAuthTransactionRequest request) throws RestConnectorException {
        final String path = "/payments";
        log.debug("[createPaymentTokenPreAuthTransaction] Request POST por creacion de una transaccion, request [{}] path [{}]", request, path);

        return restConnector.genericPost(request, CommonResponse.class, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
    }

    public CommonResponse getPaymentTokenPreAuthTransactionStatus(PaymentTokenPreAuthTransactionRequest request) throws RestConnectorException {
        String transactionId = String.valueOf(request.getIpgTransactionId());
        String path = "/payments/" + transactionId;
        log.debug("[getPaymentTokenPreAuthTransactionStatus] Request GET por consultar el estado de una transaccion, request [{}] path [{}]", request, path);

        return restConnector.genericGet(null, CommonResponse.class, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
    }

    public CommonResponse createPaymentTokenSaleTransaction(PaymentTokenSaleTransactionRequest request) throws RestConnectorException {
        final String path = "/payments";
        log.debug("[createPaymentTokenSaleTransaction] Request POST por creacion de una transaccion, request [{}] path [{}]", request, path);

        return restConnector.genericPost(request, CommonResponse.class, CommonError.class, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON);
    }

    public CommonResponse getPaymentTokenSaleTransactionStatus(PaymentTokenSaleTransactionRequest request) throws RestConnectorException {
        String transactionId = String.valueOf(request.getIpgTransactionId());
        String path = "/payments/" + transactionId;
        log.debug("[getPaymentTokenSaleTransactionStatus] Request GET por consultar el estado de una transaccion, request [{}] path [{}]", request, path);

        return restConnector.genericGet(null, CommonResponse.class, CommonError.class, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON);
    }

    public PaymentCardPaymentTokenizationResponse createCardPaymentTokenization(PaymentCardPaymentTokenizationRequest request) throws RestConnectorException {
        final String path = "/payment-tokens";
        log.debug("[createPaymentToken] Request POST por la creacion de un token, request [{}] path [{}]", request, path);
        PaymentCardPaymentTokenizationResponse response = restConnector.genericPost(request, PaymentCardPaymentTokenizationResponse.class, CommonError.class, path, MediaType.APPLICATION_JSON, MediaType.APPLICATION_JSON);
        log.debug("[createPaymentToken] Response POST por la creacion de un token, response [{}]", response);
        return response;
    }

    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        throw new NotImplementedException("user addHeader con dos parametros");
    }
    public Invocation.Builder addHeaders(Invocation.Builder builder, String requestBody) throws RestConnectorException {
        String clientRequestId = UUID.randomUUID().toString();
        String timestamp = Long.toString(Instant.now().toEpochMilli());
        if (builder != null) {
            builder.header("Content-Type",  MediaType.APPLICATION_JSON);
            builder.header("Api-Key", xApiKey);
            builder.header("Timestamp", timestamp);
            builder.header("Message-Signature", generateMessageSignature(clientRequestId, timestamp, requestBody));
            builder.header("Client-Request-Id", clientRequestId);
        }
        return builder;
    }

    @Override
    public boolean retryOnUnauthorized() {
        return  false;
    }

    public String generateMessageSignature(String clientRequestId, String timestamp, String jsonBody) {
        String encoded = null;
        try {
            String cleanedJsonBody = jsonBody.replaceAll("\\s+", "");
            String message = xApiKey + clientRequestId + timestamp + cleanedJsonBody;

            final Mac hmac256 = Mac.getInstance("HmacSHA256");
            final SecretKeySpec secretKeySpec = new SecretKeySpec(xApiSecret.getBytes("UTF-8"), "HmacSHA256");
            hmac256.init(secretKeySpec);
            final byte[] signature = hmac256.doFinal(message.getBytes("UTF-8"));
            encoded = Base64.getEncoder().encodeToString(signature);
        } catch (Exception e) {
            log.error("Error al generar la Message-Signature", e);
        }

        return encoded;
    }

    @Override
    public boolean getDisableHTTPSErrors() {
        return false;
    }
}
