package ar.com.sdd.getnetapi.core;

import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSecurityManager;
import ar.com.sdd.commons.rest.model.TokenOAuth2;
import ar.com.sdd.commons.rest.util.RestConnectorUtil;
import ar.com.sdd.commons.util.SimpleCache;
import ar.com.sdd.commons.util.SimpleCacheManager;
import ar.com.sdd.commons.util.ThreadUtil;
import ar.com.sdd.getnetapi.io.*;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.ws.rs.client.Invocation;
import javax.ws.rs.core.Form;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

@SuppressWarnings("unused")
public class GetNetApiConnector implements RestSecurityManager {

    private final static Logger log = LogManager.getLogger(GetNetApiConnector.class);

    private final RestConnector restConnector;
    private final String clientId;
    private final String clientSecret;
    private boolean doingLogin;

    private final static int MAX_RETRIES = 3;
    private final static int RETRY_WAIT_MILIS = 200;

    public GetNetApiConnector(GetNetApiConnectorContext context) {
        this.clientId = context.getGetNetApiClientId();
        this.clientSecret = context.getGetNetApiClientSecret();

        log.trace("Creando GetNetConnector para url [{}]", context.getGetNetApiBaseUrl());

        RestConnectorEnvironment environment = new RestConnectorEnvironment(context.getGetNetApiBaseUrl());
        restConnector = new RestConnector(environment, this);
    }

    public PaymentIntentResponse createPaymentIntent(PaymentIntentRequest paymentIntentRequest) throws RestConnectorException {
        final String path = "/digital-checkout/v1/payment-intent";
        log.debug("[createPaymentIntent] Request POST createPaymentIntent para el {} path [{}]", paymentIntentRequest, path);
        RestConnectorException lastException = null;
        for (int retry = MAX_RETRIES; retry > 0; retry--) {
            try {
                return restConnector.genericPost(paymentIntentRequest, PaymentIntentResponse.class, CommonError.class, path);
            } catch (RestConnectorException e) {
                lastException = e;
                if (e.getStatus() == Response.Status.UNAUTHORIZED.getStatusCode()) {
                    getAccessToken(true);
                }
            }
            ThreadUtil.sleep(RETRY_WAIT_MILIS);
        }
        throw lastException;
    }

    public PaymentCancellationResponse cancelPayment(String paymentId) throws RestConnectorException {
        final String path = "/digital-checkout/v1/payments/" + paymentId + "/cancellation";
        log.debug("[cancelPayment] Request POST cancelPayment para el paymentId [{}] path [{}]", paymentId, path);
        RestConnectorException lastException = null;
        for (int retry = MAX_RETRIES; retry > 0; retry--) {
            try {
                return restConnector.genericPost(null, PaymentCancellationResponse.class, CommonError.class, path);
            } catch (RestConnectorException e) {
                lastException = e;
                if (e.getStatus() == Response.Status.UNAUTHORIZED.getStatusCode()) {
                    getAccessToken(true);
                }
            }
            ThreadUtil.sleep(RETRY_WAIT_MILIS);
        }
        throw lastException;
    }

    /**
     * Queda escrito pero no se usa en la integracion actual (iFrame para HLCM)
     * @param paymentId
     * @return
     * @throws RestConnectorException
     */
    /*
    public PaymentReversalResponse reversePayment(String paymentId) throws RestConnectorException {
        final String path = "/digital-checkout/v1/payments/" + paymentId + "/reversal";
        log.debug("[reversePayment] Request POST reversePayment para el paymentId [{}] path [{}]", paymentId, path);
        RestConnectorException lastException = null;
        for (int retry = MAX_RETRIES; retry > 0; retry--) {
            try {
                return restConnector.genericPost(null, PaymentReversalResponse.class, CommonError.class, path);
            } catch (RestConnectorException e) {
                lastException = e;
                if (e.getStatus() == Response.Status.UNAUTHORIZED.getStatusCode()) {
                    getAccessToken(true);
                }
            }
            ThreadUtil.sleep(RETRY_WAIT_MILIS);
        }
        throw lastException;
    }
     */

    public PaymentRefundResponse refundPayment(String paymentId, PaymentRefundRequest paymentRefundRequest) throws RestConnectorException {
        final String path = "/digital-checkout/v1/payments/" + paymentId + "/refund";
        log.debug("[refundPayment] Request POST refundPayment para el paymentId [{}] path [{}]", paymentId, path);
        RestConnectorException lastException = null;
        for (int retry = MAX_RETRIES; retry > 0; retry--) {
            try {
                return restConnector.genericPost(paymentRefundRequest, PaymentRefundResponse.class, CommonError.class, path);
            } catch (RestConnectorException e) {
                lastException = e;
                if (e.getStatus() == Response.Status.UNAUTHORIZED.getStatusCode()) {
                    getAccessToken(true);
                } else if (e.getStatus() == RestConnectorUtil.ResponseStatus.UNPROCESSABLE_ENTITY.getStatusCode()) {
                    throw lastException;
                }
            }
            ThreadUtil.sleep(RETRY_WAIT_MILIS);
        }
        throw lastException;
    }

    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        if (builder != null) {
            if (!doingLogin) builder.header("Authorization", "Bearer " + getAccessToken(false));
        }
        return builder;
    }

    @Override
    public boolean retryOnUnauthorized() {
        try {
            getAccessToken(true);
            return true;
        } catch (Exception any) {
            return false;
        }
    }

    @Override
    public boolean getDisableHTTPSErrors() {
        return false;
    }

    public String getAccessToken(boolean force) throws RestConnectorException {
        SimpleCache cache = SimpleCacheManager.getInstance().getCache(GetNetApiConnector.class.getName());
        String cacheKey = SimpleCacheManager.buildKey("accessToken");
        String accessToken = (String) cache.get(cacheKey);
        if (force || accessToken == null) {
            Form form = new Form();
            form.param("client_id", clientId);
            form.param("client_secret", clientSecret);
            form.param("grant_type", "client_credentials");

            log.debug("[getAccessToken] Por recuperar token con clientId [{}] y clientSecret [{}]", clientId, clientSecret);

            doingLogin = true;
            TokenOAuth2 tokenOAuth2Response = restConnector.genericPost(form, TokenOAuth2.class, "/authentication/oauth2/access_token", MediaType.APPLICATION_JSON, MediaType.APPLICATION_FORM_URLENCODED);
            doingLogin = false;

            log.debug("[getAccessToken] Token recuperado [{}]. Expira en [{}]segs. Lo agrego al a cache", tokenOAuth2Response, tokenOAuth2Response.getExpiresIn());
            accessToken = tokenOAuth2Response.getAccessToken();
            cache.put(cacheKey, accessToken, tokenOAuth2Response.getExpiresIn());
        }

        return accessToken;
    }
}