package ar.com.sdd.gicapi.client.core;

import ar.com.sdd.commons.rest.core.RestConnector;
import ar.com.sdd.commons.rest.core.RestConnectorEnvironment;
import ar.com.sdd.commons.rest.core.RestConnectorException;
import ar.com.sdd.commons.rest.core.RestSecurityManager;
import ar.com.sdd.commons.util.SimpleCache;
import ar.com.sdd.commons.util.SimpleCacheManager;
import ar.com.sdd.gicapi.client.io.GicOAuth2Credential;
import ar.com.sdd.gicapi.client.io.GicTokenOAuth2;
import ar.com.sdd.gicapi.client.io.PaymentStatusUpdateRequest;
import ar.com.sdd.gicapi.client.io.PaymentStatusUpdateResponse;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.ws.rs.client.Invocation;

@SuppressWarnings("unused")
public class GicApiConnector implements RestSecurityManager {

    private final static Logger log = LogManager.getLogger(GicApiConnector.class);

    private final RestConnector restConnector;
    private final String clientId;
    private final String clientSecret;
    private boolean doingLogin;
    private final String origen;

    public GicApiConnector(GicApiConnectorContext context) {
        this.origen = context.getOrigen();
        this.clientId = context.getClientId();
        this.clientSecret = context.getClientSecret();

        log.trace("Creando GicApiConnector con {}", context);

        RestConnectorEnvironment environment = new RestConnectorEnvironment(context.getBaseUrl());
        restConnector = new RestConnector(environment, this);
    }

    /**
     *
     */
    public PaymentStatusUpdateResponse postPaymentStatusUpdate(PaymentStatusUpdateRequest paymentStatusUpdateRequest) throws RestConnectorException {
        String path = "/Payment/paymentStatusUpdate/" + origen;
        log.debug("[postPaymentStatusUpdate] Por notificar cambio de estado de un pago con path [{}] y {}", path, paymentStatusUpdateRequest);
        return restConnector.genericPost(paymentStatusUpdateRequest, PaymentStatusUpdateResponse.class, PaymentStatusUpdateResponse.class, path);
    }

    @SuppressWarnings("RedundantThrows")
    @Override
    public Invocation.Builder addHeaders(Invocation.Builder builder) throws RestConnectorException {
        if (builder != null) {
            if (!doingLogin) builder.header("Authorization", "Bearer " + getAccessToken(false));
        }
        return builder;
    }

    @Override
    public boolean retryOnUnauthorized() {
        try {
            getAccessToken(true);
            return true;
        } catch (Exception any) {
            return false;
        }
    }

    @Override
    public boolean getDisableHTTPSErrors() {
        return false;
    }

    public String getAccessToken(boolean force) throws RestConnectorException {
        SimpleCache cache = SimpleCacheManager.getInstance().getCache(GicApiConnector.class.getName());
        String cacheKey = SimpleCacheManager.buildKey("accessToken");
        String accessToken = (String) cache.get(cacheKey);
        if (force || accessToken == null) {
            log.debug("[getAccessToken] Por recuperar token con clientId [{}] y clientSecret [{}]", clientId, clientSecret);

            doingLogin = true;
            GicTokenOAuth2 tokenOAuth2Response = restConnector.genericPost(new GicOAuth2Credential(clientId, clientSecret, "GIC_credentials"), GicTokenOAuth2.class, "/Auth/generateToken/" + origen);
            doingLogin = false;

            log.debug("[getAccessToken] Token recuperado [{}]. Expira en [{}] segs. Lo agrego al a cache", tokenOAuth2Response, tokenOAuth2Response.getExpiresIn());
            accessToken = tokenOAuth2Response.getToken();
            cache.put(cacheKey, accessToken, tokenOAuth2Response.getExpiresIn());
        }

        return accessToken;
    }
}